# -*- coding: utf-8 -*-
##############################################################
# smscon_master - remote control master for Nokia N900 phone #
# $Id: smscon_master.py 221 2012-07-01 21:57:32Z yablacky $
##############################################################

import os
import sys
import time                
import re

from smsconlib import *

##############################################################
# For help text, assemble pretty display names of our various files:

ScriptDisplayName    = ScriptFile.replace(NAME, '').strip('_') or 'script'
ConfigDisplayName    = ConfigFile.replace(NAME, '').strip('_') or 'config'
CodeDisplayName      = CodeFile  .replace(NAME, '').strip('_') or 'code'
BootDisplayName      = BootFile  .replace(NAME, '').strip('_') or 'boot'
ScriptLogDisplayName = ScriptLog .replace(NAME, '').strip('_') or 'log'
LogDisplayName       = LogFile   .replace(NAME, '').strip('_') or 'log'

##############################################################

def ShowFile(*File, **kwargs):
    """
    Show file. Very verbose.
    @param list Strings of path name parts.
    @param list_of_2-tuples ReplacementRules where each list entry ([0] is pattern and [1] is replacement).
    @return bool True iff file exists and is non empty.
    """
    Filename = os.path.join(*File)
    try:
        f = open(Filename, 'r')
    except:
        LOGGER.warning('File "%s" not found.' % Filename)
        return False
    Lines = f.readlines() 
    f.close()
    if Lines == []:
        LOGGER.info('File "%s" is empty.' % Filename)
        return False

    ReplacementRules = kwargs.get('ReplacementRules')
    if not ReplacementRules:
        for Line in Lines:
            print Line,
    else:
        for Line in Lines:
            for Rule in ReplacementRules:
                Line = re.sub(Rule[0], Rule[1], Line)
            print Line,
    return True

def ShowSmsCommands(theConf):
    """
    Show commands that are recognized if sent via SMS.
    @return bool True iff commands could be shown.
    """
    for SmsCommand in sorted(theConf.getSmsCommandList()):
        print theConf._genConfigLine(SmsCommand, theConf['SMS_COMPREFIX'] + theConf[SmsCommand] + theConf['SMS_COMSUFFIX'])
    return True


def AuthorizedSIMs_ShowList():
    """
    Show list of authorized SIM cards from SIM database.
    @return bool True iff SIM cards available and listed.
    """
    simDB = IMSIConfiguration()
    if not simDB.load():
        return False
    for n, Text in enumerate(simDB):
        CodeType, CodeValue = simDB.UnpackCodeType(Text)
        print 'SIM_%02d = %s %s' % ( n + 1, CodeType, CodeValue )
    return True;

def AuthorizedSIMs_GrantCurrentSIM():
    """
    Authorize the currently inserted SIM card to SIM database and
    tell daemon to reload SIM database.
    @return bool True iff SIM card data written to SIM database.
    """
    ICCID = GetPhoneCode('ICCID')
    if ICCID == None:
        LOGGER.error('No SIM card present.')
        return False
    IMSI = GetPhoneCode('IMSI')
    if IMSI == None:
        LOGGER.error('The SIM card is not active (no or wrong PIN).')
        return False
    simDB = IMSIConfiguration()
    simDB.updateCode('IMSI', IMSI, None, Silent = True) # Ensure IMSI code of this SIM is removed from database.
    ok = simDB.updateCode('ICCID', ICCID, '')
    if ok:
        daemon.tell(daemon.TELL_RELOAD)
    return ok

def AuthorizedSIMs_RevokeCurrentSIM():
    """
    Remove current inserted SIM card code from SIM database and
    tell daemon to reload SIM database.
    @return bool True iff current SIM code no longer present in SIM database.
    """
    ICCID = GetPhoneCode('ICCID')
    if ICCID == None:
        LOGGER.error('No SIM card present.')
        return False
    simDB = IMSIConfiguration()
    ok1 = simDB.updateCode('ICCID', ICCID, None)
    IMSI = GetPhoneCode('IMSI')
    if IMSI == None:
        ok2 = False
        LOGGER.warning('The SIM card is not active (no or wrong PIN); could still be authorized!')
    else:
        ok2 = simDB.updateCode('IMSI', IMSI, None, Silent = True)
    if ok1 or ok2:
        daemon.tell(daemon.TELL_RELOAD)
    return ok

def AuthorizedSIMs_RevokeAllSIM():
    """
    Remove all SIM card codes from SIM database and
    tell daemon to reload SIM database. 
    @return bool True iff no SIM codes present in SIM database.
    """
    if not IsFileReadable(InstPath, CodeFile):
        return True
    ok = DeleteFile(InstPath, CodeFile)
    if ok:
        daemon.tell(daemon.TELL_RELOAD)
    return ok

##############################################################
# Command line interface

def ShowOptions(Level = 1):
    """
    Show options for smscon.
    """
    if Level >= 1:
        print '== %s %s - Nokia N900 remote control utility ==' % (NAME, VERSION)
        print 'usage: %s [Options]...' % NAME
        print ' Options:'
        print '   -start             : Start %s.' % DaemonFile
        print '   -restart           : Stop & restart %s.' % DaemonFile
        print '   -stop              : Stop %s.' % DaemonFile
        print '   -status            : Get %s status.' % DaemonFile
        print '   -log               : Show log file. Anonymized if no or wrong -pw option seen before.'
        print '   -del log           : Delete log file.'
        print '   -sms               : Show SMS commands.'
        print '   -set name "value"  : Set user setting (name = "value").'
        print '   -alarm [ref] "filename" : Define sound to be played on alarm command.'
        print '                        If ref is specified the file will only referenced, not copied.'
        print '                        Use ref only while testing several sounds.' 
        print '                        A filename of --default restores standard alarm.'
        print '   -config            : Show %s file.' % ConfigDisplayName
        print '   -script            : Show %s file.' % ScriptDisplayName
        print '   -scriptlog         : Show %s file.' % ScriptLogDisplayName
        print '   -del scriptlog     : Delete %s file.' % ScriptLogDisplayName
        print '   -boot              : Enable %s to start at phone boot.' % DaemonFile
        print '   -unboot            : Disable %s to start at phone boot.' % DaemonFile
        print '   -add imsi          : Grant authorization to current SIM card.'
        print '   -remove imsi       : Revoke authorization of current SIM card.'
        print '   -del imsi          : Revoke authorization of all SIM cards.'
        print '   -pw password       : Password to enable non-anonymized output. MASTERNUMBER is password.'
        print '   -version           : Show version number'    
        print '   -help              : This help menu'
        print '   -help2             : More help (special options, rarely used).'
    if Level >= 2:
        print ' Special options (normally not needed!):'
        print '   -init              : Ensure %s files exist; upgrades existing files' % ' & '.join((ConfigDisplayName, ScriptDisplayName))
        print '   -reset             : Restore factory settings: Stop all tasks,'
        print '                        disable boot starter and delete all the'
        print '                        %s-,' % '-, '.join((BootDisplayName, ConfigDisplayName, CodeDisplayName,
                                                        ScriptDisplayName, ScriptLogDisplayName, LogDisplayName))
        print '                        %s-, %s- and %s-files.' % (PhotoName, MapName, AudioFile)
        print '   -export filename   : Export user settings into filename.'
        print '                        Use -export! to write even existing file.'
        print '   -import filename   : Import user settings from filename.'
        print '                        Use -import! to continue import on errors.'
        print '   -imsi              : Show list of authorized SIM cards.'
        print '   -del imsi          : Revoke authorization from all SIM cards.'
        print '   -del config        : Delete %s file.' % ConfigDisplayName
        print '   -del script        : Delete %s file.' % ScriptDisplayName
        print '   -sendsms number message : Send one SMS message to number. Number may be verbatim "MASTERNUMBER"'
        print '   -sendemail recipient subject message [attachment-filename [attachment-mimetype]]: Send email.'
        print '                        If recipient is a single char the EMAILADDRESS from configuration is used.'
        print '                        The email is accepted and send only if smscon_deamon is running.'
    
        if os.geteuid() != 0:
            print 'WARNING: %s must run as "root". You are currently not "root".' % NAME

def ShowHiddenOptions():
    """
    Show hidden options for smscon.
    """
    print ' Developer options:'
    print '   -devhelp           : This extended help menu.'
    print '   -branch            : Show branch number. Deprecated. Removed in future. Shows -version now.'
    print '   -build             : Show build number.  Deprecated. Removed in future. Shows -version now.'
    print '   -init!             : Create default %s files. Do not upgrade existing ones.' % ' & '.join((ConfigDisplayName, ScriptDisplayName))
    print '   -reload            : Tell daemon to read config file again (-force-reload is also accepted).'
    print '   -get name "default": Get user setting.'
    print '   -test "test"       : Run developer test (gps1, gps2, ssh, ssh-off, cam, sms, email1, email2, call, script).'
    print '   -comtest "command" : Run %s command as if received as SMS message (no prefix/suffix required).' % NAME
    print '   -devconfig password: Show uncrypted %s file. MASTERNUMBER is password.' % ConfigDisplayName

##############################################################

def DoCommandLine():
    """
    Parse command line and perform actions.
    @return bool Success indicator.
    """
    def TellError(Msg):
        LOGGER.error(Msg)
        return False

    global ourConf, daemon, settingsChangeCount
    script = smscon_script()
    remosh = smscon_remote_sh(ourConf)

    ClientUsrID = OurUsrID = os.geteuid()
    ClientGrpID = OurGrpID = os.getegid()
    Password = None
    DelaySeconds = 0.5
    args = sys.argv[1:]
    ok = len(args) > 0
    if not ok:
        ShowOptions()  
    while (ok and len(args) > 0):
        LOGGER.debug("Command Line args: %s" % ' '.join(args))
        OriMode = Mode = args.pop(0)
        # Allow all options to start with the usual double dash as well:
        if len(Mode) > 3 and Mode[0:2] == '--' and Mode[3:1] != '-':
            Mode = Mode[1:]
        # Allow bare word options to control service (called this way by /etc/init.d/rc) 
        elif Mode in ('start', 'restart', 'stop', 'reload' 'force-reload') and len(args) == 0:
            Mode = '-' + Mode; 

        if   Mode == '-help':       ShowOptions(1) 
        elif Mode == '-help2':      ShowOptions(2) 
        elif Mode == '-devhelp':    ShowOptions(2); ShowHiddenOptions()
        elif Mode == '-version':    print VERSION        
        elif Mode == '-branch':     print VERSION
        elif Mode == '-build':      print VERSION
        elif Mode == '-status':     print "Daemon %s is %s" % (DaemonFile, IF(daemon.runs(), 'running', 'off'))
        elif Mode == '-log':
            if Password != None and Password == ourConf.get('MASTERNUMBER'):
                ok = ShowFile(LOGGER.LogFilename)
            else:   # No or invalid password: show anonymous log:
                ok = ShowFile(LOGGER.LogFilename, ReplacementRules = [
                    ('\\+?[0-9]{8,}', '+00123456789'),                          # replace potential phone numbers
                    ('([a-zA-Z][a-zA-Z0-9_\\-]*\\.)+[a-zA-Z0-9_\\-]*[a-zA-Z]', 'example.com'),  # replace potential host names
                    ('(?:[0-9]{1,3}\\.){2,3}([0-9]{1,3})', '192.168.128.\\1'),  # replace potential IPv4 (but no floats)
                ])    
        elif Mode == '-imsi':       ok = AuthorizedSIMs_ShowList()
        elif Mode == '-sms':        ok = ShowSmsCommands(ourConf)
        elif Mode == '-config':     ok = ShowFile(ourConf.Filename)
        elif Mode == '-script':     ok = ShowFile(script.Filename)
        elif Mode == '-scriptlog':  ok = ShowFile(InstPath, ScriptLog)
        elif Mode == '-devconfig':
            if len(args) == 0: return TellError("%s requires argument" % Mode)
            Password = args.pop(0)
            ok = UserConfiguration().show(Password = Password)

        elif Mode == '-pw':
            if len(args) == 0: return TellError("%s requires argument" % Mode)
            Password = args.pop(0)

        # The next commands potentially modify data and therefore need root permissions:
        elif os.geteuid() != 0 and not XDEBUG:
            return TellError('%s must run as "root" for this command.' % NAME)

        elif Mode == '-usrid':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            try:    ClientUsrID = int(args.pop(0))
            except: return TellError("%s argument must be integer." % Mode)

        elif Mode == '-grpid':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            try:    ClientGrpID = int(args.pop(0))
            except: return TellError("%s argument must be integer." % Mode)

        elif Mode in ( '-init', '-init!'):
            Force = Mode == '-init!'
            if Force and daemon.stop() + script.stop():
                time.sleep(DelaySeconds)

            if ourConf.create(Force = Force):
                LOGGER.warning('Do not edit "%s" directly. Use "%s -set OPTION VALUE" instead!' % (ourConf.Filename, NAME))
                settingsChangeCount += 1
            else:
                ok = False
            ok = script.init(Force) and ok

        elif Mode in ('-export', '-export!') :
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Filename = args.pop(0)
            if not IsFileReadable(ourConf.Filename):
                return TellError('Export to "%s" failed: No source configuration file.' % Filename)
            if Mode != '-export!' and IsFileReadable(Filename):
                return TellError('Export to "%s" failed: file already exists.' % Filename)
            # Perform export operation in client user/group privilege context.
            # This prevents creation or overwriting of files where user has
            # normally no permission:   
            try:
                os.setegid(ClientGrpID) # must set group first!
                os.seteuid(ClientUsrID)
                ok = ourConf.backup(Filename = Filename) and ok
            except Exception, e:
                ok = TellError('Export to "%s" failed: %s.' % (Filename, e))
            finally:
                os.seteuid(OurUsrID)
                os.setegid(OurGrpID)

        elif Mode in ('-import', '-import!') :
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Filename = args.pop(0)
            restoreCount = ourConf.restore(Filename = Filename, Force = Mode == '-import!')
            settingsChangeCount += abs(restoreCount)
            ok = restoreCount > 0 and ok

        elif Mode in ('-reload', '-force-reload'):
            ok = daemon.tell(daemon.TELL_RELOAD) and ok

        elif Mode in ('-start', '-restart'):
            if not IsFileReadable(InstPath, ConfigFile):
                return TellError('Config file %s not present. Use -init.' % ConfigFile)
            if not IsFileReadable(InstPath, ScriptFile):
                return TellError('Script file %s not present. Use -init.' % ScriptFile)
    
            if daemon.runs():
                if Mode == '-start':
                    LOGGER.info('Daemon %s already active.' % DaemonFile)
                    continue
                if daemon.stop():
                    time.sleep(DelaySeconds)
            daemon.start()
            time.sleep(DelaySeconds)
            PIDdaemon = daemon.runs()
            if not PIDdaemon:
                return TellError('Daemon %s failed to start.' % DaemonFile)
            LOGGER.info('Daemon %s started [PID=%s].' % (DaemonFile, PIDdaemon))

        elif Mode == '-stop':
            daemon.stop()
            remosh.stop()
            script.stop()

        elif Mode == '-reset':
            if (daemon.stop() + remosh.stop() + script.stop()):
                time.sleep(DelaySeconds)
            ok = DeleteFile(BootPath, BootFile) and ok
            ok = DeleteFile(InstPath, CodeFile) and ok
            ok = DeleteFile(InstPath, ConfigFile) and ok
            ok = DeleteFile(InstPath, ScriptFile) and ok
            ok = DeleteFile(InstPath, ScriptLog) and ok
            ok = DeleteFile(InstPath, PhotoName) and ok
            ok = DeleteFile(InstPath, MapName) and ok
            ok = DeleteFile(InstPath, AudioFile) and ok     # don't delete DefaultAudioFile.
            ok = DeleteFile(LogPath, LogFile) and ok
            ok = DeleteFile(LogPath, BootLogFile) and ok

        elif Mode == '-boot':       ok = CreateBootfile()
        elif Mode == '-unboot':     ok = DeleteFile(BootPath, BootFile)

        elif Mode in('-test', '-comtest'):
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if not daemon.runs():
                daemon.start(Mode, Func)
            elif not daemon.writeCliCommand(Mode, Func):
                return TellError('%s %s failed to start.' % (Mode[1:], Func))
            LOGGER.info('%s %s started...' % (Mode[1:], Func))

        elif Mode == '-alarm':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Ref = args.pop(0)
            if Ref == 'ref':
                if len(args) == 0: return TellError("%s ref requires argument." % Mode)
                NewAudioFilename = args.pop(0)
            else:
                NewAudioFilename, Ref = Ref, False 

            if NewAudioFilename == '--default':
                NewAudioFilename, Ref = os.path.join(InstPath, DefaultAudioFile), True
            if Ref:
                NewAudioFilename = os.path.abspath(NewAudioFilename)
            if not IsFileReadable(NewAudioFilename):
                return TellError("Sound does not exist or is not readable: %s" % NewAudioFilename)
            # Remove the old audio file, do not just write! It could be a symlink.
            Problem = ''
            Filename = os.path.join(InstPath, AudioFile)
            try:    os.remove(Filename)
            except Exception, e:
                Problem = str(e)
            if IsFileReadable(Filename):
                return TellError("Failed to remove old alarm sound '%s'. %s" % (Filename, Problem))
            # Establish the new file:
            try:
                if Ref:
                    os.symlink(NewAudioFilename, Filename)
                else:
                    os.system('cp %s %s' % (shellArg(NewAudioFilename), shellArg(Filename)))
            except Exception, e:
                Problem = str(e)
            if not IsFileReadable(Filename):
                return TellError("Failed to establish alarm sound '%s'. %s" % (NewAudioFilename, Problem))
            LOGGER.info("Alarm sound '%s' established." % (NewAudioFilename))

        elif Mode == '-del':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'log':
                                for fname in os.listdir(LogPath):
                                    if fname.startswith(LogFile):    # Account for files generated by RotatingFileHandler.
                                        ok = DeleteFile(LogPath, fname) and ok
            elif Func == 'imsi':        ok = AuthorizedSIMs_RevokeAllSIM()
            elif Func == 'config':      ok = DeleteFile(InstPath, ConfigFile)
            elif Func == 'script':      ok = DeleteFile(script.Filename)
            elif Func == 'scriptlog':   ok = DeleteFile(InstPath, ScriptLog)
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-add':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'imsi':    ok = AuthorizedSIMs_GrantCurrentSIM()
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-remove':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'imsi':    ok = AuthorizedSIMs_RevokeCurrentSIM()
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-set':
            if len(args) < 2: return TellError("%s requires 2 arguments." % Mode)
            ConfigVar   = args.pop(0)
            ConfigValue = args.pop(0)
            ok = ourConf.update(ConfigVar, ConfigValue)
            if ok: settingsChangeCount += 1

        elif Mode == '-get':
            if len(args) < 2: return TellError("%s requires 2 arguments." % Mode)
            ConfigVar   = args.pop(0)
            ConfigValue = args.pop(0)
            print ourConf.get(ConfigVar, ConfigValue)

        elif Mode == '-sendsms':
            if len(args) < 2: return TellError("%s requires 2 arguments: recipient-number message" % Mode)
            TargetNumber = Recipient = args.pop(0).upper()
            if TargetNumber == 'MASTERNUMBER':
                TargetNumber = ourConf.get('MASTERNUMBER', '')
            Message = args.pop(0)
            if re.match('^\+?0*$', TargetNumber):
                return TellError('Sending SMS to %s failed: no recipient number.' % Recipient)
            if not re.match('^\+?[0-9]+$', TargetNumber):
                return TellError('Sending SMS to %s failed: wrong recipient number.' % Recipient)
            if not SendSMSviaGSMmodem(TargetNumber, Message[0:160]):
                LOGGER.error('Sending SMS to %s failed while sending.' % Recipient)
            elif len(Message) > 160: 
                LOGGER.warning('SMS send to %s but truncated to 160 chars.' % Recipient)
            else:
                LOGGER.info('SMS send to %s.' % Recipient)

        elif Mode == '-sendemail':
            if len(args) < 3: return TellError("%s requires 3 arguments: recipient subject message [attachment-filename [attachment-mimetype]]." % Mode)
            TargetNumber = Recipient = args.pop(0)
            if len(Recipient) < 2:
                Recipient = ourConf.get('EMAILADDRESS', '')
            if not re.match('^[^@]+@[^@]+$', Recipient):
                return TellError('Sending EMAIL to %s failed: no email address.' % TargetNumber)
            Subject, Message =  args.pop(0), args.pop(0)
            AttachmentInfo = []
            len(args) > 0 and args[0][0] != '-' and AttachmentInfo.append(args.pop(0))    # AttachmentFileName
            len(args) > 0 and args[0][0] != '-' and AttachmentInfo.append(args.pop(0))    # AttachmentMimeType
            if daemon.writeCliCommand(Mode, Recipient, Subject, Message, *AttachmentInfo):
                LOGGER.info('Sending email to %s successfully queued.' % Recipient)
            else:
                return TellError('Sending email to %s failed.' % Recipient)
        else:
            return TellError('Unknown option (%s).' % OriMode)
                
    return ok;

##############################################################

def main():
    """
    Main function of smscon master utility. Does not return to caller.
    """
    global LOGGER, ourConf, daemon, settingsChangeCount

    if os.geteuid() != 0 and not XDEBUG:
        os.execvp("sudo", ["sudo"] + sys.argv[0:1]
                    + ["-usrid", str(os.geteuid()), "-grpid", str(os.getegid())]
                    + sys.argv[1:])
        # We get here on failure only!
        print 'ERROR: %s must run as "root". Try sudo command prefix to elevate your rights.' % NAME
        sys.exit(1)

    LOGGER = StartLogging('SMSCON', withConsole = True)

    ourConf = UserConfiguration()
    if ourConf.load(Silent = True) or ourConf.needsUpgrade():
        ourConf.upgrade()

    EnableDebugLogging(ourConf)

    daemon = smscon_daemon()
    settingsChangeCount = 0

    ok = DoCommandLine()

    if settingsChangeCount:
        daemon.tell(daemon.TELL_RELOAD)

    sys.exit(IF(ok, 0, 1))

##############################################################
