#!/usr/bin/env python
# -*- coding: utf-8 -*-

##############################################################
##   smscon-editor  -  Configuration UI for the SMSCON      ##
# $Id: smsconEditorMain.py 108 2011-12-28 18:46:50Z saturn $
##############################################################
# Copyright (c) 2010-2011 Christos Zamantzas
# Licenced under GPLv2

#Author: Christos Zamantzas <christos.zamantzas@gmail.com>
#Contributor: Frank Visser - import, decrypt and encrypt from config file the user variables.
#Contributor: Lutz Schwarz - refacturing, improvement of the code, as well as implement many new features.         

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published
## by the Free Software Foundation; version 2 only.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##

Version = '0.9.8-1'

import os
import sys
import time
import string
import base64
import pexpect

__FILE__ = os.path.realpath(__file__)

# smsconlib is not stored in common place. Extend sys.path to find smsconlib on phone:
sys.path = [os.path.realpath(__FILE__+'/../../smscon')] + sys.path

import smsconlib as smscon
from smsconlib import LOGGER, UserConfiguration
from smsconlib import IF, IsFileReadable

from PyQt4.QtGui import *
from PyQt4.QtCore import *
from PyQt4.QtMaemo5 import *

from smsconEditorStartupUI import *
from smsconEditorAskPassUI import *
from smsconEditorNewPassUI import *
from smsconEditorMainUI import *
from smsconEditorGeneralUI import *
from smsconEditorCommandsUI import *
from smsconEditorEmailUI import *
from smsconEditorSSHUI import *
from smsconEditorHostUI import *
from smsconEditorInitUI import *
from smsconEditorTestUI import *

##############################################################

TempPath        = '/home/user/.smscon-editor/'   # temp path of files
AppPath         = '/opt/smscon-editor/'          # main path of editor application files
PassFile        = os.path.join(AppPath, 'smsconEditorPass')
InitialPassword = '12345'
FatalHelp       = 'Please contact SMSCON support team for a solution.'

editConf        = UserConfiguration(Filename = os.path.join(TempPath, smscon.ConfigFile))

areFeedbackDetailsDesired = False

def commonPrefix(*args):
    """@return string The longest same string that all passed strings are starting with"""
    for idx in range(min(map(len, args))):
        for s in args:
            if s[idx] != args[0][idx]: 
                return s[0:idx];
    if len(args):   return args[0]
    else:           return ''

def shellArg(s):
    """@return string The parameter s quoted to be safely used as a shell parameter."""
    s = s.replace("\'", "'\"'\"'")
    return "'%s'" % s

def showMessage(message):
    '''Display a message to the user and wait for confirmation.'''

    os.system(  'dbus-send --type=method_call --dest=org.freedesktop.Notifications' \
                ' /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog' \
                ' string:%s uint32:0 string:"OK"' % shellArg(message))

def showQuickMessage(message):
    '''Display a message to the user and continue immediately.'''

    os.system(  'dbus-send --type=method_call --dest=org.freedesktop.Notifications' \
                ' /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteInfoprint' \
                ' string:%s' % shellArg(message))

def showLogRecord(logRecord):

    msg = "%s: %s" % (logRecord.levelname, logRecord.msg)
    if logRecord.levelname == 'ERROR':
       showMessage(msg)
    elif logRecord.levelname == 'WARNING':   
       showMessage(msg)
    elif logRecord.levelname == 'CRITICAL':   
       showMessage(msg)
    else:
       if areFeedbackDetailsDesired:
          showQuickMessage(msg)
       else:
          pass

LOGGER = smscon.StartLogging('EDITOR', withConsole = False, withEmitterFxn = showLogRecord)

def configYesNo(boolValue):
    return IF(boolValue, 'yes', 'no')

def configTrue(yesNoValue):
    return yesNoValue == 'yes'

def IsPasswordValid(password):  ### FIXME: what should be the behaviour if password is missing or empty?
    '''
    Check the given password.
    @return bool True if password is valid (may terminate app on hack detection).
    '''

    try:
       filesize = os.path.getsize(PassFile)
    except:
       message = 'Missing password file!\n%s' % FatalHelp
       showMessage(message)
       #doUpdatePassword()
       sys.exit(1)

    if filesize == 0:
       message = 'Empty password file!\n%s' % FatalHelp
       showMessage(message)
       sys.exit(1)

    return os.popen('echo %s|md5sum -c "%s"' % (shellArg(password), PassFile)).read() == '-: OK\n'

def CopyMainToEditConfig():
    '''
    Make a copy of the main configuration for safe editing.
    @return bool Success indicator (may terminate app on hack detection).
    '''  

    def tryCopy():
        
       global editConf
       
       if os.WEXITSTATUS(os.system('sudo smscon -export! "%s"' % editConf.Filename)) == 0:
          return True
       msg1 = 'Loading active settings failed.'

       if os.WEXITSTATUS(os.system('sudo smscon -init')) == 0:
          msg2 = '\nActivated initial settings.'
       else:
          msg2 = '\nActivating initial settings failed.'

       if not IsFileReadable(editConf.Filename):   # os.path.isfile() checks existence only, not accessibility
          showMessage(msg1+msg2)
          return False
       # Try to restore recent pre-edit settings.
       return EditConfigAndApply(None,
          startMessage    = None,
          successMessage  = msg1 + '\nActivated recent edit settings.',
          errorMessage    = msg1 + '\nFailed to activate recent edit settings.' + msg2,
          )

    ok = tryCopy() or tryCopy() # 2nd call may succeed if 1st could activate some of recent edit settings.

    global editConf
    if not editConf.load():
       if not editConf.needsUpgrade():
          message = 'Failed to read SMSCON settings.\n%s' % FatalHelp
          showMessage(message)
          sys.exit(1)
       if not editConf.upgrade():
          message = 'Failed to upgrade SMSCON settings.\n%s' % FatalHelp
          showMessage(message)
          sys.exit(1)
    if ok:
       message = 'Successfully loaded SMSCON settings.'
       showQuickMessage(message)
    else:
       message = 'Enforcedly loaded SMSCON settings.\nPlease check them all. Some seem to be invalid.'
       showMessage(message)
    return True

def EditConfigAndApply(newConf, successMessage = None, errorMessage = None, startMessage = 'Updating. Please wait...'):
    '''
    Update editConf and replace the main config with it. Restart daemon.
    @return bool Success indicator.
    '''
    global editConf
    ok = True
    if newConf:
       if startMessage:
          if areFeedbackDetailsDesired:
             showQuickMessage(startMessage)
       for k in newConf:
          ok = editConf.update(k, newConf[k]) and ok

    ok = ok and os.WEXITSTATUS(os.system('sudo smscon -import "%s"' % editConf.Filename)) == 0
    if ok:
       if areFeedbackDetailsDesired:
          if successMessage:
             showMessage(successMessage)
          else:
             pass
    else:
       if errorMessage:
          showMessage(errorMessage)
    return ok

class smsconEditorStartup(QtGui.QDialog):
    '''Application start up dialog'''
    
    def __init__(self, parent=None):
       ##Build the parent user interface
       QtGui.QDialog.__init__(self, parent)
       self.ui = Ui_smsconEditorStartupUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.showFullScreen()
       self.timer = QtCore.QBasicTimer()
       self.step = 0       
       self.timer.start(100, self)

    def timerEvent(self, event):

       if self.step >= 100:
          self.timer.stop()
          self.close()
       self.step = self.step + 1
       self.ui.progressBarStart.setValue(self.step)

class smsconEditorPasswordRequest(QtGui.QDialog): ### FIXME: add 3 re-tries for wrong password.
    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QDialog.__init__(self, parent)
       self.ui = Ui_smsconEditorAskPassUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       #self.showFullScreen()
       self.isInitialPasswordEntered = None

       ##Connect the button
       QtCore.QObject.connect(self.ui.btnEnterPassword, QtCore.SIGNAL('clicked()'), self.doEnterPassword)

    def doEnterPassword(self):

       self.close()

    ##Close app if window closes without a valid password
    def closeEvent(self, event):
       """
       Check the entered password.
       This dialog MUST NOT RETURN TO THE APPLICATION in case the entered password is wrong
       """

       event.accept()
       try:
          EnteredPassword = self.ui.linePassword.text()
          if IsPasswordValid(EnteredPassword):
             self.isInitialPasswordEntered = EnteredPassword == InitialPassword 
             return
       except:
          pass

       if len(EnteredPassword) > 0:
          Hostname = os.uname()[1]
          message = 'N900[%s]\nSMSCON Editor: wrong password (length %d) entered.' % (Hostname, len(EnteredPassword))
          os.system('sudo smscon -sendsms MASTERNUMBER %s&' % shellArg(message))
       message = 'Incorrect password. Exiting.'
       showQuickMessage(message)
       sys.exit(1)

class smsconEditorMainWindow(QtGui.QMainWindow):
    def __init__(self, parent=None):

       global editConf

       self.isUsersFirstTimeCall = not os.path.isfile(editConf.Filename)
       if self.isUsersFirstTimeCall:
          showQuickMessage('The initial password is %s' % InitialPassword)

       self.isSmsconTestActive = False
        
       ##Open Dialog - startup window
       startupDialog = smsconEditorStartup()
       startupDialog.exec_()
        
       ##Open Dialog - request password
       passwordDialog = smsconEditorPasswordRequest()
       passwordDialog.exec_()
        
       if self.isUsersFirstTimeCall:
          message =   'Seems this is your first time here. You now should\n' \
                      '\n1. Edit all settings to your needs.' \
                      '\n2. Check operation on the "Test" window.' \
                      '\n3. Enable SMSCON on device boot and' \
                      '\n4. Start SMSCON operation.'
          if passwordDialog.isInitialPasswordEntered:
             message += '\n\nAnd please go to "Change Password" and do so!'
          showMessage(message)

       elif passwordDialog.isInitialPasswordEntered:
          message = 'Please go to "Change Password" and do so!'
          showMessage(message)

       ##Make a copy of the config for editing
       CopyMainToEditConfig()
        
       if editConf['SENDERNUMBER'] != editConf["MASTERNUMBER"]:
          message =   'WARNING: Recent controlling SMS was not sent by master:' \
                      '\n%s is recent number.'    % editConf['SENDERNUMBER'] + \
                      '\n%s is master number.'    % editConf["MASTERNUMBER"]
          showMessage(message)

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorMainUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)

       ##Connect the GUI Buttons with actions
       QtCore.QObject.connect(self.ui.btnStart,    QtCore.SIGNAL('clicked()'), self.doStart)
       QtCore.QObject.connect(self.ui.btnStop,     QtCore.SIGNAL('clicked()'), self.doStop)
       QtCore.QObject.connect(self.ui.btnBoot,     QtCore.SIGNAL('clicked()'), self.doBoot)
       QtCore.QObject.connect(self.ui.btnUnBoot,   QtCore.SIGNAL('clicked()'), self.doUnBoot)
        
       ##Connect the GUI Buttons with windows
       QtCore.QObject.connect(self.ui.pushButtonGeneral,  QtCore.SIGNAL('clicked()'), self.doGeneral)
       QtCore.QObject.connect(self.ui.pushButtonCommands, QtCore.SIGNAL('clicked()'), self.doCommands)
       QtCore.QObject.connect(self.ui.pushButtonEmail,    QtCore.SIGNAL('clicked()'), self.doEmail)
       QtCore.QObject.connect(self.ui.pushButtonSSH,      QtCore.SIGNAL('clicked()'), self.doSSH)
       QtCore.QObject.connect(self.ui.pushButtonHost,     QtCore.SIGNAL('clicked()'), self.doHost)
       QtCore.QObject.connect(self.ui.pushButtonInit,     QtCore.SIGNAL('clicked()'), self.doInit)
       QtCore.QObject.connect(self.ui.pushButtonTest,     QtCore.SIGNAL('clicked()'), self.doTest)
       QtCore.QObject.connect(self.ui.pushButtonAbout,    QtCore.SIGNAL('clicked()'), self.doAbout)
       QtCore.QObject.connect(self.ui.pushButtonNewPass,  QtCore.SIGNAL('clicked()'), self.doUpdatePassword)
        

       ##Set values in the GUI.
       self.SetSystemValues()

    def closeEvent(self, event):
       """Cleanup on close."""
       event.accept()
       if self.isSmsconTestActive:
          message = 'Test mode was active.\nSwitching to normal operation...'
          showQuickMessage(message)
       self.doRestart();

    def SetSystemValues(self):
       '''Set GUI values from the current operation'''
       
       DAEMONSTATE = os.popen('sudo smscon -status').read().strip('\n') 
       BOOTSTATE = IsFileReadable('/etc/event.d/smscon_boot')
       
       if DAEMONSTATE.endswith('running'):
          self.ui.btnStop.setEnabled(True)
          self.ui.btnStop.setChecked(True)
          self.ui.btnStart.setEnabled(False)
       else:
          self.ui.btnStart.setEnabled(True)
          self.ui.btnStart.setChecked(True)
          self.ui.btnStop.setEnabled(False)
          
       if BOOTSTATE:
          self.ui.btnUnBoot.setEnabled(True)
          self.ui.btnUnBoot.setChecked(True)
          self.ui.btnBoot.setEnabled(False)
       else:
          self.ui.btnBoot.setEnabled(True)
          self.ui.btnBoot.setChecked(True)
          self.ui.btnUnBoot.setEnabled(False)
          
    def doStart(self):
       if self.isSmsconTestActive: # -start would not stop a running daemon and would left it at test mode.
          message = os.popen('sudo smscon -restart 2>&1').read()
       else:
          message = os.popen('sudo smscon -start 2>&1').read()
       self.isSmsconTestActive = False
       showMessage(message)
       self.SetSystemValues()

    def doRestart(self):
       message = os.popen('sudo smscon -restart 2>&1').read()
       self.isSmsconTestActive = False
       showMessage(message)
       self.SetSystemValues()

    def doStop(self):
       message = os.popen('sudo smscon -stop 2>&1').read()
       self.isSmsconTestActive = False
       showMessage(message)
       self.SetSystemValues()

    def doBoot(self):
       message = os.popen('sudo smscon -boot 2>&1').read()
       showMessage(message)
       self.SetSystemValues()

    def doUnBoot(self):
       message = os.popen('sudo smscon -unboot 2>&1').read()
       showMessage(message)
       self.SetSystemValues()

    ##Open new Windows:    
    def doAbout(self):
       self.aboutWin = smsconEditorAbout(self)

    def doGeneral(self):
       self.generalWin = smsconEditorGeneral(self)

    def doCommands(self):
       self.commandsWin = smsconEditorCommands(self)

    def doEmail(self):
       self.emailWin = smsconEditorEmail(self)
       
    def doSSH(self):
       self.sshWin = smsconEditorSSH(self)

    def doHost(self):
       self.hostWin = smsconEditorHost(self)
       
    def doInit(self):
       self.initWin = smsconEditorInit(self)
       
    def doTest(self):
       self.testWin = smsconEditorTest(self)
       
    def doUpdatePassword(self):
       ##Open Dialog - change password
       passwordChangeDialog = smsconEditorPasswordChange()
       passwordChangeDialog.exec_()

class smsconEditorInit(QtGui.QMainWindow):     ## Initialise SMSCON

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorInitUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Connect the GUI Buttons with actions
          ##Create Settings
       QtCore.QObject.connect(self.ui.btnInit,     QtCore.SIGNAL('clicked()'), self.doInit)
       QtCore.QObject.connect(self.ui.btnReset,    QtCore.SIGNAL('clicked()'), self.doReset)
       QtCore.QObject.connect(self.ui.btnDelIMSI,  QtCore.SIGNAL('clicked()'), self.doDelIMSI)
       QtCore.QObject.connect(self.ui.btnAddIMSI,  QtCore.SIGNAL('clicked()'), self.doAddIMSI)

    def doInit(self):
        os.system('sudo smscon -init!')
        message =   'SMSCON has been initialised to the default configuration.\n' \
                    'This editor will now quit...\n\n' \
                    'You will need to open it again and edit your new configuration.'
        showMessage(message)
        sys.exit()
        
    def doReset(self):

       # Note: DO NOT sys.exit() here! If we would do so, the recent editor copy of
       # active configuration would be (re-)established on next start of editor.
       # This is not what reset is intended to do.
       # Do not: sys.exit(0)            -- see below.
       # Do not: CopyMainToEditConfig() -- same thing.
       # Do not delete (or rename) the the recent editor copy. So that serves as a backup
       # in case making a new configuration fails before regularly terminating this editor. 

       os.system('sudo smscon -reset')
       message =   'SMSCON has been reset: The daemon has been stopped, all the config files were deleted' \
                   ' and the daemon has been removed from device boot sequence.\n' \
                   '\nIt is strongly recommended that you now:' \
                   '\n1. Initialise again then 2. Edit & test your settings' \
                   '\n3. Start SMSCON operation and 4. Enable SMSCON on device boot.'
       showMessage(message)

    def doDelIMSI(self):
       message = os.popen('sudo smscon -del imsi 2>&1').read()
       showMessage(message)

    def doAddIMSI(self):
       message = os.popen('sudo smscon -add imsi 2>&1').read()
       showMessage(message)
     
class smsconEditorGeneral(QtGui.QMainWindow):  ## General Settings

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorGeneralUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Set UI values from editConfig values
       global editConf
       
       self.ui.linePhoneMaster   .setText(editConf['MASTERNUMBER'])
       self.ui.linePhoneSender   .setText(editConf['SENDERNUMBER'])
       self.ui.spinBoxGPSTimeout .setValue(int(editConf['GPSTIMEOUT']))  
       self.ui.spinBoxGPSPolling .setValue(int(editConf['GPSPOLLING']))

       for idx, seconds in enumerate((10, 20, 30, 60, 120)):
          if int(editConf['GPSINTERVAL']) <= seconds:
             break;
       self.ui.comboBoxGPSInterval   .setCurrentIndex(idx)  # idx is 4 if GPSINTERVAL > 120 
       self.ui.comboBoxGPSSend       .setCurrentIndex(['sms', 'email', 'both'].index(editConf['GPSSEND'])) 

       self.ui.comboBoxReplyMethod   .setCurrentIndex(['sms', 'email', 'both', 'none'].index(editConf['MESSAGESEND'])) 
       self.ui.checkBoxCommandReply  .setEnabled(editConf['MESSAGESEND'] != 'none')
       self.ui.checkBoxLowBattery    .setEnabled(editConf['MESSAGESEND'] != 'none')
       self.ui.checkBoxKeyboardDetect.setEnabled(editConf['MESSAGESEND'] != 'none')
        
       self.ui.checkBoxCommandReply  .setChecked(configTrue(editConf['COMMANDREPLY']))
       self.ui.checkBoxLowBattery    .setChecked(configTrue(editConf['AUTOBATREPORT']))
       self.ui.checkBoxKeyboardDetect.setChecked(configTrue(editConf['KEYBOARDDETECT']))
       self.ui.checkBoxDeviceLock    .setChecked(configTrue(editConf['AUTODEVICELOCK']))
       self.ui.checkBoxNewSIMUnlock  .setChecked(configTrue(editConf['SIMUNLOCK']))
       self.ui.checkBoxSilence       .setChecked(configTrue(editConf['SILENCEDEVICE']))
           
       self.ui.checkBoxEnableRetry   .setChecked(configTrue(editConf['ENABLERESEND']))
       self.ui.doubleSpinBoxReplyWait.setEnabled(configTrue(editConf['ENABLERESEND']))
       self.ui.doubleSpinBoxReplyWait.setValue(float(editConf['RESENDTIME']))
       self.ui.spinBoxRetryAttempts  .setEnabled(configTrue(editConf['ENABLERESEND']))
       self.ui.spinBoxRetryAttempts  .setValue(int(editConf['RESENDMAXTRY']))
       
    def closeEvent(self, event):
       """Store settings on close."""
       event.accept()

       self.doCreateConfigGeneral()
       # self.doRestart();
        
    def doCreateConfigGeneral(self):

       newConf = {}
       newConf["MASTERNUMBER"]   = self.ui.linePhoneMaster.text()

       newConf["GPSTIMEOUT"]     = str(self.ui.spinBoxGPSTimeout.value())
       newConf["GPSPOLLING"]     = str(self.ui.spinBoxGPSPolling.value())
       newConf["GPSINTERVAL"]    = self.ui.comboBoxGPSInterval.currentText()
       newConf["GPSSEND"]        = self.ui.comboBoxGPSSend.currentText()

       newConf["MESSAGESEND"]    = self.ui.comboBoxReplyMethod.currentText()
       newConf["RESENDTIME"]     = str(self.ui.doubleSpinBoxReplyWait.value())
       newConf["RESENDMAXTRY"]   = str(self.ui.spinBoxRetryAttempts.value())
       newConf["ENABLERESEND"]   = configYesNo(self.ui.checkBoxEnableRetry.isChecked())

       newConf["COMMANDREPLY"]   = configYesNo(self.ui.checkBoxCommandReply.isChecked())
       newConf["AUTODEVICELOCK"] = configYesNo(self.ui.checkBoxDeviceLock.isChecked())
       newConf["AUTOBATREPORT"]  = configYesNo(self.ui.checkBoxLowBattery.isChecked())
       newConf["SIMUNLOCK"]      = configYesNo(self.ui.checkBoxNewSIMUnlock.isChecked())
       newConf["SILENCEDEVICE"]  = configYesNo(self.ui.checkBoxSilence.isChecked())
       newConf["KEYBOARDDETECT"] = configYesNo(self.ui.checkBoxKeyboardDetect.isChecked())

       newConf['SENDERNUMBER']   = newConf["MASTERNUMBER"]

       message =   'Settings have been successfully saved.' \
                   '\n%s remembered as master number.' % newConf["MASTERNUMBER"]
       if areFeedbackDetailsDesired:
          message +=  '\nAn SMS will be send to it if the SIM ever gets replaced.' \
                      '\n\nControlling SMS can be send from master or any other phone. ' \
                      'If numbers differ, SMS notifications will be send to both.'
       EditConfigAndApply(newConf, message)

class smsconEditorEmail(QtGui.QMainWindow):    ## Email Settings 

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorEmailUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Set UI values from editConfig values
       global editConf
       
       self.ui.lineEmailUsername .setText(editConf['USER'])
       self.ui.lineEmailPassword .setText(editConf['PASSWORD'])
       self.ui.lineEmailAddress  .setText(editConf['EMAILADDRESS'])
       self.ui.lineEmailFrom     .setText(editConf['EMAILFROM'])
       self.ui.lineEmailServer   .setText(editConf['MAILSERVER'])
       self.ui.lineEmailPort     .setText(editConf['MAILPORT'])

    def closeEvent(self, event):
       """Store settings on close."""
       event.accept()

       self.doCreateConfigEmail()
       # self.doRestart();
       
    def doCreateConfigEmail(self):

        newConf = {}
        newConf["USER"]           = self.ui.lineEmailUsername.text()
        newConf["PASSWORD"]       = self.ui.lineEmailPassword.text()
        newConf["EMAILADDRESS"]   = self.ui.lineEmailAddress.text()
        newConf["MAILSERVER"]     = self.ui.lineEmailServer.text()
        newConf["MAILPORT"]       = self.ui.lineEmailPort.text()
        newConf["EMAILFROM"]      = self.ui.lineEmailFrom.text()
        
        message =   'The mail settings have been successfully saved.'
        if areFeedbackDetailsDesired:
            message +=   '\nRecipient address: %s' % newConf["EMAILADDRESS"] + \
                         '\n' \
                         '\nSender address: %s' % newConf["EMAILFROM"] + \
                         '\nUser name: %s' % newConf["USER"] + \
                         '\nPassword: %s' % (newConf["PASSWORD"] and '******') + \
                         '\nMail server: %s - Port: %s' % (newConf["MAILSERVER"], newConf["MAILPORT"])
        EditConfigAndApply(newConf, message)
        
class smsconEditorCommands(QtGui.QMainWindow): ## Commands Renaming

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorCommandsUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Set UI values from editConfig values
       global editConf
       ##Make a copy of command strings because we may change them for displaying
       ##in case we detect old style of command prefixes:
       comConf = {}
       for cmd in editConf.getSmsCommandList():
          comConf[cmd] = editConf[cmd]

       prefix = editConf['SMS_COMPREFIX']
       if len(prefix) == 0:
          ##Check for old style prefix which was explicitly part of each
          ##command (except for COM_SHELL, where explicit prefix didn't work).
          prefix = commonPrefix(*(comConf[cmd] for cmd in comConf if cmd != 'COM_SHELL' ))
          if len(prefix) > 0:
             prefixRemoved = 0
             for cmd in comConf:
                if  comConf[cmd].startswith(prefix):
                   comConf[cmd] = comConf[cmd][len(prefix):]
                   prefixRemoved += 1
             if prefixRemoved:
                message =   'All your command names had a "%s" prefix ' % prefix + \
                            ' which has been removed and stored in the common command prefix field instead.' \
                            ' This way it\'s easier for you to change the prefix (and suffix if present).' \
                            '\n' \
                            '\nThe only difference is that now the prefix (and suffix, if present) ' \
                            ' must be send around the "Shell command" as well, e.g. send: %s' % \
                            prefix + comConf['COM_SHELL'] + editConf['SMS_COMSUFFIX']
                showMessage(message)
             else:
                prefix = ''

       ##Display the prefix and suffix separately ... 
       self.ui.linePrefix      .setText(prefix)
       self.ui.lineSuffix      .setText(editConf['SMS_COMSUFFIX']);
        
       ##... and show all the commands without the prefix (and without the suffix).
       self.ui.lineCheck       .setText(comConf['COM_CHECK'])
       self.ui.lineLog         .setText(comConf["COM_LOG"])        ##reply the smscon.log
       self.ui.lineReboot      .setText(comConf['COM_REBOOT'])
       self.ui.linePowerOff    .setText(comConf['COM_POWEROFF'])
       self.ui.linePower       .setText(comConf['COM_POWER'])
       self.ui.lineLocation    .setText(comConf['COM_LOCATION'])
       self.ui.lineRemoteOn    .setText(comConf['COM_REMOTEON'])
       self.ui.lineRemoteOff   .setText(comConf['COM_REMOTEOFF'])
       self.ui.lineCamera      .setText(comConf['COM_CAMERA'])
       self.ui.lineCall        .setText(comConf['COM_CALL'])
       self.ui.lineLock        .setText(comConf['COM_LOCK'])
       self.ui.lineUnlock      .setText(comConf['COM_UNLOCK'])
       self.ui.lineTrackOn     .setText(comConf['COM_TRACKON'])
       self.ui.lineTrackOff    .setText(comConf['COM_TRACKOFF'])
       self.ui.lineScript      .setText(comConf['COM_CUSTOM'])
       self.ui.lineScriptLog   .setText(comConf["COM_CUSTOMLOG"])  ##reply the COM_CUSTOM output
       self.ui.lineShell       .setText(comConf['COM_SHELL'])
       self.ui.lineAlarm       .setText(comConf['COM_ALARM'])
       self.ui.lineRestart     .setText(comConf['COM_RESTART'])

    def closeEvent(self, event):
       """Store settings on close."""
       event.accept()

       self.doCreateConfigCommands()
       # self.doRestart();
       
    def doCreateConfigCommands(self):

       newConf = {}
       newConf['SMS_COMPREFIX']    = self.ui.linePrefix.text()
       newConf['SMS_COMSUFFIX']    = self.ui.lineSuffix.text()

       newConf["COM_CHECK"]        = self.ui.lineCheck.text()
       newConf["COM_LOG"]          = self.ui.lineLog.text()   
       newConf["COM_LOCATION"]     = self.ui.lineLocation.text()
       newConf["COM_TRACKON"]      = self.ui.lineTrackOn.text()
       newConf["COM_TRACKOFF"]     = self.ui.lineTrackOff.text()
       newConf["COM_CAMERA"]       = self.ui.lineCamera.text()
       newConf["COM_CALL"]         = self.ui.lineCall.text()
       newConf["COM_LOCK"]         = self.ui.lineLock.text()
       newConf["COM_UNLOCK"]       = self.ui.lineUnlock.text()
       newConf["COM_REMOTEON"]     = self.ui.lineRemoteOn.text()
       newConf["COM_REMOTEOFF"]    = self.ui.lineRemoteOff.text()
       newConf["COM_POWER"]        = self.ui.linePower.text()
       newConf["COM_REBOOT"]       = self.ui.lineReboot.text()
       newConf["COM_POWEROFF"]     = self.ui.linePowerOff.text()
       newConf["COM_CUSTOM"]       = self.ui.lineScript.text()
       newConf["COM_CUSTOMLOG"]    = self.ui.lineScriptLog.text()
       newConf["COM_SHELL"]        = self.ui.lineShell.text()
       newConf["COM_ALARM"]        = self.ui.lineAlarm.text()
       newConf["COM_RESTART"]      = self.ui.lineRestart.text()

       message =      'The command names have been successfully saved.'
       if areFeedbackDetailsDesired:
          message +=  '\n' \
                      '\nCommon prefix: %s' % newConf['SMS_COMPREFIX'] + \
                      '\nCommon suffix: %s' % newConf['SMS_COMSUFFIX']
       EditConfigAndApply(newConf, message)
      
class smsconEditorSSH(QtGui.QMainWindow):      ## SSH Settings

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorSSHUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Set UI values from editConfig values
       global editConf
       self.ui.lineSSHUsername      .setText(editConf['REMOTEUSER'])
       self.ui.lineSSHPassword      .setText(editConf['REMOTEPASSWORD'])
       self.ui.lineSSHHost          .setText(editConf['REMOTEHOST'])
       self.ui.spinBoxSSHPort       .setValue(int(editConf['REMOTEPORT']))
       self.ui.lineSSHListenIP      .setText(editConf['REMOTELISTENIP'])
       self.ui.spinBoxSSHListenPort .setValue(int(editConf['REMOTELISTENPORT']))
       self.ui.spinBoxSSHLocalPort  .setValue(int(editConf['REMOTE2LOCALPORT']))

    def closeEvent(self, event):
       """Store settings on close."""
       event.accept()

       self.doCreateConfigSSH()
       # self.doRestart();
       
    def doCreateConfigSSH(self):

       newConf = {}
       newConf["REMOTEUSER"]       = self.ui.lineSSHUsername.text()
       newConf["REMOTEPASSWORD"]   = self.ui.lineSSHPassword.text()
       newConf["REMOTEHOST"]       = self.ui.lineSSHHost.text()
       newConf["REMOTEPORT"]       = str(self.ui.spinBoxSSHPort.text())
       newConf["REMOTELISTENIP"]   = self.ui.lineSSHListenIP.text()
       newConf["REMOTELISTENPORT"] = str(self.ui.spinBoxSSHListenPort.text())
       newConf["REMOTE2LOCALPORT"] = str(self.ui.spinBoxSSHLocalPort.text())

       message =   'The reverse SSH settings have been successfully saved.'
       if areFeedbackDetailsDesired:
          message +=  '\n' \
                      '\nUser name: %s'                  %  newConf["REMOTEUSER"] + \
                      '\nPassword: %s'                   % (newConf["REMOTEPASSWORD"] and '******') + \
                      '\nHost: %s - Port: %s'            % (newConf["REMOTEHOST"], newConf["REMOTEPORT"]) + \
                      '\nListening to IP: %s - Port: %s' % (newConf["REMOTELISTENIP"], newConf["REMOTELISTENPORT"]) + \
                      '\nForwarded Port: %s'             % (newConf["REMOTE2LOCALPORT"])
       EditConfigAndApply(newConf, message)

class smsconEditorHost(QtGui.QMainWindow):     ## Host Settings

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorHostUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Set UI values from editConfig values
       global editConf
       self.ui.lineHostAddress            .setText(editConf['CHECKHOST'])
       self.ui.doubleSpinBoxHostInterval  .setValue(float(editConf['CHECKTIME']))
       self.ui.checkBoxHostEnable         .setChecked(configTrue(editConf['ENABLECHECKHOST']))
       self.ui.lineHostAddress            .setEnabled(configTrue(editConf['ENABLECHECKHOST']))
       self.ui.doubleSpinBoxHostInterval  .setEnabled(configTrue(editConf['ENABLECHECKHOST']))

    def closeEvent(self, event):
       """Store settings on close."""
       event.accept()

       self.doCreateConfigHOST()
       # self.doRestart();
       
    def doCreateConfigHOST(self):

       newConf = {}
       newConf["CHECKHOST"]        = self.ui.lineHostAddress.text()
       newConf["CHECKTIME"]        = str(self.ui.doubleSpinBoxHostInterval.value())
       newConf["ENABLECHECKHOST"]  = configYesNo(self.ui.checkBoxHostEnable.isChecked())

       message =   'The remote host settings have been successfully saved.'
       if areFeedbackDetailsDesired:
          message +=  '\n' \
                      '\nCheck enabled: %s'   % newConf["ENABLECHECKHOST"] + \
                      '\nURL to check: %s'    % newConf["CHECKHOST"] + \
                      '\nCheck interval: %s'  % newConf["CHECKTIME"]
       EditConfigAndApply(newConf, message)

class smsconEditorTest(QtGui.QMainWindow):     ## Testing of main functions for settings correctness.

    def __init__(self, parent=None):

       ##Build parent user interface
       QtGui.QWidget.__init__(self, parent)
       self.ui = Ui_smsconEditorTestUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       

       ##Connect the GUI Buttons with the Tests 
       QtCore.QObject.connect(self.ui.btnTestMail1,   QtCore.SIGNAL('clicked()'), self.doSendTestMail1)
       QtCore.QObject.connect(self.ui.btnTestMail2,   QtCore.SIGNAL('clicked()'), self.doSendTestMail2)
       QtCore.QObject.connect(self.ui.btnTestGPS1,    QtCore.SIGNAL('clicked()'), self.doSendTestGPS1)
       QtCore.QObject.connect(self.ui.btnTestGPS2,    QtCore.SIGNAL('clicked()'), self.doSendTestGPS2)
       QtCore.QObject.connect(self.ui.btnTestCall,    QtCore.SIGNAL('clicked()'), self.doSendTestCall)
       QtCore.QObject.connect(self.ui.btnTestSMS,     QtCore.SIGNAL('clicked()'), self.doSendTestSMS)
       QtCore.QObject.connect(self.ui.btnTestSSH,     QtCore.SIGNAL('clicked()'), self.doSendTestSSH)
       QtCore.QObject.connect(self.ui.btnTestScript,  QtCore.SIGNAL('clicked()'), self.doSendTestScript)  
          
    def smsconTest(self, testArgs, testTopic):
       '''Initiate an SMSCON test using the given command line testArgs and given testTopic string'''
       
       if os.WEXITSTATUS(os.system('sudo smscon ' + testArgs)) == 0:
          self.isSmsconTestActive = True
          message = 'Test initiated:\n%s' % testTopic
          showQuickMessage(message)
       else:
          message = 'Test initiation failed.'
          showMessage(message)

    def doSendTestMail1(self):
       self.smsconTest('-test email1', 'An email should have been sent.')
        
    def doSendTestMail2(self):
       self.smsconTest('-test email2', 'An email with a picture taken with the front camera should have been sent.')
      
    def doSendTestGPS1(self):
       self.smsconTest('-test gps1', 'The GPS coordinates will be acquired and should be send once.')
      
    def doSendTestGPS2(self):
       self.smsconTest('-test gps2', 'The GPS coordinates will be acquired and should be send in regular periods.')
      
    def doSendTestCall(self):
       self.smsconTest('-test call', 'A phone call to the master number has been invoked.')
      
    def doSendTestSMS(self):
       self.smsconTest('-test sms', 'An SMS to the master number should have been send.')
      
    def doSendTestSSH(self):
       self.smsconTest('-test ssh', 'The reverse SSH connection should be possible now.')
      
    def doSendTestScript(self):
       self.smsconTest('-test script', 'The script execution has been invoked.')
      
class smsconEditorPasswordChange(QtGui.QDialog):

    def __init__(self, parent=None):
       ##Build the parent user interface
       QtGui.QDialog.__init__(self, parent)
       self.ui = Ui_smsconEditorNewPassUI()
       self.ui.setupUi(self)
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.show()       
          
       ##Connect the button
       QtCore.QObject.connect(self.ui.btnNewPassword, QtCore.SIGNAL('clicked()'), self.doNewPassword)
             
    def doNewPassword(self):

       NewPassword  = self.ui.lineNewPassword1.text()
       NewPassword2 = self.ui.lineNewPassword2.text()
          
       if NewPassword != NewPassword2:
          message = 'The two cases are not equal. Try again.'
          showMessage(message)
          return
       message = 'Storing new password, plase wait...'
       showQuickMessage(message)

       if NewPassword == '':
          NewPassword = InitialPassword

       try:
          try:
             os.remove(PassFile)
          except:
             pass 
          time.sleep(1)
          os.system('sudo "%s" %s "%s"' % (os.path.join(AppPath, 'smsconEditorCreatePass'),   # sudo wants abs path
                                           shellArg(NewPassword),
                                           PassFile))
       except:
          message = 'The new password could not be stored.'
       else: 
          message = 'The new password has been stored.'
       showMessage(message)
       self.close()
       
class smsconEditorAbout(QtGui.QMainWindow):
    '''About Window'''
    
    def __init__(self, parent=None):
       QMainWindow.__init__(self,parent)
       self.parent = parent
       self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
       self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
       self.setWindowTitle("About SMSCON Editor")

       aboutScrollArea = QScrollArea(self)
       aboutScrollArea.setWidgetResizable(True)
       awidget = QWidget(aboutScrollArea)
       awidget.setSizePolicy( QSizePolicy.Expanding, QSizePolicy.Expanding )
       aboutScrollArea.setSizePolicy( QSizePolicy.Expanding, QSizePolicy.Expanding )
       try:
          scroller = aboutScrollArea.property("kineticScroller").toPyObject()
          scroller.setEnabled(True)
       except:
          pass

       aboutLayout = QVBoxLayout(awidget)

       aboutIcon = QLabel()
       aboutIcon.setPixmap( QIcon.fromTheme('smscon-editor').pixmap(128,128) )
       aboutIcon.setAlignment( Qt.AlignCenter or Qt.AlignHCenter )
       aboutIcon.resize(128,128)
       
       aboutLayout.addWidget(aboutIcon)

       aboutLabel = QLabel('''<center>
       <br><b>SMSCON Editor</b><br>version: %s
       <br><br><b>The SMSCON Editor is a user interface to assist in the initialisation, 
       configuration and testing of SMSCON.</b>
       
       <br><br>Licenced under GPLv2
       <br>by <b>Christos Zamantzas</b> (Saturn)
       <br><br>Besides editing options and modifing secret pass-phrases to control the phone, 
       it allows to define email and sms contact, remote shell access (SSH) and a remote command server (HOST).
       <br><br>For more details on usage and implications see the wiki pages (click buttons below).
       
       <br><br><b>Thanks to:</b>
       <br><b>Frank Visser</b> (Digitalvoid) for 
       <br>for code to import, decrypt and encrypt the user variables from the configuration file.
       <br><br><b>Lutz Schwarz</b> (schwarz.ware@gmx.de)
       <br>for smsconlib, refacturing and improving the code as well as implementing many new features.
       <br><br><b>Benoit HERVIER</b> at http://khertan.net/
       <br>for code used in this about window.
       <br></center>''' % Version)
       
       aboutLabel.setWordWrap(True)
       aboutLabel.setAlignment( Qt.AlignCenter or Qt.AlignHCenter )

       aboutLayout.addWidget(aboutLabel)
       self.wiki1_button = QPushButton('SMSCON')
       self.wiki1_button.clicked.connect(self.open_wiki1)
       self.wiki2_button = QPushButton('SMSCON Editor')
       self.wiki2_button.clicked.connect(self.open_wiki2)
       
       awidget2 = QWidget()
       buttonLayout = QHBoxLayout(awidget2)        
       buttonLayout.addWidget(self.wiki1_button)
       buttonLayout.addWidget(self.wiki2_button)
       aboutLayout.addWidget(awidget2)

       awidget.setLayout(aboutLayout)
       aboutScrollArea.setWidget(awidget)
       self.setCentralWidget(aboutScrollArea)
       self.show()        

    def open_wiki1(self):
       QDesktopServices.openUrl(QUrl('http://wiki.maemo.org/SMSCON'))

    def open_wiki2(self):
       QDesktopServices.openUrl(QUrl('http://wiki.maemo.org/SMSCON_Editor'))

class smsconEditorMain():
        
    ##Check with what priviledges the GUI has been executed.
    if os.geteuid() == 0:
       message ='Please do not run SMSCON Editor as root.'
       showMessage(message)
       sys.exit(1)

    ##Check if temp folder is available   
    try:
       os.path.isdir(TempPath) or os.mkdir(TempPath)
    except:
       message = 'Failed to create "%s" folder. Giving up.' % TempPath
       showMessage(message)
       sys.exit(1)

    app = QtGui.QApplication(sys.argv)
    #Open Main Window
    myapp = smsconEditorMainWindow()
    myapp.show()

    sys.exit(app.exec_())
