/*
 * check_func.c - functions for different checks.
 * This file is part of PetrSU KP Library.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * PetrSU KP Library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PetrSU KP Library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include "check_func.h"
#include "../properties.h"
#include "../ss_subscribe.h"

/******************************************************************************/
/***************************** External functions *****************************/
/**
 * @brief Checks string to null or empty value.
 *
 * @param string string for check.
 *
 * @return true if string is not null or empty, false otherwise.
 */
bool is_str_null_or_empty(const char* string)
{
    if (string == NULL) {
        return true;
    }

    if (strlen(string) == 0) {
        return true;
    }

    return false;
}


/**
 * @brief Checks class structure.
 *
 * Checks: class struct not equals NULL; RTTI equals class; classtype set;
 *
 * @param class class for check.
 *
 * @return ERROR_NO if checks not fails or error code otherwise.
 */
int verify_class(const class_t *oclass)
{
    int error_code = ERROR_NO;
    
    if (oclass == NULL) {
        error_code = ERROR_INCORRECT_CLASS;
    } else if (oclass->rtti != RTTI_CLASS) {
        error_code = ERROR_INCORRECT_CLASS;
    } else if (is_str_null_or_empty(oclass->classtype) == 1) {
        error_code = ERROR_INCORRECT_CLASSTYPE;
    }

    return error_code;
}


/**
 * @brief Checks individual structure.
 *
 * Checks: individual structure not equals NULL; RTTI equals individual;
 * class set; class type; classtype set and equals parent class type;
 *
 * @param individual individual for check.
 *
 * @return ERROR_NO if checks not fails or error code otherwise.
 */
int verify_individual(const individual_t *individual)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s verify_individual START", \
            KPLIB_DEBUG_CHECKS_PREFIX);
    int error_code = ERROR_NO;

    if (individual == NULL) {
        error_code = ERROR_INCORRECT_INDIVIDUAL;
    } else if (individual->rtti != RTTI_INDIVIDUAL) {
        error_code = ERROR_INCORRECT_INDIVIDUAL;
    } else if (is_str_null_or_empty(individual->classtype) == 1) {
        error_code = ERROR_INCORRECT_CLASSTYPE;
    } else if (individual->parent_class == NULL) {
        error_code = ERROR_INCORRECT_INDIVIDUAL;
    } else if (is_str_null_or_empty(individual->parent_class->classtype) == 1) {
        error_code = ERROR_INCORRECT_CLASSTYPE;
    } else if (strcmp(individual->parent_class->classtype,
            individual->classtype) != 0) {
        error_code = ERROR_INCORRECT_INDIVIDUAL;
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s verify_individual END: error_code = %i", \
            KPLIB_DEBUG_CHECKS_PREFIX, error_code);

    return error_code;
}


/**
 * @brief Checks entity(class, individual, property) structure.
 *
 * @param entity one of structures: class, individual, property.
 *
 * @return ERROR_NO if checks not fails or error code otherwise.
 */
int verify_entity(const void *entity)
{
    if (entity == NULL) {
        return ERROR_INCORRECT_ENTITY;
    }

    int *rtti_type = (int *) entity;
    
    if ((*rtti_type >= RTTI_MAX_VALUE) || (*rtti_type <= RTTI_MIN_VALUE)) {
        return ERROR_INCORRECT_ENTITY;
    }
    
    int error_code = ERROR_NO;
    switch (*rtti_type) {
        case RTTI_CLASS: {
            error_code = verify_class((class_t *) entity);
            break;
        } case RTTI_INDIVIDUAL: {
            error_code = verify_individual((individual_t *) entity);
            break;
        } case RTTI_PROPERTY: {
            error_code = verify_property((property_t *) entity);
            break;
        } case RTTI_SBRC_CONTAINER: {
            error_code = verify_subscription_container(
                    (subscription_container_t *) entity);
        }
    }
    return error_code;
}


/**
 * @brief Checks property structure.
 *
 * Checks: class struct not equals NULL, RTTI, property name, type and about field.
 *
 * @param prop property for check.
 *
 * @return 0 if checks not fails or error code otherwise.
 */
int verify_property(const property_t *prop)
{
    int error_code = ERROR_NO;

    if (prop == NULL) {
        error_code = ERROR_INCORRECT_PROPERTY;
    } else if (prop->rtti != RTTI_PROPERTY) {
        error_code = ERROR_INCORRECT_PROPERTY;
    } else if (is_str_null_or_empty(prop->name) == true) {
        error_code = ERROR_INCORRECT_PROPERTY_NAME;
    } else if (is_str_null_or_empty(prop->about) == true) {
        error_code = ERROR_INCORRECT_PROPERTY_ABOUT;
    } else if (prop->type != OBJECTPROPERTY
                && prop->type != DATATYPEPROPERTY) {
        error_code = ERROR_INCORRECT_PROPERTY_TYPE;
    }

    return error_code;
}


//INFO: No checks for container data.
/**
 * @brief Checks subscription container structure.
 *
 * Checks: class struct not equals NULL, RTTI and list of subscription data.
 *
 * @param container subscription container for check.
 *
 * @return 0 if checks not fails or error code otherwise.
 */
int verify_subscription_container(const subscription_container_t *container)
{
    int error_code = ERROR_NO;
/*

    if (container == NULL) {
        error_code = ERROR_INCORRECT_SUBSCR_CONTAINER;
    } else if (container->rtti != RTTI_SBRC_CONTAINER) {
        error_code = ERROR_INCORRECT_SUBSCR_CONTAINER;
    }
*/

/*
    list_t *list_walker = container->sbrc_data;
    list_for_each(list_walker, &container->sbrc_data->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         subscription_data_t *sbcr_data = (subscription_data_t *)list->data;

         if (verify_individual_prop_list(sbcr_data->ind, sbcr_data->properties)) {
             return ERROR_INCORRECT_PROPLIST;
         }
    }
*/


    return error_code;
}


/**
 * @brief Checks individual and property list.
 *
 * Checks: individual and try to find properties for given individual.
 *
 * @param individual individual for check.
 * @param prop_list properties list for chek for given individual.
 *
 * @return 0 if checks not fails or error code otherwise.
 */
int verify_individual_prop_list(const individual_t *individual, list_t *prop_list)
{
    int error_code = verify_individual(individual);

    if (error_code != ERROR_NO) {
        return error_code;
    }

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &prop_list->links) {
         list_t *node = list_entry(list_walker, list_t, links);
         property_t *test_prop = (property_t *) node->data;

         if (verify_property(test_prop) != ERROR_NO) {
             return ERROR_INCORRECT_PROPLIST;
         }

         if (get_property_type(individual->parent_class, test_prop->name) == NULL) {
             return ERROR_INCORRECT_PROPLIST;
         }

    }

    return error_code;
}
