/*
 * structures.c - functions for work with C structures for OWL entities, KP and SS.
 * This file is part of PetrSU KP Library.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * PetrSU KP Library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PetrSU KP Library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include "structures.h"
#include "properties.h"
#include "classes.h"


/******************************************************************************/
/*************************** Static functions list ****************************/
static void remove_instance_from_class(class_t *oclass, individual_t *individual);
static void free_data_property_value_data(prop_val_t *prop_val);
static void free_object_property_value_data(prop_val_t *prop_val);



/******************************************************************************/
/***************************** External functions *****************************/
/**
 * @brief Free property structure.
 *
 * Free all fields, subproperties not freed, only list structure and all nodes without data.
 *
 * @param property property for free.
 */
void free_property(property_t *property)
{
    if (property == NULL) {
        return;
    }

    property->rtti = RTTI_MIN_VALUE;

    remove_entity_from_repository((void *) property);

    free(property->name);
	free(property->domain);
	free(property->about);

    property->name = NULL;
	property->domain = NULL;
	property->about = NULL;

    list_free_with_nodes(property->oneof, NULL);
    list_free_with_nodes(property->subpropertyof, NULL);

    property->oneof = NULL;
    property->subpropertyof = NULL;

    free(property);
}


/**
 * @brief Free property value struct.
 *
 * Property not freed.
 *
 * @param prop_val property value structure.
 * @param free_data_func function for free property value or NULL.
 */
void free_property_value_with_func(prop_val_t *prop_val, void (*free_data_func)(void*))
{
    if (prop_val == NULL) {
        return;
    }

    prop_val->property = NULL;

    if (free_data_func != NULL && prop_val->prop_value != NULL) {
        free_data_func(prop_val->prop_value);

    }
    prop_val->prop_value = NULL;
    free(prop_val);
}


/**
 * @brief Free property value struct.
 *
 * It checks object or data property given and then free data or object.
 * Oject (individual) not freed if it has references from other individuals.
 * Property not freed.
 *
 * @param prop_val property value structure.
 */
void free_property_value(prop_val_t *prop_val)
{
    if (prop_val == NULL) {
        return;
    }

    if (prop_val->property == NULL) {
        free(prop_val);
        return;
    }

    if (prop_val->property->type == DATATYPEPROPERTY) {
        free_data_property_value_data(prop_val);
    } else if (prop_val->property->type == OBJECTPROPERTY) {
        free_object_property_value_data(prop_val);
    }

    free(prop_val);
}




/**
 * @brief Free class structure.
 *
 * Properties, instances and superclasses not freed, only list struct.
 *
 * @param class class structure.
 */
void free_class(class_t *oclass)
{
    if (oclass == NULL) {
        return;
    }

    oclass->rtti = RTTI_MIN_VALUE;

    remove_entity_from_repository((void *) oclass);

    free(oclass->classtype);
    list_free_with_nodes(oclass->instances,  (void (*)(void *)) free_individual);
    list_free_with_nodes(oclass->oneof, NULL);
    list_free_with_nodes(oclass->superclasses, NULL);
    list_free_with_nodes(oclass->properties, NULL);

    free(oclass);
}


// FIXME: free prop_value data field, need check object or data property.
// FIXME: remove individual from global repository if no references and from parent class list.
/**
 * @brief Free individual structure.
 *
 * Parent class not free.
 *
 * @param class class structure.
 */
void free_individual(individual_t *individual)
{
    if (individual == NULL) {
        return;
    }

    if (count_references_to_individual(individual) > 0) {
        return;
    }


    remove_instance_from_class((class_t *) individual->parent_class, individual);
    remove_entity_from_repository((void *) individual);

    individual->rtti = RTTI_MIN_VALUE;
    
    free(individual->uuid);
    free(individual->classtype);

    individual->classtype = NULL;
    individual->uuid = NULL;
    individual->parent_class = NULL;

    list_free_with_nodes(individual->properties, (void (*)(void *))  free_property_value);
    individual->properties = NULL;

    free(individual);
}


/**
 * @brief Gets RTTI.
 *
 * @param entity some entity: individual, class or property.
 *
 * @return RTTI entity type on success or RTTI_MIN_VALUE (@see rtti_types) otherwise.
 */
int get_rtti_type(const void* entity)
{
    if (entity == NULL) {
        return RTTI_MIN_VALUE;
    }

    int *rtti_type = (int *) entity;

    if (*rtti_type >= RTTI_MAX_VALUE
            || *rtti_type <= RTTI_MIN_VALUE) {
        return RTTI_MIN_VALUE;
    }

    return *rtti_type;
}


/******************************************************************************/
/****************************** Static functions ******************************/
/**
 * @brief Remove individual from class instances list.
 *
 * @param class class for search instance.
 * @param individual individual for remove.
 */
static void remove_instance_from_class(class_t *oclass, individual_t *individual)
{
    if (oclass == NULL || individual == NULL) {
        return;
    }

    if (list_is_null_or_empty(oclass->instances) == 1) {
        return;
    }

    list_del_and_free_nodes_with_data(oclass->instances, (void *) individual, NULL);
}


/**
 * @brief Free object property value.
 *
 * It free only data field.
 *
 * @param prop_val value of property.
 */
static void free_object_property_value_data(prop_val_t *prop_val)
{
    if (prop_val->prop_value == NULL) {
        return;
    }

    individual_t *ind = (individual_t *) prop_val->prop_value;

    int count = count_references_to_individual(ind);
    
    if (count == 0) {
        free_individual(ind);
    }

    prop_val->prop_value = NULL;
}


/**
 * @brief Free data property value.
 *
 * It free only data field.
 *
 * @param prop_val value of property.
 */
static void free_data_property_value_data(prop_val_t *prop_val)
{
    if (prop_val->prop_value == NULL) {
        return;
    }

    free(prop_val->prop_value);

    prop_val->prop_value = NULL;
}
