/*
 * ss_subscribe.c - functions for work with subscribtions with SS.
 * This file is part of PetrSU KP Library.
 *
 * Copyright (C) 2009 - Alexandr A. Lomov. All rights reserved.
 *
 * PetrSU KP Library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PetrSU KP Library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <stdlib.h>
#include <pthread.h>
#include "ss_subscribe.h"
#include "structures.h"
#include "utils/check_func.h"
#include "kpi_low.h"
#include "ss_func.h"
#include "utils/util_func.h"
#include "utils/kp_debug.h"
#include "classes.h"
#include <string.h>
#include <bits/waitstatus.h>
#include <unistd.h>



/******************************************************************************/
/****************************** Structures list *******************************/
/** @brief Data for asynchronous subscribe thread. */
typedef struct async_sbcr_thread_s {
    pthread_t thread;   /**< thread data */
    int id;             /**< id */
} async_sbcr_thread_t;



/******************************************************************************/
/************************ Static and global entities **************************/
//static pthread_mutex_t g_sbcr_process_work;
/** @brief Mutex of subscribed containers list  */
static pthread_mutex_t g_sbcr_containers_mutext;

//static pthread_mutex_t g_sbcr_process_mutext;
//static bool g_sbrc_process_run = false;

/** @brief Subscribed containers list  */
static list_t *g_async_sbcr_containers = NULL;

/** @brief Use for exit from subscribe process */
static bool g_sbrc_process_stop = false;
//static list_t *sync_sbcr_containers = NULL;

static async_sbcr_thread_t async_sbcr_thread_info = { 0, -1 };



/******************************************************************************/
/*************************** Static functions list ****************************/
static bool is_async_sbrc_propcess_stoped();
static ss_triple_t* individual_to_triples_by_properties(individual_t *ind, list_t *properties);
static int async_subscribe(subscription_container_t *container);

static void *propcess_subscribe(void *data);
static void stop_async_sbrc_process();
static void start_async_sbrc_process();
static void add_async_subscribe(subscription_container_t *container);
static individual_t *get_or_create_individual_with_uuid(const char *class_type, const char *uuid);
static void update_individual_by_triples(individual_t *ind,
        ss_triple_t *old_triples, ss_triple_t *new_triples);
static void update_container(subscription_container_t *container,
        ss_triple_t *old_triples, ss_triple_t *new_triples);
static void update_async_container(subscription_container_t *container,
        ss_triple_t *old_triples, ss_triple_t *new_triples);
static int async_subscribe(subscription_container_t *container);
static ss_triple_t* individual_to_triples_by_properties(individual_t *ind, list_t *properties);



/******************************************************************************/
/***************************** External functions *****************************/
/**
 * @brief New subscription container.
 *
 * @return new subscription container on success or NULL otherwise.
 */
subscription_container_t* new_subscription_container()
{
    subscription_container_t *container =
            (subscription_container_t *) malloc(sizeof(subscription_container_t));

    if (container == NULL) {
        return NULL;
    }

    container->info.socket = 0;
    container->info.id[0] = '\0';

    container->rtti = RTTI_SBRC_CONTAINER;
    container->callback = NULL;
    container->sbrc_data = NULL;
    container->is_updated = false;
    container->subscription_id = 0;
    container->subscription_type = 0;

    return container;
}


/**
 * @brief New data for subscription container.
 *
 * Subscription data contains individual and list of properties or NULL for all
 * properties of individual.
 *
 * @param ind individual.
 * @param properties list of properties for subscribe or NULL for all properties.
 *
 * @return new subscription data on success or NULL otherwise.
 */
subscription_data_t* new_subscription_data(individual_t *ind, list_t *properties)
{
    if (verify_individual(ind) != 0) {
        return NULL;
    }

    subscription_data_t *data =
            (subscription_data_t *) malloc(sizeof(subscription_data_t));

    if (data == NULL) {
        return NULL;
    }

    data->ind = ind;
    data->properties = properties;

    return data;
}


/**
 * @brief Free subscription container data.
 *
 * Free subscription data structure, individual and properties not freed. You
 * can free properties list (only list structure will be freed, not properties),
 * using parameter free_prop_list.
 *
 * @param data subscription data for free.
 * @param free_prop_list if it equals true - list structure will be freed, without properties.
 */
void free_subscription_data(subscription_data_t *data, bool free_prop_list)
{
    if (data == NULL) {
        return;
    }

    if (free_prop_list == true) {
        list_free_with_nodes(data->properties, NULL);
    }
    data->properties = NULL;

    free(data);
}


/**
 * @brief Free subscription container.
 *
 * Free subscription container structure.
 * Also free subscription's data structures, individual and properties not freed,
 * only list structure of properties.
 *
 * @param container subscription container for free.
 */
void free_subscription_container(subscription_container_t *container)
{
    if (container == NULL) {
        return;
    }

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &container->sbrc_data->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         subscription_data_t *sbcr_data = (subscription_data_t *)list->data;

         free_subscription_data(sbcr_data, true);
    }

    list_free_with_nodes(container->sbrc_data, NULL);
    container->rtti = RTTI_MIN_VALUE;
    free(container);
}


/**
 * @brief Sets callback function for subscription.
 *
 * This functions call then some properties changed in container.
 *
 * @param container subscription container.
 * @param reference to callback function.
 */
void set_subscription_container_callback(subscription_container_t *container,
        void (*func)(subscription_container_t *, list_t *))
{
    if (container == NULL) {
        return;
    }

    container->callback = func;
}


/**
 * @brief Add individual to subscription.
 *
 * @param ind individual.
 * @param properties list of properties for subscribe or NULL for all properties.
 *
 * @return 0 on success or not otherwise.
 */
int add_individual_to_subscription_container(subscription_container_t *container,
        individual_t *ind, list_t *properties)
{
    if (verify_individual(ind) != ERROR_NO) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_BMED)
                ("\n%s add_individual_to_subscription_container: incorrect individual",
                KPLIB_DEBUG_SS_SBCR_PREFIX);
        return -1;
    }

    if (container == NULL) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_BMED)
                ("\n%s add_individual_to_subscription_container: incorrect container",
                KPLIB_DEBUG_SS_SBCR_PREFIX);
        return -1;
    }

    subscription_data_t *data = new_subscription_data(ind, properties);

    if (container->sbrc_data == NULL) {
        container->sbrc_data = list_get_new_list();
    }

    list_add_data(data, container->sbrc_data);

    return 0;
}


//static ss_subscribe_async(subscription_container_t *container)
/**
 * @brief Try to find individual with given uuid or create new individual.
 *
 * @param class_type type of individual, classtype of parent class.
 * @param uuid UUID of individual.
 *
 * @return individual (founded or new).
 */
static individual_t *get_or_create_individual_with_uuid(const char *class_type, const char *uuid)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s get_or_create_individual_with_uuid START: classtype = %s; uuid = %s", \
            KPLIB_DEBUG_SS_SBCR_PREFIX, class_type, uuid);
    individual_t *ind = (individual_t *) get_individual_from_repository_by_uuid(uuid);
    
    if (ind == NULL) {

        ss_triple_t * triple = NULL;
        ss_triple_t * triple_req = NULL;
        ss_add_triple(&triple_req, uuid, RDF_TYPE, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
        if(ss_query(get_ss_info(), triple_req, &triple) < 0) {
            printf("Unable to query sensors\n");
            return -1;
        }

        ss_delete_triples(triple_req);

        printf("\nClasstype = %s", triple->object);


        const class_t *oclass = get_class_from_repository_by_classtype(triple->object);
        ss_delete_triples(triple);
        ind = new_individual(oclass);
        ind->uuid = strdup(uuid);
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s get_or_create_individual_with_uuid END", \
            KPLIB_DEBUG_SS_SBCR_PREFIX);

    return ind;
}




/**
 * @brief Wait data from SS for container and update it.
 *
 * @param container subscription container.
 *
 * @return 0 on seccess or not otherwise.
 */
int wait_subscribe(subscription_container_t *container)
{
    ss_triple_t * n_val = NULL;
    ss_triple_t * o_val = NULL;

    while (true) {
        int status = ss_subscribe_indication(get_ss_info(), &container->info, &n_val, &o_val, 1000);

        if(status == 0) {
            continue; /* timeout */
        }

        if(status < 0) {
            //ss_perror(get_ss_info()->ss_errno);
            continue;
        }

        if (status == 1) {
            printf("\nReceived subscribe indication [ID = %s]\n", container->info.id);

           // update_container(container, o_val, n_val);

            ss_delete_triples(n_val);
            ss_delete_triples(o_val);

            n_val = NULL;
            n_val = NULL;

            container->is_updated = true;
            break;
        }
    }
    return 0;
}


/**
 * @brief Subscribe container.
 *
 * @param container subscription container for subscribe.
 * @param is_asynchronous sibscription mode (true - async, false - sync).
 *
 * @return 0 on seccess or not otherwise.
 */
int ss_subscribe_container(subscription_container_t *container, bool is_asynchronous)
{
    if (verify_subscription_container(container) != ERROR_NO) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Can't subscribe: invalid container", KPLIB_DEBUG_SS_SBCR_PREFIX);
        return ERROR_INCORRECT_SUBSCR_CONTAINER;
    }

    ss_triple_t *triples = NULL;

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &container->sbrc_data->links) {
        list_t *node = list_entry(list_walker, list_t, links);
        subscription_data_t *sbrc_data = (subscription_data_t *) node->data;
        ss_triple_t *new_triples = individual_to_triples_by_properties(sbrc_data->ind, sbrc_data->properties);

        triples = concat_triplets(triples, new_triples);
    }

    ss_triple_t *subsc_triples = NULL;

    if (ss_subscribe(get_ss_info(), &container->info, triples, &subsc_triples) < 0) {
        ss_delete_triples(triples);
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Can't subscribe (errno = %i).", \
            KPLIB_DEBUG_SS_SBCR_PREFIX, get_ss_info()->ss_errno);
        return -1;
    }
    ss_delete_triples(triples);

    triples = subsc_triples;
    list_t *node = list_entry(container->sbrc_data->links.next, list_t, links);
    subscription_data_t *data = (subscription_data_t *) node->data;

    while (triples != NULL) {
        printf("\nTriple %s = %s", triples->object, triples->subject);
        set_property_by_name(data->ind, triples->predicate, get_str_duplicate(triples->object));
        triples = triples->next;
    }

    container->is_updated = true;
    if (is_asynchronous == true) {
        add_async_subscribe(container);
        start_async_sbrc_process();
        //async_sbcr_thread_info.id = pthread_create(&async_sbcr_thread_info.thread, NULL, propcess_subscribe, NULL);
    }


    return 0;
}


/**
 * @brief Unsubscribe given container.
 *
 * @param container subscription container for unsubscribe.
 *
 * @return 0 on seccess or not otherwise.
 */
int ss_unsubscribe_container(subscription_container_t *container)
{

    if (g_async_sbcr_containers != NULL)  {
    list_t *del_node = NULL;
    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &g_async_sbcr_containers->links) {
         list_t *node = list_entry(list_walker, list_t, links);
         if (node->data == container) {
             del_node = node;
             break;
         }
    }


    if (del_node != NULL) {
        //printf("\nThread cancell = %i", pthread_cancel(async_sbcr_thread_info.thread));
        stop_async_sbrc_process();

        pthread_mutex_lock(&g_sbcr_containers_mutext);

        list_del_and_free_node(del_node, NULL);

        pthread_mutex_unlock(&g_sbcr_containers_mutext);
        start_async_sbrc_process();
        //async_sbcr_thread_info.id = pthread_create(&async_sbcr_thread_info.thread, NULL, propcess_subscribe, NULL);
    }
    }
    return ss_unsubscribe(get_ss_info(), &container->info);
}


/**
 * @brief Stop asynchronous subscribe.
 * Unsubscribe all asynchronous containers and remove all containers from list.
 */
void stop_all_subscriptions()
{
    stop_async_sbrc_process();

    list_free_with_nodes(g_async_sbcr_containers, (void (*)(void*)) free_subscription_container);
}



/******************************************************************************/
/***************************** Static functions *******************************/
/** @brief Checks process of asynchronous subscribe: stopped or work. */
static bool is_async_sbrc_propcess_stoped()
{
    return g_sbrc_process_stop;
}


/** @brief Checks asynschronous subscribe.
 *
 * Gets containers from asynchronous subscription list, checks for new data
 * for them, nad update container if needed.
 *
 * @param data information for process (not used now).
 */
static void *propcess_subscribe(void *data)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Subscribe process START.", KPLIB_DEBUG_SS_SBCR_PREFIX);

    list_t *list_walker = NULL;

    while (is_async_sbrc_propcess_stoped() == false) {
        pthread_mutex_lock(&g_sbcr_containers_mutext);

        subscription_container_t *container = NULL;
        list_walker = list_get_next_node(g_async_sbcr_containers, list_walker,
            (void *) &container);

        if (list_walker == NULL) {
            pthread_mutex_unlock(&g_sbcr_containers_mutext);
            break;
        }

        async_subscribe(container);
        pthread_mutex_unlock(&g_sbcr_containers_mutext);
        sleep(1);
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Subscribe process END (%i ).", \
        KPLIB_DEBUG_SS_SBCR_PREFIX, (int) async_sbcr_thread_info.thread);

    pthread_exit(NULL);
}


/** @brief Stop asynchronous subscription process. */
static void stop_async_sbrc_process()
{
    if (async_sbcr_thread_info.id != 0) {
        return;
    }

    g_sbrc_process_stop = true;
    pthread_join(async_sbcr_thread_info.thread, NULL);
    async_sbcr_thread_info.id = -1;
    pthread_mutex_lock(&g_sbcr_containers_mutext);
    pthread_mutex_unlock(&g_sbcr_containers_mutext);

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Async propcess stopped (thread = %i)", \
            KPLIB_DEBUG_SS_SBCR_PREFIX, (int) async_sbcr_thread_info.thread);
}


/** @brief Start asynchronous subscription process. */
static void start_async_sbrc_process()
{
    if (async_sbcr_thread_info.id != 0) {
        g_sbrc_process_stop = false;
        async_sbcr_thread_info.id = pthread_create(&async_sbcr_thread_info.thread, NULL, propcess_subscribe, NULL);

        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_HIGH)("\n%s Start async propcess, code = %i, thread = %i", \
            KPLIB_DEBUG_SS_SBCR_PREFIX, async_sbcr_thread_info.id, (int) async_sbcr_thread_info.thread);
    }
}


/**
 * @brief Adds subscription container to asynchronous subscription list
 *
 * @param container subscription container.
 */
static void add_async_subscribe(subscription_container_t *container)
{
    if (container == NULL) {
        return;
    }

    pthread_mutex_lock (&g_sbcr_containers_mutext);

    g_async_sbcr_containers = list_get_new_list_if_null(g_async_sbcr_containers);
    list_add_data(container, g_async_sbcr_containers);

    pthread_mutex_unlock(&g_sbcr_containers_mutext);
}


// FIXME: fucking function, kill and rewrite.
/**
 * @brief Update individual using old properties data and new.
 * Try to found property with given old data and update it by new.
 * It create new property if has only new data nd remove property if has only
 * old property data.
 *
 * @param ind individual.
 * @param old_triples old properties data.
 * @param new_triples new properties data.
 */
static void update_individual_by_triples(individual_t *ind,
        ss_triple_t *old_triples, ss_triple_t *new_triples)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s update_individual_by_triples START",\
            KPLIB_DEBUG_SS_SBCR_PREFIX);

    if (ind == NULL) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("%s update_individual_by_triples END: individual = NULL", KPLIB_DEBUG_SS_SBCR_PREFIX);
        return;
    }

    while (new_triples != NULL || old_triples != NULL) {
        char *uuid = (old_triples == NULL) ? new_triples->subject : old_triples->subject;

        if (strcmp(uuid, ind->uuid) != 0) {
            new_triples = new_triples->next;
            old_triples = old_triples->next;
            continue;
        }

        char *prop_name = (new_triples == NULL) ? old_triples->predicate : new_triples->predicate;
        property_t *prop_type = (property_t *) get_property_type(ind->parent_class, (const char *) prop_name);

        void *object = NULL;
        if (prop_type->type == OBJECTPROPERTY) {
            char *oclass = (new_triples == NULL) ? old_triples->subject : new_triples->subject;
            char *uuid = (new_triples == NULL) ? old_triples->object : new_triples->object;
            object = get_or_create_individual_with_uuid(oclass, uuid);
        } else {
            object = (new_triples == NULL) ? strdup(old_triples->object) : strdup(new_triples->object);
        }


        if (new_triples == NULL) {          // Triple was removed.
            unset_property_for_individual(ind, old_triples->predicate, object);
        } else if (old_triples == NULL) {   // New triple added.
            set_property_by_name(ind, new_triples->predicate, object);
        } else {                            // Triple was chanched.
            void *old_data = old_triples->object;
            if (prop_type->type == OBJECTPROPERTY) {
                old_data = (individual_t *) get_individual_from_repository_by_uuid(old_triples->object);
            }
            update_property_with_data(ind, old_triples->predicate, old_data, (void *) object);
        }

        if (new_triples != NULL) {
            new_triples = new_triples->next;
        }

        if (old_triples != NULL) {
            old_triples = old_triples->next;
        }
    }
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s update_individual_by_triples END",\
            KPLIB_DEBUG_SS_SBCR_PREFIX);
}


/**
 * @brief Update subscribe container.
 * Gets individuals from container call function for update individual
 * (@see update_individual_by_triples).
 *
 * @param container subscription container for update.
 * @param old_triples old properties data.
 * @param new_triples new properties data.
 */
static void update_container(subscription_container_t *container,
        ss_triple_t *old_triples, ss_triple_t *new_triples)
{
    if (container == NULL) {
        return;
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s update_container START",\
            KPLIB_DEBUG_SS_SBCR_PREFIX);

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &container->sbrc_data->links) {
        list_t *node = list_entry(list_walker, list_t, links);
        subscription_data_t *sbcr_data = (subscription_data_t *) node->data;

        update_individual_by_triples(sbcr_data->ind, old_triples, new_triples);
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)("\n%s update_container END",\
            KPLIB_DEBUG_SS_SBCR_PREFIX);

}


/**
 * @brief Update asynchronous subscribe container. (not use now).
 *
 * @param container subscription container for update.
 * @param old_triples old properties data.
 * @param new_triples new properties data.
 */
static void update_async_container(subscription_container_t *container,
        ss_triple_t *old_triples, ss_triple_t *new_triples)
{
    list_t *node = list_entry(container->sbrc_data->links.next, list_t, links);
    subscription_data_t *data = (subscription_data_t *) node->data;

    while (new_triples != NULL) {
        update_property_with_data(data->ind, old_triples->predicate, old_triples->object, (void *)strdup(new_triples->object));
        new_triples = new_triples->next;
        old_triples = old_triples->next;
    }
}


/**
 * @brief Checks data for container from SS and update container if needed.
 *
 * @param container subscription container for check.
 *
 * @return 0 on seccess or not otherwise.
 */
static int async_subscribe(subscription_container_t *container)
{
    ss_triple_t * n_val = NULL;
    ss_triple_t * o_val = NULL;


    int status = ss_subscribe_indication(get_ss_info(), &container->info, &n_val, &o_val, 100);

    if (status == 1) {
        printf("!!!Received subscribe indication [ID = %s]\n", container->info.id);
        update_async_container(container, o_val, n_val);
        update_container(container, o_val, n_val);

        ss_delete_triples(n_val);
        ss_delete_triples(o_val);

        n_val = NULL;
        n_val = NULL;

        container->is_updated = true;
    }

    return 0;
}


/**
 * @brief Convert individual properties to KPI Low triples.
 *
 * @param ind individual for convert.
 * @param properties properties for convert or NULL - all properties.
 *
 * @return triples on seccess or NULL otherwise.
 */
static ss_triple_t* individual_to_triples_by_properties(individual_t *ind, list_t *properties)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)
            ("\n%s individual_to_triples_by_properties START\n", KPLIB_DEBUG_SS_SBCR_PREFIX);
    if (verify_individual_prop_list(ind, properties) != ERROR_NO) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)
            ("\n%s individual_to_triples_by_properties END: prop_list error\n", KPLIB_DEBUG_SS_SBCR_PREFIX);

        return NULL;
    }

    list_t *convert_props =
            (properties == NULL) ? ind->parent_class->properties : properties;

    ss_triple_t *triples = NULL;

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &convert_props->links) {
         list_t *node = list_entry(list_walker, list_t, links);
         property_t *property = (property_t *) node->data;

        if (property->type == DATATYPEPROPERTY) {
             ss_add_triple(&triples, ind->uuid, property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
        }

        if (property->type == OBJECTPROPERTY) {
             ss_add_triple(&triples, ind->uuid, property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
             //ss_add_triple(&triples, ind->uuid, property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_BNODE);
        }

    }
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED)
            ("\n%s individual_to_triples_by_properties END\n", KPLIB_DEBUG_SS_SBCR_PREFIX);
    return triples;
}
    
