/**
* @file ss_properties.c
* @Author Vanag Pavel <vanag@cs.karelia.ru>
* @date   05 April, 2010
* @brief  functions for work with properties throw SS.
* @version 0.2alpha
*
* @section LICENSE
*
* PetrSU KP library is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* PetrSU KP library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with \<program name\>; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor,
* Boston, MA  02110-1301  USA
*
* @section DESCRIPTION
*
* This file is part of PetrSU KP library.
* There are functions for work with properties throw SS.
*
*/

#include "ss_properties.h"

#ifndef _SS_PROPERTIES_C_
#define _SS_PROPERTIES_C_


/**
* \fn int ss_set_property(void *entity, char *propname, void *data)
*
* \brief Sets property for entity in SS and localy by name.
*
* Checks either entity is individual or class
* Checks entity correctness: cardinality, property values limitation
* and sets property for entity in SS and localy.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] void *entity. Pointer to the individual struct.
* \param[in] char *propname. Pointer to the char array with name.
* \param[in] void *data. Pointer to the value.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_set_property(void *entity, char *propname, void *data)
{
	int status;
	int error_code = 0;

	status = set_property_by_name(entity, propname, data);
	if (status < 0) return status;

	switch (get_rtti_type(entity)) {
		case RTTI_CLASS: {
			error_code = ss_set_property_for_class((class_t *) entity, propname, data);
			break;
		}
		case RTTI_INDIVIDUAL: {
			error_code = ss_set_property_for_individual((individual_t *) entity, propname, data);
			break;
		}
		default: {
			error_code = -1;
		}
	}

	return error_code;

}

int ss_set_property_for_individual(individual_t *ind, char *propname, void *data)
{
	ss_triple_t * triples = NULL;

	property_t *prop = (property_t *) get_property_type(ind->parent_class, propname);

	if (prop->type == OBJECTPROPERTY) {
		if (get_rtti_type(data) != RTTI_INDIVIDUAL) {
			set_error(ERROR_INCORRECT_PROPERTY_VALUE);
			return -1;
		}
		individual_t *object = (individual_t *) data;
		ss_add_triple(&triples, ind->uuid, propname, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}
	else if (prop->type == DATATYPEPROPERTY) {
		ss_add_triple(&triples, ind->uuid, propname, (char *) data, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}


	ss_insert(get_ss_info(), triples, NULL);

	ss_delete_triples(triples);

	return 0;
}

int ss_set_property_for_class(class_t *ind, char *propname, void *data)
{
	return 0;
}


/**
* \fn int ss_set_property_with_value_struct(void *entity, prop_val_t *prop_val)
*
* \brief Sets property for entity in SS and localy by name.
*
* Checks either entity is individual or class
* Checks entity correctness: cardinality, property values limitation
* and sets property for entity in SS and localy.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] void *entity. Pointer to the individual struct.
* \param[in] prop_val_t *prop_val. Pointer to the value struct.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_set_property_with_value_struct(void *entity, prop_val_t *prop_val)
{
	return ss_set_property(entity, prop_val->property->name, prop_val->prop_value);
}

/**
* \fn prop_val_t* ss_get_property(individual_t *ind, char *propname)
*
* \brief Gets property for individual from SS by name.
*
* Gets the property from the SS by name and returns it.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Pointer to the char array with name.
* \return prop_val_t*. NULL if properties with given name doesn't exists,
*                      pointer to value struct otherwise.
*/
prop_val_t* ss_get_property(individual_t *ind, char *propname)
{
	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;
	void * data = NULL;
	char * uuid = NULL;

	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return NULL;
	}

	if (propname == NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return NULL;
	}

	property_t *prop = (property_t *) get_property_type(ind->parent_class, propname);
	if (!prop)
		return NULL;

	ss_add_triple(&request, ind->uuid, prop->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		printf("Unable to query\n");
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return NULL;
	}
	ss_delete_triples(request);
	if (first_triple == NULL) {
		printf("No triples returned\n");
		return NULL;
	}

	if (prop->type == OBJECTPROPERTY) {
		uuid = (char *) first_triple->object;
		const individual_t *ind_tmp = get_individual_from_repository_by_uuid(uuid);
		if (ind_tmp == NULL)
		{
			const class_t *oclass = get_class_from_repository_by_classtype(first_triple->predicate);
			individual_t *ind_tmp2 = new_individual(oclass);
			if (ind_tmp2 != NULL)
			{
				ind_tmp2->uuid = strdup(uuid);
				data = (void *) ind_tmp2;
			}
			else data = NULL;
		}
		else data = (void *) ind_tmp;
	}
	else if (prop->type == DATATYPEPROPERTY) {
		data = (void *) strdup(first_triple->object);
	}
	ss_delete_triples(first_triple);

	prop_val_t* value = new_prop_value(prop, data);

	return value;
}

/**
* \fn prop_val_t* ss_get_property_with_data(individual_t *ind, char *propname, void *data)
*
* \brief Gets first founded property for individual from SS by name and value.
*
* Gets the property from the SS by name and value and returns first founded.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Pointer to the char array with name.
* \param[in] void *data. Pointer to the value.
* \return prop_val_t*. NULL if properties with given name doesn't exists,
*                      pointer to value struct otherwise.
*/
prop_val_t* ss_get_property_with_data(individual_t *ind, char *propname, void *data)
{
	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;
	char *uuid;

	if (data == NULL)
		return ss_get_property(ind, propname);

	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return NULL;
	}

	if (propname == NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return NULL;
	}

	property_t *prop = (property_t *) get_property_type(ind->parent_class, propname);
	if (!prop)
		return NULL;

	if (prop->type == OBJECTPROPERTY) {
		if (get_rtti_type(data) != RTTI_INDIVIDUAL) {
			set_error(ERROR_INCORRECT_PROPERTY_VALUE);
			return NULL;
		}
		individual_t *object = (individual_t *) data;
		ss_add_triple(&request, ind->uuid, propname, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}
	else if (prop->type == DATATYPEPROPERTY) {
		ss_add_triple(&request, ind->uuid, propname, (char *) data, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}


	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return NULL;
	}

	ss_delete_triples(request);

	if (!first_triple) return NULL;

	if (prop->type == OBJECTPROPERTY) {
		uuid = (char *) first_triple->object;
		const individual_t *ind_tmp = get_individual_from_repository_by_uuid(uuid);
		if (ind_tmp == NULL)
		{
			const class_t *oclass = get_class_from_repository_by_classtype(first_triple->predicate);
			individual_t *ind_tmp2 = new_individual(oclass);
			if (ind_tmp2 != NULL)
			{
				ind_tmp2->uuid = uuid;
				data = (void *) ind_tmp2;
			}
			else data = NULL;
		}
		else data = (void *) ind_tmp;
	}
	else if (prop->type == DATATYPEPROPERTY) {
		data = (void *) first_triple->object;
	}
	ss_delete_triples(first_triple);

	prop_val_t* value = new_prop_value(prop, (void *) data);

	return value;
}

/**
* \fn list_t* ss_get_properties(individual_t *ind, char *propname, int max)
*
* \brief Gets list of properties for individual from SS by name and value.
*
* Gets the property from the SS by name and value and returns list of properties.
* Returns all founded but not more then max.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Pointer to the char array with name.
* \param[in] int max. Maximum returns properties.
* \return list_t*. NULL if properties with given name doesn't exists,
*                      pointer to list of value structs otherwise.
*/


list_t* ss_get_properties(individual_t *ind, char *propname, int max)
{
	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;
	ss_triple_t * triple = NULL;
	list_t *properties;
	void * data;
	int i;
	char *uuid;

	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return NULL;
	}

	if (propname == NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return NULL;
	}

	ss_add_triple(&request, ind->uuid, propname, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return NULL;
	}
	ss_delete_triples(request);
	if (first_triple == NULL) {
		return NULL;
	}


	property_t *prop = (property_t *) get_property_type(ind->parent_class, propname);

	properties = list_get_new_list();

	triple = first_triple;
	for(i = 0; i < max && triple; i++)
	{
		if (prop->type == OBJECTPROPERTY) {
			uuid = (char *) first_triple->object;
			const individual_t *ind_tmp = get_individual_from_repository_by_uuid(uuid);
			if (ind_tmp == NULL)
			{
				const class_t *oclass = get_class_from_repository_by_classtype(first_triple->predicate);
				individual_t *ind_tmp2 = new_individual(oclass);
				if (ind_tmp2 != NULL)
				{
					ind_tmp2->uuid = uuid;
					data = (void *) ind_tmp2;
				}
				else data = NULL;
			}
			else data = (void *) ind_tmp;
		}
		else if (prop->type == DATATYPEPROPERTY) {
			data = (void *) triple->object;
		}
		prop_val_t* value = new_prop_value(prop, (void *) data);
		list_t *node = list_get_new_node(value);

		list_add(&node->links, &properties->links);

		triple = triple->next;
		printf("i=%d\n",i);
	}

	ss_delete_triples(first_triple);

	return properties;

}


/**
* \fn bool ss_has_property_value(individual_t* ind, char *propname, void *data)
*
* \brief Checks if individual has property with given value.
*
* Checks if individual has property with given name and value.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Pointer to the char array with name.
* \param[in] void *data. Pointer to the value.
* \return int. 0 if property doesn't exists, 1 otherwise.
*/
bool ss_has_property_value(individual_t* ind, char *propname, void *data)
{
	prop_val_t* value;

	value = ss_get_property_with_data(ind, propname, data);

	if (value == NULL)
	{
		return false;
	}
	else
	{
		/*free_property_value(value, free);*/
		return true;
	}

}

/**
* \fn int ss_update_property_with_data(individual_t *ind, char *propname, void *old_data, void *new_data)
*
* \brief Updates property of individual in SS and localy.
*
* Checks individual correctness: cardinality, prop-
* erty values limitation, and converts the individual’s given
* properties to triplets, which will be updated in SS, other
* properties not changed in SS. Same localy.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Char array with name of property.
* \param[in] void *old_data. Pointer to the old value (DATATYPE or OBJECTTYPE).
* \param[in] void *new_data. Pointer to the new value (DATATYPE or OBJECTTYPE).
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_update_property_with_data(individual_t *ind, char *propname, void *old_data, void *new_data)
{
	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	if (propname == NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return -1;
	}

	property_t *prop = (property_t *) get_property_type(ind->parent_class, propname);
	if (!prop) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return -1;
	}

	ss_triple_t *old_triple = NULL;
	ss_triple_t *new_triple = NULL;

	reset_ss_errno();

	ss_add_triple(&old_triple, ind->uuid, prop->name, old_data, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	ss_add_triple(&new_triple, ind->uuid, prop->name, new_data, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	int code = ss_update(get_ss_info(), new_triple, old_triple, NULL);

	ss_delete_triples(old_triple);
	ss_delete_triples(new_triple);

	if (get_ss_info()->ss_errno != 0) {
		return -1;
	}

	if (update_property_with_data(ind, propname, old_data, new_data) != 0) {
		return -1;
	}

	return 0;
}

int ss_update_properties(individual_t *individual, list_t *properties)
{
	return 0;
}

/**
* \fn int ss_remove_properties_by_name(individual_t *ind, char *propname)
*
* \brief Removes all properties with given name for individual in SS and localy.
*
* Finds properties with given name for individual and removes it from SS.
* Then try find all local property with given name and remove it.
* After all you have a struct without removed properties.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] char *propname. Char array with name of property.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_remove_properties_by_name(individual_t *ind, char *propname)
{
	ss_triple_t * request = NULL;

	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	if (propname == NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return -1;
	}

	ss_add_triple(&request, ind->uuid, propname, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	if(ss_remove(get_ss_info(), request) < 0) {
		printf("Unable to remove\n");
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return -1;
	}
	ss_delete_triples(request);

	return 0;
}

/**
* \fn int ss_remove_property_by_value_struct(individual_t *ind, prop_val_t *prop_val)
*
* \brief Removes all properties with given name and value for individual in SS and localy.
*
* Get name and value from value struct, finds properties with given
* name and value for individual and removes it from SS.
* Then try find all local property with given name and value and remove it.
* After all you have a struct without removed properties.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * ind. Pointer to the individual struct.
* \param[in] prop_val_t *prop_val. Pointer to the value struct.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_remove_property_by_value_struct(individual_t *ind, prop_val_t *prop_val)
{
	ss_triple_t * request = NULL;

	reset_error();

	int error_code = verify_individual(ind);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	if (prop_val == NULL || prop_val->property==NULL) {
		set_error(ERROR_INCORRECT_PROPERTY_NAME);
		return -1;
	}

	property_t *prop = (property_t *) get_property_type(ind->parent_class, prop_val->property->name);
	if (!prop)
		return -1;

	if (prop_val->prop_value == NULL)
	{
		ss_add_triple(&request, ind->uuid, prop_val->property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}
	else if (prop->type == OBJECTPROPERTY) {
		if (get_rtti_type(prop_val->prop_value) != RTTI_INDIVIDUAL) {
			set_error(ERROR_INCORRECT_PROPERTY_VALUE);
			return -1;
		}
		individual_t *object = (individual_t *) prop_val->prop_value;
		ss_add_triple(&request, ind->uuid, prop_val->property->name, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}
	else if (prop->type == DATATYPEPROPERTY) {
		ss_add_triple(&request, ind->uuid, prop_val->property->name, (char *) prop_val->prop_value, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	}

	if(ss_remove(get_ss_info(), request) < 0) {
		printf("Unable to remove\n");
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return -1;
	}
	ss_delete_triples(request);

	return 0;
}


#endif	/* _SS_PROPERTIES_C_ */
