/*
 * ss_classes.c - functions for work with classes throw SS.
 * This file is part of PetrSU KP library.
 *
 * Copyright (C) 2009 - Pavel Vanag. All rights reserved.
 *
 * PetrSU KP library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#ifndef _SS_CLASSES_C
#define	_SS_CLASSES_C

#include "ss_classes.h"

/**
 * @brief Initialize individual for SS.
 *
 * This function generate UUID and registry given individuals in SS.
 *
 * @param individual_t individual for initialize.
 *
 * @return 0 on success or not otherwise.
 */
int init_individual(individual_t *ind)
{
    int error_code = verify_individual(ind);

    if (error_code != ERROR_NO) {
        return error_code;
    }

    ind->uuid = generate_uuid();

    ss_triple_t *triple = NULL;

    /* insert sensor to the smart space */
    ss_add_triple(&triple, ind->uuid, RDF_TYPE, ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	ss_insert(get_ss_info(), triple, NULL);
    ss_delete_triples(triple);
    triple = NULL;

    return 0;
}

/**
 * @brief Initialize individual for SS by given uuid.
 *
 * This function registry given individuals in SS with given uuid.
 *
 * @param individual_t individual for initialize.
 * @param const char * uuid of individual.
 *
 * @return 0 on success or not otherwise.
 */
int init_individual_with_uuid(individual_t *ind, const char *uuid)
{
    int error_code = verify_individual(ind);

    if (error_code != ERROR_NO) {
        return error_code;
    }

    if (is_str_null_or_empty(uuid) == true) {
        return -1;
    }

    ind->uuid = strndup(uuid, KPLIB_UUID_MAX_LEN);

    ss_triple_t *triple = NULL;

    ss_add_triple(&triple, ind->uuid, RDF_TYPE, ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	ss_insert(get_ss_info(), triple, NULL);
    ss_delete_triples(triple);
    triple = NULL;

    return 0;
}

/**
* \fn int ss_insert_individual(individual_t *individual)
*
* \brief Inserts given individual to SS
*
* Checks individual correctness: cardinality, property values limitation, and
* converts the individual to triplets, which will be placed to SS.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_insert_individual(individual_t *individual)
{
	int error_code = verify_individual(individual);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	ss_triple_t * triples = NULL;
	triples = individual_to_triples (individual);

	if (triples == NULL) return -1;

	if (ss_insert(get_ss_info(), triples, NULL) != 0)
	{
		printf("\nSS errno = %i, %s\n", get_ss_info()->ss_errno, get_ss_info()->node_id);
		ss_delete_triples(triples);
		return -1;
	}
	printf("\ninserted\n");

	ss_delete_triples(triples);
	return 0;
}

/**
* \fn int ss_insert_by_pattern(individual_t *individual, void *pattern)
*
* \brief Inserts given individual to SS by pattern
*
* Checks individual correctness: cardinality, property values limitation, and
* converts the individual to triplets by pattern, which will be placed to SS.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \param[in] void *pattern. Entity contains pattern (individual or class).
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_insert_by_pattern(individual_t *individual, void *pattern)
{
	int error_code = verify_individual(individual);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	ss_triple_t * triples = NULL;
	triples = individual_to_triples_by_pattern (individual, pattern);

	if (triples == NULL) return -1;

	ss_insert(get_ss_info(), triples, NULL);
	ss_delete_triples(triples);

	return 0;
}

/**
* \fn int ss_update_individual(individual_t *individual)
*
* \brief Updates given individual in SS
*
* Checks individual correctness: cardinality, property values limitation, and
* converts the individual to triplets, which will be updated in
* SS, other properties (which has’t individual, but that are in
* SS) will be removed from SS.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_update_individual(individual_t *individual)
{
	ss_remove_individual(individual);
	return ss_insert_individual(individual);
}

/**
 * \fn int ss_update_by_pattern(individual_t *individual, void *pattern)
 *
 * \brief Updates given individual in SS by pattern
 *
 * Checks individual correctness: cardinality, property values limitation, and
 * converts the individual to triplets by pattern, which will be updated in
 * SS, other properties (which has’t individual, but that are in
 * SS) will be removed from SS.
 * Function sets global PetrSU KP Library's errno.
 *
 * \param[in] individual_t * individual. Pointer to the individual struct.
 * \param[in] void * pattern. Pointer to the individual struct that considered as pattern.
 * \return int. Status of the operation when completed (0 if successfull, otherwise -1).
 * */
int ss_update_by_pattern(individual_t *individual, void *pattern)
{
        ss_remove_individual(individual);
        return ss_insert_by_pattern(individual, pattern);
}


/**
* \fn int ss_exists(individual_t *individual)
*
* \brief Checks if given individual exists in SS
*
* Tries to ﬁnd individual by given individual data (uuid, properties).
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return int. Status (1 if individual exists in SS, otherwise 0).
*/
int ss_exists(individual_t *individual)
{
	int error_code = verify_individual(individual);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;

	request = individual_to_triples (individual);

	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(request);
		return false;
	}
	ss_delete_triples(request);
	if (first_triple == NULL)
		return false;
	else {
		ss_delete_triples(first_triple);
		return true;
	}

}

/**
* \fn int ss_exists_class(class_t *class)
*
* \brief Checks if any individual of given class exists in SS
*
* Tries to ﬁnd individual of given class.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] class_t *class. Pointer to the class struct.
* \return int. Status (1 if individual exists in SS, otherwise 0).
*/
int ss_exists_class(class_t *oclass)
{
	int error_code = verify_class(oclass);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;

	request = class_to_triples (oclass);
	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		return false;
	}
	ss_delete_triples(request);
	if (first_triple == NULL)
		return false;
	else {
		ss_delete_triples(first_triple);
		return true;
	}

}

/**
* \fn int ss_exists_by_individual_uuid(individual_t *individual)
*
* \brief Checks if uuid of given individual exists in SS
*
* Tries to ﬁnd individual by uuid.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return int. Status (1 if individual exists in SS, otherwise 0).
*/
int ss_exists_by_individual_uuid(individual_t *individual)
{
	return ss_exists_by_uuid(individual->uuid);
}

/**
* \fn int ss_exists_by_uuid(char  *uuid)
*
* \brief Checks if individual with given uuid exists in SS
*
* Tries to ﬁnd individual by uuid.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] char *uuid. Char array with uuid.
* \return int. Status (1 if individual exists in SS, otherwise 0).
*/
int ss_exists_by_uuid(char  *uuid)
{
	ss_triple_t * request = NULL;
	ss_triple_t * first_triple = NULL;

	if (!uuid)
		return -1;

	/*ss_add_triple(&request, get_ss_info()->space_id, "_hasuuid", uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);*/
	ss_add_triple(&request, uuid, "_classtype", SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
	if(ss_query(get_ss_info(), request, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		return false;
	}
	ss_delete_triples(request);
	if (first_triple == NULL)
		return false;
	else {
		ss_delete_triples(first_triple);
		return true;
	}
}

/**
* \fn int ss_exists_by_pattern(void *pattern)
*
* \brief Checks if individual matches given pattern exists in SS
*
* Tries to ﬁnd individual by pattern.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] void *pattern. Pointer to the individual or class struct.
* \return int. Status (1 if individual exists in SS, otherwise 0).
*/
int ss_exists_by_pattern(void *pattern)
{
	if (get_rtti_type(pattern) == RTTI_INDIVIDUAL) {
		individual_t *individual = (individual_t *) pattern;
		return ss_exists(individual);
	}
	else if (get_rtti_type(pattern) == RTTI_CLASS) {
		class_t *oclass = (class_t *) pattern;
		return ss_exists_class(oclass);
	}
	else return false;
}

/**
* \fn list_t* ss_get_individuals_by_class(class_t *class)
*
* \brief Finds all individuals of given class in SS
*
* Tries to ﬁnd individual by class.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] class_t *class. Pointer to the class struct.
* \return list_t*. Null if there is no individuals of such class in SS,
*                 list of founded individuals otherwise.
*/
list_t* ss_get_individuals_by_class(class_t *oclass)
{
	int error_code = verify_class(oclass);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return NULL;
	}

	ss_triple_t *triple_rqst = NULL;
	ss_triple_t *triple_res = NULL;


	ss_add_triple(&triple_rqst, SS_RDF_SIB_ANY, "_classtype", oclass->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	if(ss_query(get_ss_info(), triple_rqst, &triple_res) < 0) {
		printf("\nQUERY: SS errno = %i, %s\n", get_ss_info()->ss_errno, get_ss_info()->node_id);
		ss_delete_triples(triple_rqst);
		return NULL;
	}
	ss_delete_triples(triple_rqst);
	if (triple_res == NULL)
	{
		printf("\nempty for %s\n", oclass->classtype);
		return NULL;
	}


	list_t *uuid_list = triples_to_individuals(triple_res);

	/*ss_triple_t *triples_walker = triple_res;
	while (triples_walker != NULL) {
		char *uuid = strdup(triples_walker->subject);
		printf("\nINFO: %s - %s - %s", triples_walker->subject, triples_walker->predicate, triples_walker->object);
		list_t *node = list_get_new_node(uuid);
		list_add_node(node, uuid_list);
		triples_walker = triples_walker->next;
	}*/

	ss_delete_triples(triple_res);

	return uuid_list;
}

/**
* \fn list_t* ss_get_individuals_by_pattern(individual_t *individual_pattern)
*
* \brief Finds all individuals matches given pattern in SS
*
* Tries to ﬁnd individual by pattern.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual_pattern. Pointer to the individual struct (pattern).
* \return list_t*. Null if there is no individuals of such class in SS,
*                 list of founded individuals otherwise.
*/
list_t* ss_get_individuals_by_pattern(individual_t *individual_pattern)
{
	printf("\n\n\t\tss_get_individuals_by_pattern\n");
	list_t *node = NULL;
	list_t *node3 = NULL;
	list_head_t *pos = NULL;
	list_head_t *pos2 = NULL;
	list_head_t *pos3 = NULL;
	list_head_t *tmp = NULL;
	prop_val_t *prop;
	property_t *proptype;
	ss_triple_t * triples = NULL;
	ss_triple_t * first_triple = NULL;
	individual_t *object;
	individual_t *ind;
	list_t *inds2;

	int error_code = verify_individual(individual_pattern);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return NULL;
	}

	if (individual_pattern->properties && !list_empty(&individual_pattern->properties->links)) {
		list_for_each(pos, &individual_pattern->properties->links) {
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (proptype == NULL) {
				set_error(ERROR_INCORRECT_PROPLIST);
				return NULL;
			}

			if (proptype->type == OBJECTPROPERTY) {
				if (prop->prop_value == NULL)
				{
					ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
				else
				{
					if (get_rtti_type(prop->prop_value) != RTTI_INDIVIDUAL) {
						set_error(ERROR_INCORRECT_PROPERTY_VALUE);
						return NULL;
					}
					object = (individual_t *) prop->prop_value;
					if (object->uuid)
						ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
					else
						ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
			}
			else if (proptype->type == DATATYPEPROPERTY)
			{
				ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, (char *) prop->prop_value, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
		}
	}
	else
	{
		set_error(ERROR_INCORRECT_PROPLIST);
		return NULL;
	}

	if(ss_query(get_ss_info(), triples, &first_triple) < 0) {
		set_error(ERROR_UNKNOWN);
		ss_delete_triples(triples);
		return NULL;
	}
	ss_delete_triples(triples);
	if (first_triple == NULL) {
		return NULL;
	}

	list_t *inds = triples_to_individuals (first_triple);

	if (inds && !list_empty(&inds->links))
	{
		list_for_each (pos, &individual_pattern->properties->links)
		{
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (proptype->type == OBJECTPROPERTY)
			{
				if (prop->prop_value != NULL)
				{
					object = (individual_t *) prop->prop_value;
					list_for_each_safe (pos2, tmp, &inds->links)
					{
						ind = (individual_t *)node->data;

						if (!(inds2 = ss_get_individuals_by_pattern(object)))
						{
							list_del(pos2);
							return inds;
						}

						if (list_empty(&inds2->links))
						{
							list_del(pos2);
							return inds;
						}

						list_for_each(pos3, &inds2->links)
						{
							node3 = list_entry(pos, list_t, links);

							if (!ss_has_property_value(ind, proptype->name, node3->data))
							{
								list_del(pos3);
								return inds;
							}
						}
					}
				}
			}
		}

	}
	else
	{
		ss_delete_triples(first_triple);
		return NULL;
	}

	ss_delete_triples(first_triple);

	return inds;
}

/**
* \fn int ss_remove_individual(individual_t *individual)
*
* \brief Removes given individual from SS
*
* Tries to ﬁnd individuals description(triplets) in the SS, then removes
* all triplets from SS and from individual, UUID removed too.
* After work you have empty individual struct.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return int. Status of the operation when completed (0 if successfull, otherwise -1).
*/
int ss_remove_individual(individual_t *individual)
{
	ss_triple_t * triples = NULL;

	int error_code = verify_individual(individual);

	if (error_code != ERROR_NO) {
		set_error(error_code);
		return -1;
	}

	triples = individual_to_triples (individual);

	if (triples == NULL) return -1;

	if (ss_remove(get_ss_info(), triples) != 0)
	{
		printf("\nSS errno = %i, %s\n", get_ss_info()->ss_errno, get_ss_info()->node_id);
		ss_delete_triples(triples);
		return -1;
	}

	ss_delete_triples(triples);
	return 0;
}

#endif	/* _SS_CLASSES_C */
