/*
 * properties.c - functions for work with properties.
 * This file is part of PetrSU KP library.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * PetrSU KP library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <string.h>
#include "properties.h"
#include "utils/list.h"


static int set_property_for_class(class_t *oclass, const char *propname, const void *data);
static int set_property_for_individual(individual_t *ind, const char *propname, const void *data);

static prop_val_t* get_property_value_by_value(individual_t *ind, prop_val_t *prop_value);
static list_t* get_property_node_by_value(individual_t *ind, prop_val_t *prop_value);

static int update_property_for_individual(individual_t *ind, const char *propname, \
        const void* old_data, const void *new_data);


static list_t* find_properties_values_by_name(individual_t *ind, const char *propname,\
        bool only_one);

static int verify_value_property(const property_t *prop, const void *data);
static int verify_property_value_struct(const prop_val_t *prop_val);

static void free_property_value_data(prop_val_t *prop_val);
int update_property_value_data(prop_val_t* prop_value, void* new_data);


static int check_cardinality_for_unset(individual_t *ind, prop_val_t *prop_val);

/* Not usedd functions */
/*
static int update_property_for_individual_with_data(individual_t *ind,
        const char *propname, void *old_data,  void *new_data);


static list_t* get_property_node_by_data(individual_t *ind, \
        const char *propname, const void *data);

*/



/******************************************************************************/
/**************************** External functions ******************************/
/**
 * @brief Set new property for entity.
 *
 * Set property for one individuals or for all individuals from class.
 *
 * @param entity class or individual.
 * @param propname name of the property.
 * @param value value to set.
 *
 * @return 0 on success or not otherwise.
 */
int set_property_by_name(void *entity, char *propname, void *value)
{
    if (entity == NULL
            || is_str_null_or_empty(propname) == true) {
        return -1;
    }

    int *rtti_type = (int *) entity;

    int error_code = 0;
    switch (*rtti_type) {
        case RTTI_CLASS: {
            error_code = set_property_for_class((class_t *) entity, propname, value);
            break;
        }
        case RTTI_INDIVIDUAL: {
            error_code = set_property_for_individual((individual_t *) entity, propname, value);
            break;
        }
        default: {
            error_code = -1;
        }
    }

    return error_code;
}


/**
 * @brief Set new property for entity.
 *
 * Set property for one individuals or for all individuals from class.
 *
 * @param entity class or individual.
 * @param property property.
 * @param value value to set.
 *
 * @return 0 on success or not otherwise.
 */
int set_property(void *entity, property_t *property, void *value)
{
    if (entity == NULL
            || verify_property((const property_t *) property) != 0
            || value == NULL) {
        return -1;
    }

    return set_property_by_name(entity, property->name, value);
}


/**
 * @brief Set property with ginen prop_val_t struct.
 *
 * Set property for one individuals or for all individuals from class.
 *
 * @param entity class or individual.
 * @param prop_val value to set.
 *
 * @return 0 on success or not otherwise.
 */
int set_property_with_prop_val(void *entity, const prop_val_t *prop_val)
{
    return set_property_by_name(entity, prop_val->property->name, prop_val->prop_value);
}


/**
 * @brief Get property by name.
 *
 * @param ind individual.
 * @param propname name of property.
 *
 * @return 0 on success or not otherwise.
 */
const prop_val_t* get_property(individual_t *ind, const char *propname)
{
    list_t *list = find_properties_values_by_name(ind, propname, true);

    if (list == NULL) {
        return NULL;
    }

    list_t* node = list_get_first_prev_node(list);

    if (node == NULL) {
        list_free_with_nodes(list, NULL);
        return NULL;
    }

    void *data = node->data;
    list_free_with_nodes(list, NULL);

    if (data == NULL) {
        return NULL;
    }

    return (prop_val_t *) data;
}


/**
 * @brief Get node of properties list with given property name .
 *
 * @param ind individual.
 * @param propname name of property.
 *
 * @return node of list on success or NULL otherwise.
 */
const list_t* get_property_node(individual_t *ind, const char *propname)
{
    list_t *list = find_properties_values_by_name(ind, propname, true);

    if (list == NULL) {
        return NULL;
    }

    list_t* node = list_get_first_prev_node(list);

    if (node == NULL) {
        return NULL;
    }

    return node;
}


/**
 * @brief Get property value with given data.
 *
 * @param ind individual.
 * @param propname name of property.
 * @param data value of property.
 *
 * @return property value struct on success or NULL otherwise.
 */
const prop_val_t* get_property_with_data(individual_t *ind,
        const char *propname, void *data)
{
    property_t *prop_type = (property_t *) get_property_type(
            ind->parent_class, propname);
    prop_val_t *test_prop_val = new_prop_value(prop_type, data);
    prop_val_t *prop_val = get_property_value_by_value(ind, test_prop_val);

    free_property_value_with_func(test_prop_val, NULL);

    return prop_val;
}


/**
 * @brief Get all properties with given name.
 *
 * @param ind individual.
 * @param propname name of property.
 *
 * @return properties values list on success or NULL otherwise.
 */
list_t* get_properties(individual_t *ind, const char *propname)
{
    return find_properties_values_by_name(ind, propname, false);
}


/**
 * @brief Update property with given name.
 *
 * @param ind individual.
 * @param propname name of property.
 * @param new_data value of property.
 *
 * @return 0 on success or not otherwise.
 */
int update_property(void *entity, const char *propname, void *new_data)
{
    return update_property_with_data(entity, propname, NULL, new_data);
}


/**
 * @brief Update property using old value.
 *
 * @param ind individual.
 * @param propname name of property.
 * @param old_data old value of property.
 * @param new_data new value of property.
 *
 * @return 0 on success or not otherwise.
 */
int update_property_with_data(void *entity, const char *propname,
        const void *old_data, void *new_data)
{
    int rtti_type = get_rtti_type(entity);

    if (rtti_type == -1) {
        return -1;
    }

    int error_code = 0;
    switch(rtti_type) {
        case RTTI_INDIVIDUAL: {
            error_code = update_property_for_individual(
                    (individual_t *) entity, propname, old_data, new_data);
            break;
        }
        case RTTI_CLASS: {
            // No code, maybe write later, if needed.
            break;
        }
    }

    return error_code;
}


/**
 * @brief Update property value.
 *
 * @param prop_value value of property.
 * @param new_data new value of property.
 *
 * @return 0 on success or not otherwise.
 */
int update_property_value_data(prop_val_t* prop_value, void* new_data)
{
    if (prop_value == NULL) {
        return ERROR_INCORRECT_PROPERTY_VALUE;
    }

    if (verify_value_property(prop_value->property, new_data) != 0) {
        return ERROR_INCORRECT_PROPERTY_VALUE;
    }

    free_property_value_data(prop_value);
    prop_value->prop_value = new_data;

    return 0;
}


/**
 * @brief Unset property value.
 *
 * @param ind individual.
 * @param propname name of property.
 * @param data value of property.
 *
 * @return 0 on success or not otherwise.
 */
int unset_property_for_individual(individual_t *ind, char *propname, void *data)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s unset_property_for_individual START", \
                KPLIB_DEBUG_CLASSSES_PREFIX);
    reset_error();
    
    if (verify_individual(ind) != 0) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
            ("\n%s unset_property_for_individual END: incorrect individual", \
            KPLIB_DEBUG_CLASSSES_PREFIX);
        return set_error(ERROR_INCORRECT_INDIVIDUAL);
    }

    list_t *prop_node = NULL;

    if (data == NULL) {
        prop_node = (list_t *) get_property_node(ind, propname);
    } else {
        property_t *prop_type = (property_t *) get_property_type(ind->parent_class, propname);
        prop_val_t *old_prop_value = new_prop_value(prop_type, (void *) data);
        
        prop_node = get_property_node_by_value(ind, old_prop_value);
        free_property_value_with_func(old_prop_value, NULL);
    }
        
    if (prop_node == NULL) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
            ("\n%s unset_property_for_individual END: no property", \
            KPLIB_DEBUG_CLASSSES_PREFIX);
        return 0;
    }

    if (check_cardinality_for_unset(ind, prop_node->data) != 0) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
            ("\n%s unset_property_for_individual END: min cardinality error", \
            KPLIB_DEBUG_CLASSSES_PREFIX);
        return set_error(ERROR_MINCARDINALITY_VIOLATED);
    }

    list_del(&prop_node->links);
    free_property_value_data((prop_val_t *) prop_node->data);
    list_free_node(prop_node, NULL);

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s unset_property_for_individual END", \
                KPLIB_DEBUG_CLASSSES_PREFIX);
    return 0;
}


/**
 * @brief Get property struct by given name.
 *
 * @param class class.
 * @param propname name of property.
 *
 * @return property struct on success or NULL otherwise.
 */
const property_t* get_property_type(const class_t *oclass, const char *propname)
{
    reset_error();
    int error_code = verify_class(oclass);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return NULL;
    }

    if (is_str_null_or_empty(propname) == true
            || error_code != 0) {
        set_error(ERROR_INCORRECT_PROPERTY_NAME);
        return NULL;
    }

    if (oclass->properties == NULL) {
        set_error(ERROR_CANT_FIND_PROPERTY);
        return NULL;
    }

    list_head_t *pos = NULL;
    list_for_each(pos, &oclass->properties->links) {
         list_t *list = list_entry(pos, list_t, links);
         property_t *test_prop = (property_t *)list->data;
         if (strcmp(test_prop->name, propname) == 0) {
            return test_prop;
        }
    }

    set_error(ERROR_CANT_FIND_PROPERTY);
    return NULL;
}


/**
 * @brief Checks property for individual.
 *
 * @param individual individual.
 * @param prop property.
 *
 * @return true if individual has property, otherwise false.
 */
bool is_individual_has_property(individual_t *individual, property_t *prop)
{
    return false;
}


/******************************************************************************/
/***************************** Static functions *******************************/
// FIXME: is_prop_values_equal need add is_properties_equal
/**
 * @brief Count how many properties has individual by given property value struct.
 *
 * Checks only properties references, data not checks.
 *
 * @param individual individual.
 * @param prop_val property value.
 *
 * @return count of properties.
 */
static int count_properties_by_prop_value(individual_t *ind, prop_val_t * prop_val)
{
    if (ind == NULL || prop_val == NULL) {
        return -1;
    }

    int prop_count = 0;
    
    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &ind->properties->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         prop_val_t *test_prop = (prop_val_t *)list->data;

         if (test_prop == NULL) {
             continue;
         }

         if (prop_val->property == test_prop->property) { //(is_prop_values_equal(prop_val, test_prop) == true) {
            ++prop_count;
        }
    }

    return prop_count;
}


/**
 * @brief Count how many properties has individual by given property.
 *
 * @param individual individual.
 * @param property_t property.
 *
 * @return count of properties.
 */
static int count_properties(individual_t *ind, property_t *prop)
{
    if (ind == NULL || prop == NULL) {
        return -1;
    }

    prop_val_t *prop_val = new_prop_value(prop, NULL);

    int count = count_properties_by_prop_value(ind, prop_val);

    free_property_value_with_func(prop_val, NULL);
    prop_val = NULL;

    return count;
}


/**
 * @brief Checks cardinality when property set.
 *
 * @param individual individual.
 * @param prop_val property value.
 *
 * @return 0 on success or -1 if cardinality violation.
 */
static int check_cardinality_for_set(individual_t *ind, prop_val_t *prop_val)
{
    if (ind == NULL || prop_val == NULL) {
        return -1;
    }

    int prop_count = count_properties_by_prop_value(ind, prop_val);

    int cardinality = prop_val->property->maxcardinality;
    printf("\nCardinallity = %i\n", prop_count);
    if (cardinality < 0) {
        return 0;
    }

    if (prop_count >= cardinality) {
        return -1;
    }

    return 0;
}


/**
 * @brief Checks cardinality when property unset.
 *
 * @param individual individual.
 * @param prop_val property value.
 *
 * @return 0 on success or -1 if cardinality violation.
 */
static int check_cardinality_for_unset(individual_t *ind, prop_val_t *prop_val)
{
    if (ind == NULL || prop_val == NULL) {
        return -1;
    }

    int prop_count = count_properties_by_prop_value(ind, prop_val);

    int cardinality = prop_val->property->mincardinality;

    if (cardinality < 0) {
        return 0;
    }

    if (prop_count <= cardinality) {
        return -1;
    }

    return 0;
}



/**
 * @brief Set property for individual.
 *
 * @param individual individual.
 * @param propname property name.
 * @param data data for set.
 *
 * @return 0 on success or not otherwise.
 */
static int set_property_for_individual(individual_t *ind,
        const char *propname, const void *data)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s set_property_for_individual START", \
                KPLIB_DEBUG_PROP_PREFIX);

    property_t *prop = (property_t *) get_property_type(ind->parent_class,
            propname);

    if (verify_value_property(prop, data) != 0) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s set_property_for_individual END: "\
                "property not verifed", KPLIB_DEBUG_PROP_PREFIX);
        return -1;
    }

    if (ind->properties == NULL) {
        ind->properties = list_get_new_list();
    }

    prop_val_t* value = new_prop_value(prop, (void *) data);

    if (check_cardinality_for_set(ind, value) == 0) {
        list_add_data(value, ind->properties);
    } else {
        free_property_value_with_func(value, NULL);
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s set_property_for_individual END: "\
                "cardinality error", KPLIB_DEBUG_PROP_PREFIX);
        return -1;
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) ("\n%s set_property_for_individual END", \
                KPLIB_DEBUG_PROP_PREFIX);
    return 0;
}


/**
 * @brief Set property for class.
 *
 * Set property for all individuals, owned by given class.
 *
 * @param class class.
 * @param propname property name.
 * @param data data for set.
 *
 * @return 0 on success or not otherwise.
 */
static int set_property_for_class(class_t *ind,
        const char *propname, const void *data)
{
    return 0;
}


/**
 * @brief Checks equality of given properties values.
 *
 * @param a one property value.
 * @param b another property value.
 *
 * @return true if equals or false otherwise.
 */
bool is_prop_values_equal(prop_val_t *a, prop_val_t *b)
{
    if (a == NULL && b == NULL) {
        return true;
    } else if (a == NULL || b == NULL) {
        return false;
    }

    property_t *prop_a = a->property;
    property_t *prop_b = b->property;

    if (strcmp(prop_a->name, prop_b->name) != 0 ) {
        return false;
    }

    if (strcmp(prop_a->about, prop_b->about) != 0) {
        return false;
    }

    if (prop_a->type != prop_b->type) {
        return false;
    }

    if (a->prop_value == NULL && b->prop_value == NULL) {
        return true;
    } else if (a->prop_value == NULL || b->prop_value == NULL) {
        return false;
    }

    if (prop_a->type == DATATYPEPROPERTY
            && strcmp(a->prop_value, b->prop_value) == 0) {
        return true;
    }

    if (prop_a->type == OBJECTPROPERTY
            && a->prop_value == b->prop_value) {
        return true;
    }
    return false;
}


/**
 * @brief Checks equality of properties names of given properties values.
 * 
 * @param a one property value.
 * @param b another property value.
 * 
 * @return true if equals or false otherwise.
 */
bool is_prop_names_equal(prop_val_t *a, prop_val_t *b)
{
	if (a == NULL && b == NULL) {
		return true;
	} else if (a == NULL || b == NULL) {
		return false;
	}

        property_t *prop_a = a->property;
	property_t *prop_b = b->property;

	if (strcmp(prop_a->name, prop_b->name) != 0 ) {
		return false;
	}

        if (strcmp(prop_a->about, prop_b->about) != 0) {
		return false;
	}

        if (prop_a->type != prop_b->type) {
		return false;
	}

        return true;
}


/**
 * @brief Gets proerty value by given value.
 *
 * @param ind individual.
 * @param prop_value property value.
 *
 * @return property value on success or NULL otherwise.
 */
static prop_val_t* get_property_value_by_value(individual_t *ind, prop_val_t *prop_value)
{
    if (ind == NULL || prop_value == NULL) {
        return NULL;
    }

    if (list_is_null_or_empty(ind->properties) == 1) {
        return NULL;
    }

    list_head_t *pos = NULL;
    list_for_each(pos, &ind->properties->links) {
         list_t *list = list_entry(pos, list_t, links);
         prop_val_t *test_prop = (prop_val_t *)list->data;

         if (is_prop_values_equal(prop_value, test_prop) == true) {

            return test_prop;
        }
    }

    return NULL;
}


/**
 * @brief Gets proerty value list node by given property value.
 *
 * @param ind individual.
 * @param prop_value property value.
 *
 * @return list of node on success or NULL otherwise.
 */
static list_t* get_property_node_by_value(individual_t *ind, prop_val_t *prop_value)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s get_property_node_by_value START", \
            KPLIB_DEBUG_PROP_PREFIX);

    if( list_is_null_or_empty(ind->properties) == 1 || prop_value == NULL) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s get_property_node_by_value END: " \
            "no properties or value not set", KPLIB_DEBUG_PROP_PREFIX);
        return NULL;
    }
    
    list_head_t *pos = NULL;

    list_for_each(pos, &ind->properties->links) {
         list_t *list = list_entry(pos, list_t, links);
         
         prop_val_t *test_prop = (prop_val_t *)list->data;


         if (is_prop_values_equal(prop_value, test_prop) == true) {
             KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s get_property_node_by_value END", \
                KPLIB_DEBUG_PROP_PREFIX);
            return list;
        }
    }

    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_LOW) ("\n%s get_property_node_by_value END: " \
            "no nodes", KPLIB_DEBUG_PROP_PREFIX);
    return NULL;
}


/**
 * @brief Gets proerties values with properties with given property name.
 *
 * @param ind individual.
 * @param propname property name.
 * @param only_one if true gets only one firs property value.
 *
 * @return properties values list on success or NULL otherwise.
 */
static list_t* find_properties_values_by_name(individual_t *ind, const char *propname, bool only_one)
{
     if (is_str_null_or_empty(propname) == true
            || verify_individual(ind) != 0 ) {
        return NULL;
    }

    if (ind->properties == NULL) {
        return NULL;
    }

    list_head_t *pos = NULL;
    list_t *prop_list = list_get_new_list();
    bool is_prop_find = false;

    list_for_each(pos, &ind->properties->links) {
         list_t *list = list_entry(pos, list_t, links);
         prop_val_t *prop_val = (prop_val_t *)list->data;

         if (strcmp(prop_val->property->name, propname) == 0) {
            list_add_data(prop_val, prop_list);
            is_prop_find = true;
        }

         if (is_prop_find == true && only_one == true) {
             break;
         }
    }

    if (list_empty(&prop_list->links) == 1) {
        list_free(prop_list);
        prop_list = NULL;
    }

    return prop_list;
}



/* FIXME: Add part for check property and individual, add free functions.
 * remove old data or return it.
 */
/**
 * @brief Update property.
 *
 * Find property with given old data and update it by new data.
 *
 * @param ind individual.
 * @param propname property name.
 * @param old_data old property value data.
 * @param new_data new property value data.
 *
 * @return properties values list on success or NULL otherwise.
 */
static int update_property_for_individual(individual_t *ind,
        const char *propname, const void *old_data, const void *new_data)
{
    prop_val_t *prop_value = NULL;

    if (old_data == NULL) {
        prop_value = (prop_val_t *) get_property(ind, propname);
    } else {
        property_t *prop_type = (property_t *) get_property_type(ind->parent_class, propname);
        prop_val_t *old_prop_value = new_prop_value(prop_type, (void *) old_data);
        prop_value = get_property_value_by_value(ind, old_prop_value);

        free_property_value_with_func(old_prop_value, NULL);
    }
    
    return update_property_value_data(prop_value, (void *) new_data);
}

// TODO: remove it, use function from structures.
/**
 * @brief Free property value.
 *
 * @param prop_val property value.
 */
static void free_property_value_data(prop_val_t *prop_val)
{
    if (prop_val == NULL) {
        return;
    }

    if (prop_val->property == NULL) {
        return;
    }

    if (prop_val->property->type == DATATYPEPROPERTY) {
        free(prop_val->prop_value);
        prop_val->prop_value = NULL;
    } else if (prop_val->property->type == OBJECTPROPERTY) {
        prop_val->prop_value = NULL;
    }
}


/**
 * @brief Checks data for property.
 *
 * Can be given data set for given property.
 *
 * @param property property.
 * @param data data for check.
 *
 * @return 0 on success or -1 otherwise.
 */
static int verify_value_property(const property_t *prop, const void *data)
{
    if (prop == NULL) {
        return -1;
    }

    if (prop->type == OBJECTPROPERTY) {
	if (data != NULL)
	{
	        int *rtti_type = (int *) data;

	        if (*rtti_type != RTTI_INDIVIDUAL) {
	            return -1;
	        }
	}
    } else if (prop->type != DATATYPEPROPERTY) {
        return -1;
    }

    return 0;
}


/**
 * @brief Checks property value.
 *
 * @param prop_value property value.
 *
 * @return 0 on success or -1 otherwise.
 */
static int verify_property_value_struct(const prop_val_t *prop_val)
{
    if (prop_val == NULL) {
        return -1;
    }

    property_t *prop = prop_val->property;
    void *data = prop_val->prop_value;

    return verify_value_property(prop, data);
}



/******************************************************************************/
/************************ Not used functions **********************************/
/* Template: Update all properties for individual with given name */
/*
int update_properties(const void *entity, const char *propname, const void *new_data)
{
    int rtti_type = get_rtti_type(entity);

    if (rtti_type == -1) {
        return -1;
    }

    switch(rtti_type) {
        case RTTI_INDIVIDUAL: {
            update_properties_for_individual((individual_t *) entity, propname, new_data);
            break;
        }
        case RTTI_CLASS: {

            break;
        }
    }


    return 0;
}
*/



/* Template: update property with data */
/*
 * FIXME: Add part for check property and individual, add free functions.
 * remove old data or return it.
 */
/*
static int update_property_for_individual_with_data(individual_t *ind,
        const char *propname, void *old_data,  void *new_data)
{
    property_t *prop = get_property_type(ind->parent_class, propname);

    prop_val_t *old_prop_value = new_prop_value(prop, old_data);
    prop_val_t *prop_value = get_property_value_by_value(ind, old_prop_value);

    if (prop_value == NULL) {
        return -1;
    }

    free_property_value(old_prop_value, NULL);
    free_property_value_data(prop_value);
    return update_property_value_data(prop_value, new_data);
}
*/

/* update s properties for individual */
/*
static int update_properties_for_individual(individual_t *ind, const char *propname,  void *new_data)
{
    list_t *properties = get_properties(ind, propname);

    if (properties == NULL) {
        return ERROR_CANT_FIND_PROPERTY;
    }

    list_head_t *pos = NULL;
    list_for_each(pos, &properties->links) {
         list_t *list = list_entry(pos, list_t, links);

         if (list->data == NULL) {
             continue;
         }

         prop_val_t *prop_val = (prop_val_t *)list->data;
         update_property_value_data(prop_val, new_data);
    }

    list_free(properties);

    return 0;
}
*/

//FIXIT: return list node, not prop_val_t
/*
static list_t* get_property_node_by_data(individual_t *ind, \
        const char *propname, const void *data)
{
    if (ind == NULL || is_str_null_or_empty(propname) == true) {
        return NULL;
    }

    property_t *prop_type = (property_t *) get_property_type(ind->parent_class, propname);
    prop_val_t *prop_value = new_prop_value(prop_type, (void *) data);

    list_t *node = NULL;//get_property_value_by_value(prop_value, prop_value);
    free_property_value_with_func(prop_value, NULL);

    return node;
}
*/
