/*
 * kpi_interface.c - functions for providing interface to kpi_low.
 * This file is part of PetrSU KP library.
 *
 * Copyright (C) 2009 - Pavel Vanag. All rights reserved.
 *
 * PetrSU KP library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#ifndef _KPI_INTERFACE_C_
#define	_KPI_INTERFACE_C_

#include "kpi_interface.h"
#include "utils/util_func.h"

void print_triples(ss_triple_t * triples)
{
	char *subject;
	char *predicate;
	char *object;

	while (triples)
	{
		subject = (char *) triples->subject;
		predicate = (char *) triples->predicate;
		object = (char *) triples->object;
		printf("%s --- %s --- %s\n", subject, predicate, object);

		triples = triples->next;
	}
}

/**
* \fn ss_triple_t *individual_to_triples (individual_t *ind)
*
* \brief Converts individual to triples.
*
* Converts individual properties with values and uuid with classtype to triples.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return ss_triple_t *. List of triples or NULL if error accures.
*/
ss_triple_t *individual_to_triples (individual_t *ind)
{
	list_t *node = NULL;
	list_head_t *pos = NULL;
	ss_triple_t * triples = NULL;
	prop_val_t *prop = NULL;
	property_t *proptype = NULL;
	individual_t *object = NULL;

	reset_error();

	/*ss_add_triple(&triples, get_ss_info()->space_id, "_hasuuid", ind->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);*/
	ss_add_triple(&triples, ind->uuid, "_classtype", ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	if (ind->properties && !list_empty(&ind->properties->links)) {
		list_for_each(pos, &ind->properties->links) {
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (proptype == NULL) {
				set_error(ERROR_INCORRECT_PROPLIST);
				return NULL;
			}

			if (proptype->type == OBJECTPROPERTY) {
				if (get_rtti_type(prop->prop_value) != RTTI_INDIVIDUAL) {
					set_error(ERROR_INCORRECT_PROPERTY_VALUE);
					return NULL;
				}
				object = (individual_t *) prop->prop_value;
				ss_add_triple(&triples, ind->uuid, proptype->name, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
			else if (proptype->type == DATATYPEPROPERTY)
			{
				ss_add_triple(&triples, ind->uuid, proptype->name, (char *) prop->prop_value, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
		}
	}
	else set_error(ERROR_CANT_FIND_PROPERTY);

	return triples;
}

/**
* \fn ss_triple_t *individual_to_triples_any (individual_t *ind)
*
* \brief Converts individual to triples.
*
* Converts individual properties without values and uuid with classtype to triples.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \return ss_triple_t *. List of triples or NULL if error accures.
*/
ss_triple_t *individual_to_triples_any (individual_t *ind)
{
	list_t *node = NULL;
	list_head_t *pos = NULL;
	ss_triple_t * triples = NULL;
	prop_val_t *prop = NULL;
	property_t *proptype = NULL;

	reset_error();

	/*ss_add_triple(&triples, get_ss_info()->space_id, "_hasuuid", ind->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);*/
	ss_add_triple(&triples, ind->uuid, "_classtype", ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	if (ind->properties && !list_empty(&ind->properties->links)) {
		list_for_each(pos, &ind->properties->links) {
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (proptype == NULL) {
				set_error(ERROR_INCORRECT_PROPLIST);
				return NULL;
			}

			if (proptype->type == OBJECTPROPERTY) {
				ss_add_triple(&triples, ind->uuid, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
			else if (proptype->type == DATATYPEPROPERTY)
			{
				ss_add_triple(&triples, ind->uuid, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
		}
	}
	else set_error(ERROR_CANT_FIND_PROPERTY);

	print_triples(triples);
	return triples;
}

/**
* \fn ss_triple_t *individual_to_triples_by_pattern (individual_t *ind, void *pattern)
*
* \brief Converts individual to triples.
*
* Converts individual properties by pattern with values and uuid with classtype to triples.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \param[in] void *pattern. Entity of pettern (class or individual).
* \return ss_triple_t *. List of triples or NULL if error accures.
*/
ss_triple_t *individual_to_triples_by_pattern (individual_t *ind, void *pattern)
{
	list_t *node = NULL;
	list_head_t *pos = NULL;
	ss_triple_t * triples = NULL;
	prop_val_t *prop = NULL;
	property_t *proptype = NULL;
	list_t *prop_pattern = NULL;
	individual_t *object = NULL;

	reset_error();

	if (get_rtti_type(pattern) == RTTI_INDIVIDUAL) {
		individual_t *obj = (individual_t *) pattern;
		prop_pattern = obj->properties;
	}
	else if (get_rtti_type(pattern) == RTTI_CLASS) {
		class_t *obj = (class_t *) pattern;
		prop_pattern = obj->properties;
	}
	else {
		set_error(ERROR_INCORRECT_ENTITY);
		return NULL;
	}

	/*ss_add_triple(&triples, get_ss_info()->space_id, "_hasuuid", ind->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);*/
	ss_add_triple(&triples, ind->uuid, "_classtype", ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	if (!list_empty(&ind->properties->links)) {
		list_for_each(pos, &ind->properties->links) {
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (((get_rtti_type(pattern) == RTTI_CLASS) && is_in_property_list(proptype, prop_pattern)) || ((get_rtti_type(pattern) == RTTI_INDIVIDUAL) && is_in_property_values_list(prop, prop_pattern)))
			{
				printf("inside\n");
				if (proptype == NULL) {
					set_error(ERROR_INCORRECT_PROPLIST);
					return NULL;
				}

				if (proptype->type == OBJECTPROPERTY) {
					if (get_rtti_type(prop->prop_value) != RTTI_INDIVIDUAL) {
						set_error(ERROR_INCORRECT_PROPERTY_VALUE);
						return NULL;
					}
					object = (individual_t *) prop->prop_value;
					ss_add_triple(&triples, ind->uuid, proptype->name, object->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
				else if (proptype->type == DATATYPEPROPERTY)
				{
					ss_add_triple(&triples, ind->uuid, proptype->name, (char *) prop->prop_value, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
			}
		}
	}
	else set_error(ERROR_CANT_FIND_PROPERTY);

	print_triples(triples);
	return triples;
}


/**
* \fn ss_triple_t *individual_to_triples_by_pattern_any (individual_t *ind, void *pattern)
*
* \brief Converts individual to triples.
*
* Converts individual properties by pattern without values and uuid with classtype to triples.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] individual_t * individual. Pointer to the individual struct.
* \param[in] void *pattern. Entity of pettern (class or individual).
* \return ss_triple_t *. List of triples or NULL if error accures.
*/
ss_triple_t *individual_to_triples_by_pattern_any (individual_t *ind, void *pattern)
{
	list_t *node = NULL;
	list_head_t *pos = NULL;
	ss_triple_t * triples = NULL;
	prop_val_t *prop = NULL;
	property_t *proptype = NULL;
	list_head_t *prop_pattern = NULL;

	reset_error();

	if (get_rtti_type(pattern) == RTTI_INDIVIDUAL) {
		individual_t *obj = (individual_t *) pattern;
		prop_pattern = &obj->properties->links;
	}
	else if (get_rtti_type(pattern) == RTTI_CLASS) {
		class_t *obj = (class_t *) pattern;
		prop_pattern = &obj->properties->links;
	}
	else {
		set_error(ERROR_INCORRECT_ENTITY);
		return NULL;
	}

	/*ss_add_triple(&triples, get_ss_info()->space_id, "_hasuuid", ind->uuid, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);*/
	ss_add_triple(&triples, ind->uuid, "_classtype", ind->classtype, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);

	if (!list_empty(&ind->properties->links)) {
		list_for_each(pos, &ind->properties->links) {
			node = list_entry(pos, list_t, links);
			prop = (prop_val_t *)node->data;
			proptype = prop->property;

			if (((get_rtti_type(pattern) == RTTI_CLASS) && is_in_property_list(proptype, prop_pattern)) || ((get_rtti_type(pattern) == RTTI_INDIVIDUAL) && is_in_property_values_list(prop, prop_pattern)))
			{
				if (proptype == NULL) {
					set_error(ERROR_INCORRECT_PROPLIST);
					return NULL;
				}

				if (proptype->type == OBJECTPROPERTY) {
					ss_add_triple(&triples, ind->uuid, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
				else if (proptype->type == DATATYPEPROPERTY)
				{
					ss_add_triple(&triples, ind->uuid, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
				}
			}
		}
	}
	else set_error(ERROR_CANT_FIND_PROPERTY);

	print_triples(triples);
	return triples;
}

/**
* \fn ss_triple_t *class_to_triples (class_t *class)
*
* \brief Converts individual to triples.
*
* Converts class properties with values with classtype to triples.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] class_t *class. Pointer to the class struct.
* \return ss_triple_t *. List of triples or NULL if error accures.
*/
ss_triple_t *class_to_triples (class_t *oclass)
{
	list_t *node = NULL;
	list_head_t *pos = NULL;
	ss_triple_t * triples = NULL;
	property_t *proptype;

	reset_error();

	if (!list_empty(&oclass->properties->links)) {
		list_for_each(pos, &oclass->properties->links) {
			node = list_entry(pos, list_t, links);
			proptype = (property_t *) node->data;

			if (proptype == NULL) {
				set_error(ERROR_INCORRECT_PROPLIST);
				return NULL;
			}

			if (proptype->type == OBJECTPROPERTY) {
				ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
			else if (proptype->type == DATATYPEPROPERTY)
			{
				ss_add_triple(&triples, SS_RDF_SIB_ANY, proptype->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
			}
		}
	}
	else set_error(ERROR_CANT_FIND_PROPERTY);

	return triples;
}

/**
* \fn ss_triple_t *class_to_triples (class_t *class)
*
* \brief Converts triples to individuals.
*
* Converts properties without values and uuid with classtype to individual struct.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] ss_triple_t *triples. Pointer to the list of triples.
* \return list_t *. List of all individuals.
*/
list_t *triples_to_individuals (ss_triple_t *triples)
{
	list_t *inds = NULL;
	individual_t *ind2;
	char *subject;
	char *predicate;
	char *object;

	reset_error();
	if (triples)
		inds = list_get_new_list();
	else return NULL;

	printf("\n\n\t\ttriples_to_individuals\n");

	while (triples)
	{
		subject = (char *) triples->subject;
		predicate = (char *) triples->predicate;
		object = (char *) triples->object;
		printf("\n\t--- %s - %s - %s\n", subject, predicate, object);

		if (strcmp(predicate,"_hasuuid") == 0)
		{
			const individual_t *ind = get_individual_from_repository_by_uuid(subject);
			if (!ind)
			{
				const class_t* oclass = get_class_from_repository_by_classtype(object);
				if (oclass)
				{
					ind2 = new_individual(oclass);
					ind2->uuid = subject;
				}
				else continue;
			}

			if (ind)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind = NULL;
			}
			else if (ind2)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind2 = NULL;
			}
		}
		else if (strcmp(predicate,"_classtype") == 0)
		{
			const individual_t *ind = get_individual_from_repository_by_uuid(subject);
			if (!ind)
			{
				const class_t* oclass = get_class_from_repository_by_classtype(object);
				if (oclass)
				{
					ind2 = new_individual(oclass);
					ind2->uuid = strdup(subject);
				}
				else continue;
			}
			if (ind)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind = NULL;
			}
			else if (ind2)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind2 = NULL;
			}
		}
		else
		{
			const individual_t *ind = get_individual_from_repository_by_uuid(subject);
			if (!ind)
			{
				const class_t* oclass = get_class_from_repository_by_classtype(object);
				if (oclass)
				{
					ind2 = new_individual(oclass);
					ind2->uuid = subject;
					set_property_by_name((void *) ind2, predicate, object);
				}
				else continue;
			}
			else set_property_by_name((void *) ind, predicate, object);
			if (ind)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind = NULL;
			}
			else if (ind2)
			{
				list_t *node = list_get_new_node((void *) ind);
				list_add(&node->links, &inds->links);
				ind2 = NULL;
			}
		}

		triples = triples -> next;
	}


	return inds;
}

/**
* \fn individual_t *triples_to_individual_first (ss_triple_t *triples)
*
* \brief Converts triples to individual.
*
* Converts properties without values and uuid with classtype to individual struct.
* Function sets global PetrSU KP Library's errno.
*
* \param[in] ss_triple_t *triples. Pointer to the list of triples.
* \return individual_t . First individual only.
*/
individual_t *triples_to_individual_first (ss_triple_t *triples)
{
	list_t *inds = NULL;
	individual_t *ind = NULL;
	char *subject;
	char *predicate;
	char *object;

	reset_error();
	if (triples)
		inds = list_get_new_list();
	else return NULL;

	print_triples(triples);

	while (triples)
	{
		subject = (char *) triples->subject;
		predicate = (char *) triples->predicate;
		object = (char *) triples->object;

		if (strcmp(predicate,"_classtype") == 0)
		{
			if (ind == NULL)
			{
				const individual_t *ind = get_individual_from_repository_by_uuid(subject);
				if (!ind)
				{
					individual_t *ind2 = new_individual(get_class_from_repository_by_classtype(object));
					ind2->uuid = subject;
					ind2 -> classtype = object;
				}
			}
		}
		else
		{
			if (ind == NULL)
			{
				const individual_t *ind = get_individual_from_repository_by_uuid(subject);
				if (!ind)
				{
					individual_t *ind2 = new_individual(get_class_from_repository_by_classtype(predicate));
					ind2->uuid = subject;
					set_property_by_name(ind2, predicate, object);
				}
			}
			else set_property_by_name(ind, predicate, object);
		}

		triples = triples -> next;
	}


	return NULL;
}


// FIXME: convert or not convert object properties.
/**
 * @brief Convert individual to triplets using given list properties.
 *
 * Checks properties, if they can be set for given individuals it convert it to
 * triples, without set object (use SS_RDF_SIB_ANY - any value).
 * It not convert object properties.
 *
 * @param ind individual.
 * @param properties list of properties.
 * It can be NULL - all properties that be set for individual are converted to triplets.
 *
 * @return 0 on success or not otherwise.
 */
ss_triple_t* individual_to_triples_by_properties_any(individual_t *ind, list_t *properties)
{
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
            ("\n%s individual_to_triples_by_properties_any START\n", KPLIB_DEBUG_KPIINTER_PREFIX);

    if (verify_individual_prop_list(ind, properties) != ERROR_NO) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
                ("\n%s individual_to_triples_by_properties_any END: prop_list error\n", KPLIB_DEBUG_KPIINTER_PREFIX);

        return NULL;
    }

    list_t *convert_props =
            (properties == NULL) ? ind->parent_class->properties : properties;

    ss_triple_t *triples = NULL;
    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &convert_props->links) {
         list_t *node = list_entry(list_walker, list_t, links);
         property_t *property = (property_t *) node->data;

        if (property->type == DATATYPEPROPERTY) {
            ss_add_triple(&triples, ind->uuid, property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
        } else if (property->type == OBJECTPROPERTY) {
/*
            ss_add_triple(&triples, ind->uuid, property->name, SS_RDF_SIB_ANY, SS_RDF_TYPE_URI, SS_RDF_TYPE_URI);
*/
        }
    }
    KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_MED) \
            ("\n%s individual_to_triples_by_properties_any END\n", KPLIB_DEBUG_KPIINTER_PREFIX);
    return triples;
}



#endif	/* _KPI_INTERFACE_C_ */
