/*
 * classes.c - functions for work with classes.
 * This file is part of PetrSU KP library.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * PetrSU KP library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PetrSU KP library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include <string.h>
#include "classes.h"
#include "structures.h"
#include "utils/check_func.h"
#include "utils/kp_error.h"
#include "utils/util_func.h"
#include "utils/list.h"


#ifndef _CLASSES_C
#define	_CLASSES_C

/**
 * @brief Repository for classes.
 * All classes must be inserted to repository.
 * It's used for search and other check things.
 */
static list_t *g_classes_repository = NULL;

/**
 * @brief Repository for individuals.
 * All individuals inserted in repository, then it created by @see new_individual.
 * You do not need add created individual again by hand.
 */
static list_t *g_individuals_repository = NULL;

/**
 * @brief Repository for properties. 
 * All properties must be inserted to repository. 
 */
static list_t *g_properties_repository = NULL;



/******************************************************************************/
/*************************** Static functions list ****************************/
static list_t *get_global_repository_by_rtti(int rtti);
static void add_individual_to_repository(individual_t *individual);
static void add_individual_to_class(class_t *oclass, individual_t *individual);



/******************************************************************************/
/**************************** External functions ******************************/
/**
 * @brief Create new individual.
 *
 * Create new individual without UUID.
 *
 * @param classtype type of class.
 *
 * @return individual of given class type on success or NULL otherwise.
 */
individual_t* new_individual(const class_t *oclass)
{
    reset_error();

    int error_code = verify_class(oclass);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return NULL;
    }

    individual_t *individual = (individual_t *) malloc(sizeof(individual_t));

    if (individual == NULL) {
        set_error(ERROR_OUT_OF_MEMORY);
        return NULL;
    }

    individual->rtti = RTTI_INDIVIDUAL;
    individual->parent_class = oclass;
    individual->classtype = get_str_duplicate(oclass->classtype);
    individual->properties = NULL;
    individual->uuid = NULL;

    error_code = verify_individual(individual);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        free_individual(individual);
        return NULL;
    }

    add_individual_to_class((class_t *) oclass, individual);
    add_individual_to_repository(individual);

    return individual;
}


/**
 * @brief Returns a list of superclasses.
 *
 * @param class class.
 *
 * @return list of superclasses on success or NULL otherwise.
 */
list_t* get_superclasses(const class_t *oclass)
{
    reset_error();

    int error_code = verify_class(oclass);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return NULL;
    }

    return oclass->superclasses;
}


/**
 * @brief Checks is class has superclass.
 *
 * @param class class for check.
 * @param superclass alleged superclass of class.
 *
 * @return 1 if the class is a subclass of superclass, -1 on error, .
 */
int is_subclassof(const class_t *subclass, const class_t *superclass)
{
    reset_error();

    int error_code = verify_class(subclass);
    error_code = verify_class(superclass);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return -1;
    }

    if (subclass->superclasses == NULL) {
        return 0;
    }

    list_head_t *pos = NULL;
    list_for_each(pos, &subclass->superclasses->links) {
         list_t *list = list_entry(pos, list_t, links);
         class_t *test_class = (class_t *)list->data;

         if (strcmp(test_class->classtype, superclass->classtype) == 0) {
            return 1;
        }
    }

    return 0;
}


/**
 * @brief Checks type of individual.
 *
 * @param individual object of some class for check.
 * @param class allaged class of individual.
 *
 * @return returns 1 if the individual is object of given class.
 */
int is_classtype_of(const individual_t *individual, const class_t *oclass)
{
    int error_code = verify_class(oclass);
    error_code = verify_individual(individual);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return -1;
    }

    if (strcmp(oclass->classtype, individual->classtype) == 0) {
        return 1;
    }

    return 0;
}


/**
 * @brief Create new property value structure.
 *
 * @param prop property for value.
 * @param data data for value of property.
 *
 * @return new property value structure or NULL on error.
 */
prop_val_t* new_prop_value(property_t *prop, void *data)
{
    if (prop == NULL) {
        return NULL;
    }

    prop_val_t *value = (prop_val_t *) malloc(sizeof(prop_val_t));
    value->property = prop;
    value->prop_value = data;


    return value;
}


/**
 * @brief Count references to given individual.
 *
 * It gets other individuals and search references to individual from their
 * properties.
 *
 * @param individual individual for count.
 *
 * @return count of references.
 */
int count_references_to_individual(individual_t *individual)
{
    if (individual == NULL) {
        return 0;
    }

    list_t *repository = get_global_repository_by_rtti(RTTI_INDIVIDUAL);

    if (repository == NULL) {
        return 0;
    }

    int ind_counter = 0;

    list_head_t *ind_walker = NULL;
    list_for_each (ind_walker, &repository->links) {
         list_t *list = list_entry(ind_walker, list_t, links);
         individual_t *test_ind = (individual_t *) list->data;

         if (test_ind == individual
                 || list_is_null_or_empty(test_ind->properties) == 1) {
             continue;
         }

         list_head_t *prop_walker = NULL;
         list_for_each (prop_walker, &test_ind->properties->links) {
            list_t *list = list_entry(prop_walker, list_t, links);
            prop_val_t *test_propval = (prop_val_t *) list->data;

            if (test_propval == NULL) {
                continue;
            } else if (test_propval->prop_value == individual) {
                ++ind_counter;
            }
         }
    }

    return ind_counter;
}


/**
 * @brief Set UUID.
 *
 * Set new UUID for individuals. You need reinit individual to work with SS.
 *
 * @param ind individual.
 * @param uuid UUID.
 *
 * @return ERROR_NO on success or error code otherwise.
 */
int set_uuid(individual_t *ind, const char *uuid)
{
   int error_code = verify_individual(ind);

   if (error_code != ERROR_NO) {
        return set_error(error_code);
   }

   if (is_str_null_or_empty(uuid) == true) {
       return set_error(ERROR_INCORRECT_UUID);
   }

   if (ind->uuid != NULL) {
       free(ind->uuid);
   }

   ind->uuid = strdup(uuid);

   return reset_error();
}


/**
 * @brief Add entity to repository.
 *
 * It set error and return it on failure.
 * You need to add individual, if you create it manuly.
 * Function @see new_individual() add individual to repository automatically.
 *
 * @param entity individual, property or class.
 *
 * @return ERROR_NO on success or not otherwise.
 */
int add_entity_to_repository(void *entity)
{
    int error_code = verify_entity(entity);

    if (error_code != ERROR_NO) {
        return set_error(error_code);
    }

    int rtti = get_rtti_type(entity);

    list_t *repository = get_global_repository_by_rtti(rtti);

    if (repository == NULL) {
        return set_error(ERROR_INCORRECT_ENTITY);
    }

    if (list_count_nodes_with_data(repository, entity) == 0) {
        list_add_data(entity, repository);
    }

    return reset_error();
}


/**
 * @brief Add class to class repository.
 *
 * It set error and return it on failure.
 *
 * @param class class for insert.
 *
 * @return ERROR_NO on success or not otherwise.
 */
int add_class_to_repository(class_t *oclass)
{
    int error_code = verify_class(oclass);

    if (error_code != ERROR_NO) {
        set_error(error_code);
        return error_code;
    }

    g_classes_repository = list_get_new_list_if_null(g_classes_repository);
    list_add_data(oclass, g_classes_repository);

    reset_error();
    return ERROR_NO;
}


/**
 * @brief Add property to property repository.
 *
 * It set error and return it on failure.
 *
 * @param property for insert.
 *
 * @return ERROR_NO on success or not otherwise.
 */
int add_property_to_repository(property_t *property)
{
    int error_code = verify_property(property);

    if (error_code != ERROR_NO) {
        return set_error(error_code);
    }

    g_properties_repository = list_get_new_list_if_null(g_properties_repository);
    list_add_data(property, g_properties_repository);

    return reset_error();
}


/**
 * @brief Get property from repository by given name.
 *
 * @param name property name.
 *
 * @return property from repository or NULL if it not found.
 */
const property_t* get_property_from_repository_by_name(const char *name)
{
    list_t *repository = get_global_repository_by_rtti(RTTI_PROPERTY);

    if (is_str_null_or_empty(name) == true
            || list_is_null_or_empty(repository) == 1) {
        KPLIB_DEBUG(KPLIB_DEBUG_LEVEL_BMED) \
                ("%s get_property_from_repository_by_name: parametr error or repository = NULL", \
                KPLIB_DEBUG_CLASSSES_PREFIX);

        return NULL;
    }

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &repository->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         property_t *test_prop = (property_t *) list->data;

         if (strcmp(test_prop->name, name) == 0) {
            return test_prop;
        }
    }

    return NULL;
}


/**
 * @brief Get class from repository by given classtype.
 *
 * @param classtype name of classtype.
 *
 * @return class from repository or NULL if it not found.
 */
const class_t* get_class_from_repository_by_classtype(const char *classtype)
{
    list_t *repository = get_global_repository_by_rtti(RTTI_CLASS);

    if (is_str_null_or_empty(classtype) == true
            || list_is_null_or_empty(repository) == 1) {
        return NULL;
    }

    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &repository->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         class_t *test_class = (class_t *) list->data;

         if (strcmp(test_class->classtype, classtype) == 0) {
            return test_class;
        }
    }

    return NULL;
}


/**
 * @brief Get individuals from repository by given classtype.
 *
 * @param classtype name of classtype.
 *
 * @return individuals list or NULL if not found. Do not free list.
 */
const list_t* get_individuals_from_repository_by_classtype(const char *classtype)
{
    const class_t *class = get_class_from_repository_by_classtype(classtype);

    if (class == NULL) {
        return NULL;
    }

    return class->instances;
}


/**
 * @brief Get individuals from repository by given uuid.
 *
 * @param uuid individual's uuid.
 *
 * @return individual or NULL if it not found.
 */
const individual_t* get_individual_from_repository_by_uuid(const char *uuid)
{
    list_t *repository = get_global_repository_by_rtti(RTTI_INDIVIDUAL);

    if (is_str_null_or_empty(uuid) == true
            || list_is_null_or_empty(repository) == 1) {
        return NULL;
    }
    
    list_head_t *list_walker = NULL;
    list_for_each (list_walker, &repository->links) {
         list_t *list = list_entry(list_walker, list_t, links);
         individual_t *test_ind = (individual_t *) list->data;

         if (test_ind->uuid == NULL) {
             continue;
         }

         if (strcmp(test_ind->uuid, uuid) == 0) {
             return test_ind;
         }
    }

    return NULL;
}


/**
 * @brief Remove entity from repository.
 *
 * @param entity individual, class or property.
 */
void remove_entity_from_repository(void *entity)
{
    int rtti = get_rtti_type(entity);

    if (rtti <= RTTI_MIN_VALUE) {
        return;
    }

    list_t *repository = get_global_repository_by_rtti(rtti);

    list_del_and_free_nodes_with_data(repository, entity, NULL);
}


// FIXME: need get all classes, now only parent.
/**
 * @brief Gets all inheritance classes.
 *
 * @param individual individual for get parents.
 *
 * @return returns list with classes on success or NULL otherwise.
 */
list_t* get_individual_inheritance_classes(individual_t *individual)
{
    if (verify_individual(individual) != ERROR_NO) {
        return NULL;
    }

    const class_t *parent = individual->parent_class;

    list_t *classes = list_get_new_list();

    list_add_data((void *)parent, classes);
    return 0;
}


void clean_repositories()
{
    list_free_with_nodes(g_individuals_repository, (void (*)(void *)) free_individual);
    g_individuals_repository = NULL;

    list_free_with_nodes(g_classes_repository, (void (*)(void *)) free_class);
    g_classes_repository = NULL;

    list_free_with_nodes(g_properties_repository, (void (*)(void *)) free_property);
    g_properties_repository = NULL;
}



/******************************************************************************/
/***************************** Static functions *******************************/
/**
 * @brief Gets a global repository by given RTTI.
 *
 * @param rtti Run-time type information.
 *
 * @return global repository.
 */
static list_t *get_global_repository_by_rtti(int rtti)
{
    switch (rtti) {
        case RTTI_INDIVIDUAL: {
            g_individuals_repository = list_get_new_list_if_null(g_individuals_repository);
            return g_individuals_repository;
            break;
        } case RTTI_PROPERTY: {
            g_properties_repository = list_get_new_list_if_null(g_properties_repository);
            return g_properties_repository;
            break;
        } case RTTI_CLASS: {
            g_classes_repository = list_get_new_list_if_null(g_classes_repository);
            return g_classes_repository;
            break;
        }
    }
    return NULL;
}


/**
 * @brief Add individual to repository.
 *
 * It used when new individual create.
 *
 * @param individual individual for insert.
 */
static void add_individual_to_repository(individual_t *individual)
{
    if (individual == NULL) {
        return;
    }

    list_t *repo = get_global_repository_by_rtti(RTTI_INDIVIDUAL);

    list_add_data(individual, repo);
}


/**
 * @brief Add individual to instances list for given class.
 *
 * @param class class.
 * @param individual individual;
 */
static void add_individual_to_class(class_t *oclass,
        individual_t *individual)
{
    if (oclass == NULL || individual == NULL) {
        return;
    }

    oclass->instances = list_get_new_list_if_null(oclass->instances);

    list_add_data((void *) individual, oclass->instances);
}


#endif	/* _CLASSES_C */
