/*
 * structures.h - describe C structures for OWL entities, KP and SS.
 * This file is part of PetrSU KP library.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * PetrSU KP library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PetrSU KP library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PetrSU KP library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include "utils/kp_bool.h"
#include "utils/list.h"

#ifndef _STRUCTURES_H
#define	_STRUCTURES_H



/******************************************************************************/
/******************************** Enums list **********************************/
/**
 * @brief Run-time type information.
 */
enum rtti_types {
    RTTI_MIN_VALUE = 0,     /**< Use for check min value of rtti field. */
    RTTI_INDIVIDUAL = 1,    /**< Individual type. */
    RTTI_CLASS = 2,         /**< Class type. */
    RTTI_PROPERTY = 3,      /**< Property type. */
    RTTI_SBRC_CONTAINER,    /**< Subscribe container type. */
    RTTI_MAX_VALUE          /**< Use for check max value of rtti field. */
};


/**
 * @brief Types of properties.
 */
enum property_types {
    DATATYPEPROPERTY = 1,   /**< Data-property type. */
    OBJECTPROPERTY = 2      /**< Object-property type. */
};



/******************************************************************************/
/****************************** Structures list *******************************/
/**
 * @brief Property structure, represents OWL property.
 */
typedef struct property_s {
	int rtti;               /**< Run-time type information. */
	int type;               /**< Property type: object, data. */
	char *name;             /**< Name of property. */
	char *domain;           /**< Property domain. */
	char *about;            /**< About field. */
	list_t *subpropertyof;  /**< Parent properties list. */
	list_t *oneof;          /**< Propertie's oneof value (OWL oneof). */
	int mincardinality;     /**< Minimal cardinality. */
	int maxcardinality;     /**< Maximum cardinality. */
} property_t;


/**
 * @brief Property value structure.
 *
 * Use for individuals.
 */
typedef struct prop_val_s {
    property_t *property;       /**< Reference to property struct. */
    void *prop_value;           /**< Value for property. */
} prop_val_t;


/**
 * @brief Class structure, represents OWL class.
 */
typedef struct class_s {
	int rtti;               /**< Run-time type information. */
	char *classtype;        /**< Type of class, name. */
	list_t *superclasses;   /**< List of superclasses. */
	list_t *oneof;          /**< Class oneof value (OWL oneof). */
	list_t *properties;     /**< Properties list for class. */
	list_t *instances;      /**< List of individuals. */
} class_t;


/**
 * @brief Individual structure.
 */
typedef struct individual_s {
	int rtti;                       /**< Run-time type information. */
	char *uuid;                     /**< UUID of individual. */
	char *classtype;                /**< Individual's class type. */
	list_t *properties;             /**< Properties values list. */
	const class_t *parent_class;    /**< Parent class. */
} individual_t;



/******************************************************************************/
/****************************** Functions list ********************************/
void free_property(property_t *property);
void free_property_value_with_func(prop_val_t *prop_val, void (*free_data_func)(void*));
void free_class(class_t *oclass);
void free_individual(individual_t *individual);

prop_val_t* new_prop_value(property_t *prop, void *data);

int get_rtti_type(const void* entity);

#endif	/* _STRUCTURES_H */
