/*
 * src/Game/SlysicsCore.cpp
 *
 * Copyright (C) 2009 Wilson Tang.
 *
 * This file is part of Slysics and is free software, distributed,
 * and licensed under the GNU General Public License, version 3.
 *
 * You should have received a copy of the GNU General Public License.
 * If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
*/

#include "SlysicsCore.h"
#include <stdlib.h>
#include <sys/stat.h>

SlysicsCore::SlysicsCore()
{
	stopped = true;
	paused = false;
	hatOff = false;
	headOff = false;
	armOff = false;
	legOff = false;
	fallenOff = false;
	exploded = false;
	lineToolPressed = false;
	mouseDown = false;
	moveSpawn = false;
	setSpawnRotation = false;
	quit = false;
	flagged = false;

	scrollDown = false;
	scrollLeft = false;
	scrollRight = false;
	scrollUp = false;


	spawn.spawnPosition.prevX = 0;
	spawn.spawnPosition.prevY = 0;
	spawn.spawnPosition.x = 0;
	spawn.spawnPosition.y = 0;
	spawn.angle = 0.0f;

	screenOffset.prevX = 40;
	screenOffset.prevY = 40;
	screenOffset.x = 40;
	screenOffset.y = 40;

	toolSelection = 0;
	setSecondary = -1;
	pencilLineType = 0;
	pencilPrevLineType = 0;
	lineToolLineType = 0;
	lineToolPrevLineType = 0;
	mousePreviousX = -1;
	mousePreviousY = -1;
}

SlysicsCore::~SlysicsCore()
{

}

int SlysicsCore::InitializeSlysicsCore()
{
	//Config stuff
	configCore.FillConfigurationList();
	printf( "Loading Configuration...\n" );
	if ( configCore.LoadConfigurationFile() == -1 )
		configCore.WriteConfigurationFile();

	configCore.UpdateVars();

	screenBits = configCore.GetIntegerVariable("screenBits");
	screenHeight = configCore.GetIntegerVariable("screenHeight");
	screenWidth = configCore.GetIntegerVariable("screenWidth");

	renderingCore.InitializeScreen(screenWidth, screenHeight, screenBits, 1);

	sledObject.InitializeSled();
	sledderObject.InitializeSledder();

	uiButtons[0].LoadImage( "data/icon_pencil.png" );
	uiButtons[1].LoadImage( "data/icon_lineTool.png" );
	uiButtons[2].LoadImage( "data/icon_eraser.png" );
	uiButtons[3].LoadImage( "data/icon_move.png" );
	uiButtons[4].LoadImage( "data/icon_play.png" );
	uiButtons[10].LoadImage( "data/icon_pause.png" );
	uiButtons[5].LoadImage( "data/icon_stop.png" );
	uiButtons[6].LoadImage( "data/icon_flag.png" );
	uiButtons[7].LoadImage( "data/icon_saveMain.png" );
	uiButtons[8].LoadImage( "data/icon_delete.png" );
	uiButtons[9].LoadImage( "data/icon_quit.png" );
	uiButtons[11].LoadImage( "data/icon_config.png" );
	uiButtons[12].LoadImage( "data/icon_save.png" );
	uiButtons[13].LoadImage( "data/icon_load.png" );
	spawn.spawnImage.LoadImage( "data/spawn.png" );
	toSpawnIcon.LoadImage( "data/toSpawn.png" );
	toSpawnIcon.SetVisible( false );

	screenSurface = renderingCore.ReturnScreenPointer();

	renderingCore.AddRenderObject( &spawn.spawnImage );
	renderingCore.AddRenderObject( &sledObject.sledImage );
	renderingCore.AddRenderObject( &sledObject.sledCrumpledImage );
	renderingCore.AddRenderObject( &sledderObject.headImage );
	renderingCore.AddRenderObject( &sledderObject.legImage );
	renderingCore.AddRenderObject( &sledderObject.hatImage );
	renderingCore.AddRenderObject( &sledderObject.bodyImage );
	renderingCore.AddRenderObject( &sledderObject.armImage );

	for ( int i = 0; i < 10; i++ )
	{
		uiButtons[i].SetPosition( 32 + 80 + i * 64, screenHeight - 32 );
		renderingCore.AddRenderObject( &uiButtons[i] );
	}
	uiButtons[11].SetPosition( screenWidth - 32, screenHeight - 32 );
	renderingCore.AddRenderObject( &uiButtons[11] );

	uiButtons[12].SetPosition( 32 + 80 + 7 * 64 - 6, screenHeight - 32 - 64 - 10 );
	renderingCore.AddRenderObject( &uiButtons[12] );
	uiButtons[13].SetPosition( 32 + 80 + 7 * 64 + 4, screenHeight - 32 - 64 - 10 - 64 - 4 );
	renderingCore.AddRenderObject( &uiButtons[13] );

	uiButtons[12].SetVisible( false );
	uiButtons[13].SetVisible( false );

	renderingCore.AddRenderObject( &toSpawnIcon );

	uiButtons[10].SetPosition( 32 + 80 + 4 * 64, screenHeight - 32 );
	renderingCore.AddRenderObject( &uiButtons[10] );
	uiButtons[10].SetVisible( false );

	sledObject.sledImage.SetVisible( false );
	sledObject.sledCrumpledImage.SetVisible( false );
	sledderObject.bodyImage.SetVisible( false  );
	sledderObject.headImage.SetVisible( false  );
	sledderObject.armImage.SetVisible( false  );
	sledderObject.legImage.SetVisible( false  );
	sledderObject.hatImage.SetVisible( false );

	gameCanvas.SetPhysicsCore( &physicsCore );
	gameCanvas.SetScreenSize(screenWidth, screenHeight);

	string path = getenv( "HOME" );
	path += "/.slysics";
	mkdir( path.c_str(), 0775 );
	path = getenv( "HOME" );
	path += "/.slysics/maps";
	mkdir( path.c_str(), 0775 );



	configMenu.SetupMenu( &configCore );
	loadMenu.SetupMenu( &gameCanvas, &spawn );

#ifndef MAEMO
	fontSans12.LoadFont( "data/DejaVuSans.ttf", 12 );
#else
	fontSans12.LoadFont( "/usr/share/fonts/nokia/nosnr.ttf", 12 );
#endif
	guiKBoard.InitKeyboard();
}

void SlysicsCore::ProcessUIClick( int x, int y )
{
	for ( int i = 0; i < 10; i++ )
	{
		if ( ( x > 80 + i * 64 ) &&
				( x < 80 + i * 64 + 64 ) &&
				( y > screenHeight - 64 ) )
		{
			if ( i <= 3 && stopped || i <= 3 && paused )
			{
				if ( i == 1 )
					lineToolPressed = true;

				if ( toolSelection == i && ( toolSelection == 0 || toolSelection == 1 || toolSelection == 3 ) )
					setSecondary = 2;

				toolSelection = i;
				break;
			}

			if ( i == 6 )
			{
				printf( "flagith\n" );
				break;
			}

			if ( i == 7 )
			{
				setSecondary = 4;
				break;
			}

			if ( i == 8 && stopped )
			{
				gameCanvas.ClearCanvas();
				gameCanvas.ClearTraceLines();
				break;
			}

			if ( i == 9 && stopped )
			{
				quit = true;
				break;
			}

			if ( i == 4 )
			{
				if ( stopped )
				{
					tOrigX = screenOffset.prevX;
					tOrigY = screenOffset.prevY;
					stopped = false;

					if ( !sledderObject.created )
					{
						float rotatedVelX = ( ( spawn.initialVelocity.vel * cos( double( spawn.angle ) ) ) - ( sin( double( spawn.angle ) ) ) );
						float rotatedVelY = ( ( spawn.initialVelocity.vel * sin( double( spawn.angle ) ) ) + ( cos( double( spawn.angle ) ) ) );

						physicsCore.CreateSled( &sledObject, &sledderObject, ( spawn.spawnPosition.x + 12 ) / 10, ( spawn.spawnPosition.y + 12 ) / 10, spawn.angle );
						sledObject.SetVelocity( b2Vec2( rotatedVelX, rotatedVelY ), 0.0f );
						sledderObject.SetVelocity( b2Vec2( rotatedVelX, rotatedVelY ), 0.0f );

						sledVelocity.prevVel = spawn.initialVelocity.vel;
						headOff = false;
						armOff = false;
						legOff = false;
					}

					uiButtons[10].SetVisible( true );
					uiButtons[4].SetVisible( false );
					spawn.spawnImage.SetVisible( false );

					sledObject.sledImage.SetVisible( true );
					sledObject.sledCrumpledImage.SetVisible( false );
					sledderObject.bodyImage.SetVisible( true );
					sledderObject.headImage.SetVisible( true );
					sledderObject.armImage.SetVisible( true );
					sledderObject.legImage.SetVisible( true );
					sledderObject.hatImage.SetVisible( true );

					sledderBodyVectors.body.prevX = ( spawn.spawnPosition.x + 12 );
					sledderBodyVectors.body.prevY = ( spawn.spawnPosition.y + 12 );

					sledPosition.prevX = ( spawn.spawnPosition.x + 12 );
					sledPosition.prevY = ( spawn.spawnPosition.y + 12 );

					sledderBodyVectors.hat.prevX = ( spawn.spawnPosition.x + 12 );
					sledderBodyVectors.hat.prevY = ( spawn.spawnPosition.y + 12 );

					gameCanvas.ClearTraceLines();

					hatOff = false;
					fallenOff = false;

					for ( int i = 0; i < 4; i++ )
					{
						uiButtons[i].SetVisible( false );
					}
					for ( int i = 7; i < 10; i++ )
					{
						uiButtons[i].SetVisible( false );
					}

					uiButtons[11].SetVisible( false );
				}
				else
				{
					if ( paused )
					{
						paused = false;
						uiButtons[10].SetVisible( true );
						uiButtons[4].SetVisible( false );

						for ( int i = 0; i < 4; i++ )
						{
							uiButtons[i].SetVisible( false );
						}
						for ( int i = 7; i < 10; i++ )
						{
							uiButtons[i].SetVisible( false );
						}
					}
					else
					{
						paused = true;
						uiButtons[10].SetVisible( false );
						uiButtons[4].SetVisible( true );
						screenOffset.prevX = screenOffset.x;
						screenOffset.prevY = screenOffset.y;

						for ( int i = 0; i < 4; i++ )
						{
							uiButtons[i].SetVisible( true );
						}
					}
				}

				break;
			}

			if ( i == 5 && !stopped )
			{
				stopped = true;

				if ( sledderObject.created )
					physicsCore.DestroySled( &sledObject, &sledderObject );

				screenOffset.prevX = tOrigX;
				screenOffset.prevY = tOrigY;
				screenOffset.x = screenOffset.prevX;
				screenOffset.y = screenOffset.prevY;

				uiButtons[10].SetVisible( false );
				uiButtons[4].SetVisible( true );
				spawn.spawnImage.SetVisible( true );
				sledObject.sledImage.SetVisible( false );
				sledObject.sledCrumpledImage.SetVisible( false );
				sledderObject.bodyImage.SetVisible( false  );
				sledderObject.headImage.SetVisible( false  );
				sledderObject.armImage.SetVisible( false  );
				sledderObject.legImage.SetVisible( false  );
				sledderObject.hatImage.SetVisible( false );
				sledObject.crumpled = false;
				headOff = false;
				armOff = false;
				hatOff = false;
				legOff = false;

				paused = false;

				for ( int i = 0; i < 10; i++ )
				{
					uiButtons[i].SetVisible( true );
				}

				uiButtons[11].SetVisible( true );

				break;
			}
		}
	}
}

void SlysicsCore::Events()
{
	//Poll inputs
	while ( SDL_PollEvent( &inputEvents ) )
	{
		if ( !guiKBoard.GetVisibility() )
		{
			if ( !configMenu.isVisible() && !loadMenu.isVisible() )
			{
				if ( inputEvents.type == SDL_KEYDOWN )
				{
					if ( inputEvents.key.keysym.sym == SDLK_LEFT )
					{
						scrollLeft = true;
					}
					if ( inputEvents.key.keysym.sym == SDLK_RIGHT )
					{
						scrollRight = true;
					}
					if ( inputEvents.key.keysym.sym == SDLK_UP )
					{
						scrollUp = true;
					}
					if ( inputEvents.key.keysym.sym == SDLK_DOWN )
					{
						scrollDown = true;
					}
				}
				if ( inputEvents.type == SDL_KEYUP )
				{
					if ( inputEvents.key.keysym.sym == SDLK_LEFT )
					{
						scrollLeft = false;
					}
					if ( inputEvents.key.keysym.sym == SDLK_RIGHT )
					{
						scrollRight = false;
					}
					if ( inputEvents.key.keysym.sym == SDLK_UP )
					{
						scrollUp = false;
					}
					if ( inputEvents.key.keysym.sym == SDLK_DOWN )
					{
						scrollDown = false;
					}
				}
			}

			if ( !scrollDown && !scrollLeft && !scrollRight && !scrollUp )
			{
				if ( inputEvents.type == SDL_MOUSEBUTTONDOWN )
				{
					if ( inputEvents.button.x > screenWidth - 80 && inputEvents.button.y > screenHeight - 64 && stopped )
					{
						if ( !configMenu.isVisible() )
							configMenu.SetVisible( true );
						else
							configMenu.SetVisible( false );
					}

					if ( !configMenu.isVisible() && !loadMenu.isVisible() )
					{
						int spawnRotationIconX = int( ( 0.0f * cos( double( spawn.angle + b2_pi / 2.0f ) ) ) - ( -37.0f * sin( double( spawn.angle + b2_pi / 2.0f ) ) ) ) + spawn.spawnPosition.x + screenOffset.x;
						int spawnRotationIconY = int( ( 0.0f * sin( double( spawn.angle + b2_pi / 2.0f ) ) ) + ( -37.0f * cos( double( spawn.angle + b2_pi / 2.0f ) ) ) ) + spawn.spawnPosition.y + screenOffset.y;
						int mouseX = inputEvents.button.x;
						int mouseY = inputEvents.button.y;
						if ( ( mouseX >= spawnRotationIconX - 14 &&  mouseX <= spawnRotationIconX + 14 && mouseY >= spawnRotationIconY - 14 && mouseY <= spawnRotationIconY + 14 && stopped ) )
						{
							moveSpawn = false;
							setSpawnRotation = true;
						}
						else if ( ( ( inputEvents.button.x > 80 && inputEvents.button.x < 800 - 80 - 6*64 ) || ( inputEvents.button.x > 80 + 6*64 && inputEvents.button.x < 800 - 80 - 3*64 ) || ( inputEvents.button.x > 80 + 4*64 && inputEvents.button.x < 800 - 80 - 5*64 && !stopped ) || ( stopped && inputEvents.button.x > 80 + 7*64 && inputEvents.button.x < 800 - 80 - 2*64 ) ) && inputEvents.button.y > screenHeight - 64 )
						{
							ProcessUIClick( inputEvents.button.x, inputEvents.button.y );
						}
						else if ( ( inputEvents.button.x > 80 + 64*4 && inputEvents.button.x < 800 - 80 - 5*64 ) && stopped && !paused && inputEvents.button.y > screenHeight - 64 )
						{
							setSecondary = 0;
						}
						else if ( inputEvents.button.y < screenHeight - 64 && ( stopped || paused ) )
						{
							if ( stopped && inputEvents.button.x > spawn.spawnPosition.x + screenOffset.x - 40 && inputEvents.button.x < spawn.spawnPosition.x + screenOffset.x + 40 && inputEvents.button.y > spawn.spawnPosition.y + screenOffset.y - 40 && inputEvents.button.y < spawn.spawnPosition.y + screenOffset.y + 40 )
							{
								moveSpawn = true;
								mousePreviousX = inputEvents.button.x;
								mousePreviousY = inputEvents.button.y;
							}
							else if ( stopped || paused )
							{
								mousePreviousX = inputEvents.button.x;
								mousePreviousY = inputEvents.button.y;
								mouseDown = true;
								if ( toolSelection == 1 && configCore.GetVars().snapToPreviousLine )
								{
									mousePreviousX = gameCanvas.GetClosestLineX( mousePreviousX, mousePreviousY, screenOffset.x, screenOffset.y );
									mousePreviousY = gameCanvas.GetClosestLineY( mousePreviousX, mousePreviousY, screenOffset.x, screenOffset.y );
								}
							}
						}
					}
				}

				if ( inputEvents.type == SDL_MOUSEBUTTONUP )
				{
					if ( !configMenu.isVisible() && !loadMenu.isVisible() )
					{
						if ( !flagged && ( setSecondary == 0 || setSecondary == -1 ) && !moveSpawn && !setSpawnRotation && !mouseDown && ( ( inputEvents.button.x > 80 + 4*64 && inputEvents.button.x < screenWidth - 80 - 5*64 && stopped ) || ( inputEvents.button.x > 80 + 5*64 && inputEvents.button.x < screenWidth - 80 - 4*64 ) || ( inputEvents.button.x > 80 + 8*64 && inputEvents.button.x < screenWidth - 80 ) ) && inputEvents.button.y > screenHeight - 64 )
						{
							ProcessUIClick( inputEvents.button.x, inputEvents.button.y );
						}
						else if ( inputEvents.button.y < screenHeight - 64 )
						{
							if ( mouseDown && ( ( !moveSpawn && !setSpawnRotation && toolSelection == 1 && stopped ) || ( toolSelection == 1 && paused ) ) && setSecondary == -1 && !lineToolPressed )
							{
								gameCanvas.CreateLine( mousePreviousX, mousePreviousY, inputEvents.button.x, inputEvents.button.y, screenOffset.x, screenOffset.y, lineToolLineType );
							}

						}
						else
							flagged = false;

						if ( ( toolSelection == 3 && stopped || toolSelection == 3 && paused ) )
						{
							screenOffset.prevX = screenOffset.x;
							screenOffset.prevY = screenOffset.y;

							if ( toolSelection == 3 && setSecondary == 3 && inputEvents.motion.y < screenHeight - 64 && inputEvents.motion.x > 80 + 3*64 && inputEvents.motion.x < 80 + 4*64 )
							{
								toSpawnIcon.SetVisible( false );
								screenOffset.x = ( screenOffset.x - ( screenOffset.x - ( -spawn.spawnPosition.x ) ) ) + screenWidth/2 - 12;
								screenOffset.y = ( screenOffset.y - ( screenOffset.y - ( -spawn.spawnPosition.y ) ) ) + screenHeight/2 - 12;

								screenOffset.prevX = screenOffset.x;
								screenOffset.prevY = screenOffset.y;
							}
						}

						if ( setSecondary == 4 )
						{
							uiButtons[12].SetVisible( false );
							uiButtons[13].SetVisible( false );

							if ( inputEvents.motion.x > screenWidth - 80 - 3*64 && inputEvents.motion.x < screenWidth - 80 - 2*64 )
							{
								if ( gameCanvas.GetNumberOfLines() > 0 && inputEvents.motion.y < screenHeight - 64 - 10 && inputEvents.motion.y > screenHeight - 64 - 10 - 64 )
								{
									guiKBoard.SetVisibility( true );//gameCanvas.SaveMap( spawn.spawnPosition.x, spawn.spawnPosition.y, spawn.angle, spawn.initialVelocity.vel );
									guiKBoard.ClearText();
									guiKBoard.Render( screenSurface );
									SDL_Flip( screenSurface );
								}
								else if ( inputEvents.motion.y < screenHeight - 64 - 10 - 64 && inputEvents.motion.y > screenHeight - 64 - 10 - 64 - 64 )
								{
									loadMenu.FindFiles();
									loadMenu.SetVisible( true );
									for ( int i = 0; i < 8; i++ )
									{
										uiButtons[i].SetVisible( false );
									}
									uiButtons[11].SetVisible( false );

								}
							}
						}
					}

					spawn.spawnPosition.prevX = spawn.spawnPosition.x;
					spawn.spawnPosition.prevY = spawn.spawnPosition.y;

					mousePreviousX = -1;
					mousePreviousY = -1;
					mouseDown = false;
					moveSpawn = false;
					setSpawnRotation = false;
					lineToolPressed = false;
					setSecondary = -1;
					toSpawnIcon.SetVisible( false );
				}

				configMenu.InputEvents( &inputEvents );
				if ( loadMenu.isVisible() )
				{
					if ( inputEvents.type == SDL_MOUSEBUTTONDOWN )
					{
						if ( inputEvents.button.x > screenWidth - 80 - 64 && inputEvents.button.x < screenWidth - 80 && inputEvents.button.y > screenHeight - 64 )
						{
							loadMenu.SetVisible( false );
							flagged = true;
							for ( int i = 0; i < 8; i++ )
							{
								uiButtons[i].SetVisible( true );
							}
							uiButtons[11].SetVisible( true );
						}
						else
							loadMenu.InputEvents( &inputEvents );
					}
				}
			}
			else
			{
				spawn.spawnPosition.prevX = spawn.spawnPosition.x;
				spawn.spawnPosition.prevY = spawn.spawnPosition.y;

				mousePreviousX = -1;
				mousePreviousY = -1;
				mouseDown = false;
				moveSpawn = false;
				setSpawnRotation = false;
				lineToolPressed = false;
				setSecondary = -1;
				toSpawnIcon.SetVisible( false );
			}
		}
		else
		{
			gameCanvas.SaveMap( guiKBoard.ProcessEvents( &inputEvents ), spawn.spawnPosition.x, spawn.spawnPosition.y, spawn.angle, spawn.initialVelocity.vel );
		}
	}

	if ( scrollLeft )
	{
		screenOffset.x += 20;
		screenOffset.prevX += 20;
	}
	if ( scrollRight )
	{
		screenOffset.x -= 20;
		screenOffset.prevX -= 20;
	}
	if ( scrollUp )
	{
		screenOffset.y += 20;
		screenOffset.prevY += 20;
	}
	if ( scrollDown )
	{
		screenOffset.y -= 20;
		screenOffset.prevY -= 20;
	}

	if ( setSpawnRotation )
	{
		if ( inputEvents.motion.x > ( spawn.spawnPosition.x + screenOffset.x ) - 20 && inputEvents.motion.x < ( spawn.spawnPosition.x + screenOffset.x ) + 20 && inputEvents.motion.y > ( spawn.spawnPosition.y + screenOffset.y ) - 20 && inputEvents.motion.y < ( spawn.spawnPosition.y + screenOffset.y ) + 20 )
		{
			spawn.angle = 0.0f;
		}
		else
		{
			if ( inputEvents.motion.x > ( spawn.spawnPosition.x + screenOffset.x ) )
			{
				float hypo = sqrt( pow( float( inputEvents.motion.x - ( spawn.spawnPosition.x + screenOffset.x ) ), 2 ) + pow( float( inputEvents.motion.y - ( spawn.spawnPosition.y + screenOffset.y ) ), 2 ) );
				spawn.angle = asin( float( inputEvents.motion.y - ( spawn.spawnPosition.y + screenOffset.y ) ) / hypo );
			}
			else
			{
				if ( inputEvents.motion.y > ( spawn.spawnPosition.y + screenOffset.y ) )
				{
					spawn.angle = -atan( ( float( ( spawn.spawnPosition.x + screenOffset.x ) ) - float( inputEvents.motion.x ) ) / ( float( ( spawn.spawnPosition.y + screenOffset.y ) ) - float( inputEvents.motion.y ) ) ) + b2_pi / 2.0f;
				}
				else
				{
					spawn.angle = atan( ( float( ( spawn.spawnPosition.y + screenOffset.y ) ) - float( inputEvents.motion.y ) ) / ( float( ( spawn.spawnPosition.x + screenOffset.x ) ) - float( inputEvents.motion.x ) ) ) + b2_pi;
				}
			}
		}
	}

	if ( setSecondary == 0 )
	{
		if ( inputEvents.motion.y < screenHeight - 64 )
		{
			setSecondary = 1;
			spawn.initialVelocity.prevVel = spawn.initialVelocity.vel;
		}
	}

	if ( setSecondary == 1 )
	{
		if ( inputEvents.motion.x < 68 + 80 + 4*64 )
		{
			spawn.initialVelocity.vel = float( screenHeight- 74 - inputEvents.motion.y - 100 );

			if ( spawn.initialVelocity.vel > 100.0f )
				spawn.initialVelocity.vel = 100.0f;

			if ( spawn.initialVelocity.vel < -100.0f )
				spawn.initialVelocity.vel = -100.0f;

			if ( spawn.initialVelocity.vel <= 4.0f && spawn.initialVelocity.vel >= -4.0f )
				spawn.initialVelocity.vel = 0.0f;

			if ( spawn.initialVelocity.vel <= spawn.initialVelocity.prevVel + 4.0f && spawn.initialVelocity.vel >= spawn.initialVelocity.prevVel - 4.0f )
				spawn.initialVelocity.vel = spawn.initialVelocity.prevVel;
		}
	}

	if ( setSecondary == 2 )
	{
		if ( inputEvents.motion.y < screenHeight - 64 )
		{
			setSecondary = 3;
			if ( toolSelection == 0 )
				pencilPrevLineType = pencilLineType;
			else
				lineToolPrevLineType = lineToolLineType;
		}
	}

	if ( setSecondary == 3 )
	{
		int mouseXPosition = inputEvents.motion.x;
		int mouseYPosition = inputEvents.motion.y;
		if ( toolSelection == 0 )
		{
			if ( mouseXPosition > 84 && mouseXPosition < 140 && mouseYPosition > screenHeight - 134 && mouseYPosition < screenHeight - 78 )
				pencilLineType = 0;
			else if ( mouseXPosition > 152 && mouseXPosition < 208 && mouseYPosition > screenHeight - 134 && mouseYPosition < screenHeight - 78 )
				pencilLineType = 1;
			else if ( mouseXPosition > 84 && mouseXPosition < 140 && mouseYPosition > screenHeight - 202 && mouseYPosition < screenHeight - 146 )
				pencilLineType = 2;
		}
		else if ( toolSelection == 1 )
		{
			if ( mouseXPosition > 84 && mouseXPosition < 140 && mouseYPosition > screenHeight - 134 && mouseYPosition < screenHeight - 78 )
				lineToolLineType = 1;
			else if ( mouseXPosition > 152 && mouseXPosition < 208 && mouseYPosition > screenHeight - 134 && mouseYPosition < screenHeight - 78 )
				lineToolLineType = 0;
			else if ( mouseXPosition > 152 && mouseXPosition < 208 && mouseYPosition > screenHeight - 202 && mouseYPosition < screenHeight - 146 )
				lineToolLineType = 2;
		}
		else if ( toolSelection == 3 )
		{

		}
	}

	if ( setSecondary == 4 )
	{
		uiButtons[12].SetVisible( true );
		uiButtons[13].SetVisible( true );
	}

	if ( moveSpawn )
	{
		spawn.spawnPosition.x = spawn.spawnPosition.prevX + inputEvents.motion.x - mousePreviousX;
		spawn.spawnPosition.y = spawn.spawnPosition.prevY + inputEvents.motion.y - mousePreviousY;
	}

	if ( mouseDown )
	{
		if ( toolSelection == 0 )
		{
			if ( gameCanvas.CreateLine( mousePreviousX, mousePreviousY, inputEvents.motion.x, inputEvents.motion.y, screenOffset.x, screenOffset.y, pencilLineType ) )
			{
				mousePreviousX = inputEvents.motion.x;
				mousePreviousY = inputEvents.motion.y;
			}
		}
		else if ( toolSelection == 1 )
		{

		}
		else if ( toolSelection == 2 )
		{
			gameCanvas.DestroyLine( inputEvents.motion.x, inputEvents.motion.y, screenOffset.x, screenOffset.y );
		}
		else if ( toolSelection == 3 )
		{
			screenOffset.x = screenOffset.prevX + inputEvents.motion.x - mousePreviousX;
			screenOffset.y = screenOffset.prevY + inputEvents.motion.y - mousePreviousY;
		}
	}
}

void SlysicsCore::SlysicsMainLoop()
{
	while ( !quit )
	{
		Events();

		if ( !stopped && sledObject.created && !paused && sledObject.sledBody)
		{
			screenOffset.x = ( screenOffset.x - ( screenOffset.x - ( int( -sledObject.sledBody->GetPosition().x * 10.0f ) ) ) ) + screenWidth/2;
			screenOffset.y = ( screenOffset.y - ( screenOffset.y - ( int( -sledObject.sledBody->GetPosition().y * 10.0f ) ) ) ) + screenHeight/2;
		}

		Trace();

		if ( !paused )
			physicsCore.StepTime();

		timerCore.UpdateTimers();

		ProcessTimers();
		ProcessLines();

		SledderCrashProcess();
		Render();
	}
}

void SlysicsCore::Trace()
{
	if ( !stopped && !paused )
	{
		if ( configCore.GetVars().traceLine_body && sledderObject.sledderBodyBody )
		{
			sledderBodyVectors.body.x = ( int( sledderObject.sledderBodyBody->GetPosition().x * 10.0f ) );
			sledderBodyVectors.body.y = ( int( sledderObject.sledderBodyBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledderBodyVectors.body.x - sledderBodyVectors.body.prevX ), 2 ) + pow( float( sledderBodyVectors.body.y - sledderBodyVectors.body.prevY ), 2 ) );
			if ( length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledderBodyVectors.body.prevX, sledderBodyVectors.body.prevY, sledderBodyVectors.body.x, sledderBodyVectors.body.y, 0, 200, 255, 255 );
				sledderBodyVectors.body.prevX = sledderBodyVectors.body.x;
				sledderBodyVectors.body.prevY = sledderBodyVectors.body.y;
			}
		}

		if ( configCore.GetVars().traceLine_sled && sledObject.sledBody )
		{
			sledPosition.x = ( int( sledObject.sledBody->GetPosition().x * 10.0f ) );
			sledPosition.y = ( int( sledObject.sledBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledPosition.x - sledPosition.prevX ), 2 ) + pow( float( sledPosition.y - sledPosition.prevY ), 2 ) );
			if ( fallenOff && length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledPosition.prevX, sledPosition.prevY, sledPosition.x, sledPosition.y, 0, 0, 0, 255 );
				sledPosition.prevX = sledPosition.x;
				sledPosition.prevY = sledPosition.y;
			}
		}

		if ( configCore.GetVars().traceLine_hat && sledderObject.sledderHatBody )
		{
			sledderBodyVectors.hat.x = ( int( sledderObject.sledderHatBody->GetPosition().x * 10.0f ) );
			sledderBodyVectors.hat.y = ( int( sledderObject.sledderHatBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledderBodyVectors.hat.x - sledderBodyVectors.hat.prevX ), 2 ) + pow( float( sledderBodyVectors.hat.y - sledderBodyVectors.hat.prevY ), 2 ) );
			if ( hatOff && length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledderBodyVectors.hat.prevX, sledderBodyVectors.hat.prevY, sledderBodyVectors.hat.x, sledderBodyVectors.hat.y, 255, 100, 0, 255 );
				sledderBodyVectors.hat.prevX = sledderBodyVectors.hat.x;
				sledderBodyVectors.hat.prevY = sledderBodyVectors.hat.y;
			}
		}

		if ( headOff && configCore.GetVars().traceLine_head && sledderObject.sledderHeadBody )
		{
			sledderBodyVectors.head.x = ( int( sledderObject.sledderHeadBody->GetPosition().x * 10.0f ) );
			sledderBodyVectors.head.y = ( int( sledderObject.sledderHeadBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledderBodyVectors.head.x - sledderBodyVectors.head.prevX ), 2 ) + pow( float( sledderBodyVectors.head.y - sledderBodyVectors.head.prevY ), 2 ) );
			if ( length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledderBodyVectors.head.prevX, sledderBodyVectors.head.prevY, sledderBodyVectors.head.x, sledderBodyVectors.head.y, 36, 255, 0, 255 );
				sledderBodyVectors.head.prevX = sledderBodyVectors.head.x;
				sledderBodyVectors.head.prevY = sledderBodyVectors.head.y;
			}
		}

		if ( armOff && configCore.GetVars().traceLine_arm && sledderObject.sledderArmBody )
		{
			sledderBodyVectors.arm.x = ( int( sledderObject.sledderArmBody->GetPosition().x * 10.0f ) );
			sledderBodyVectors.arm.y = ( int( sledderObject.sledderArmBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledderBodyVectors.arm.x - sledderBodyVectors.arm.prevX ), 2 ) + pow( float( sledderBodyVectors.arm.y - sledderBodyVectors.arm.prevY ), 2 ) );
			if ( length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledderBodyVectors.arm.prevX, sledderBodyVectors.arm.prevY, sledderBodyVectors.arm.x, sledderBodyVectors.arm.y, 255, 0, 234, 255 );
				sledderBodyVectors.arm.prevX = sledderBodyVectors.arm.x;
				sledderBodyVectors.arm.prevY = sledderBodyVectors.arm.y;
			}
		}

		if ( legOff && configCore.GetVars().traceLine_leg && sledderObject.sledderLegBody )
		{
			sledderBodyVectors.leg.x = ( int( sledderObject.sledderLegBody->GetPosition().x * 10.0f ) );
			sledderBodyVectors.leg.y = ( int( sledderObject.sledderLegBody->GetPosition().y * 10.0f ) );
			float length = sqrt( pow( float( sledderBodyVectors.leg.x - sledderBodyVectors.leg.prevX ), 2 ) + pow( float( sledderBodyVectors.leg.y - sledderBodyVectors.leg.prevY ), 2 ) );
			if ( length >= MIN_TRACE_LINE_LENGTH )
			{
				gameCanvas.CreateTraceLine( sledderBodyVectors.leg.prevX, sledderBodyVectors.leg.prevY, sledderBodyVectors.leg.x, sledderBodyVectors.leg.y, 255, 204, 0, 255 );
				sledderBodyVectors.leg.prevX = sledderBodyVectors.leg.x;
				sledderBodyVectors.leg.prevY = sledderBodyVectors.leg.y;
			}
		}
	}
}

void SlysicsCore::ProcessTimers()
{

}

void SlysicsCore::SledderCrashProcess()
{

	if ( sledObject.sledBody )
		sledVelocity.vel = ( sledObject.sledBody->GetLinearVelocity().x + sledObject.sledBody->GetLinearVelocity().y );
	if ( sledderObject.armToSledJoint != NULL )
	{
		float bodyForce = ( sledderObject.bodyToSledJoint->GetReactionForce( 10.0f ).x + sledderObject.bodyToSledJoint->GetReactionForce( 10.0f ).y ) / 2.0f;
		float armForce = ( sledderObject.armToSledJoint->GetReactionForce( 10.0f ).x + sledderObject.armToSledJoint->GetReactionForce( 10.0f ).y ) / 2.0f;
		float legForce = ( sledderObject.legToSledJoint->GetReactionForce( 10.0f ).x + sledderObject.legToSledJoint->GetReactionForce( 10.0f ).y ) / 2.0f;
		if ( armForce >= FALL_OFF_FORCE || armForce <= -FALL_OFF_FORCE || legForce >= FALL_OFF_FORCE || legForce <= -FALL_OFF_FORCE || bodyForce >= 120.0f || bodyForce <= -120.0f )
		{
			physicsCore.DeleteJoint( sledderObject.armToSledJoint );
			physicsCore.DeleteJoint( sledderObject.legToSledJoint );
			physicsCore.DeleteJoint( sledderObject.bodyToSledJoint );

			sledderObject.bodyToSledJoint = NULL;
			sledderObject.armToSledJoint = NULL;
			sledderObject.legToSledJoint = NULL;

			fallenOff = true;
			sledPosition.prevX = sledObject.sledBody->GetPosition().x * 10.0f;
			sledPosition.prevY = sledObject.sledBody->GetPosition().y * 10.0f;
		}
	}

	if ( sledObject.sledBody != NULL && ( sledVelocity.vel - sledVelocity.prevVel > MAX_SLEDCRUSH_VELOCITY_CHANGE || sledVelocity.vel - sledVelocity.prevVel < -MAX_SLEDCRUSH_VELOCITY_CHANGE ) )
		sledObject.SetCrumpled( true );

	if ( sledderObject.hatToHeadJoint != NULL )
	{
		float force = ( sledderObject.hatToHeadJoint->GetReactionForce( 10.0f ).x + sledderObject.hatToHeadJoint->GetReactionForce( 10.0f ).y ) / 2.0f;
		if ( force >= HAT_FALL_OFF_FORCE || force <= -HAT_FALL_OFF_FORCE )
		{
			physicsCore.DeleteJoint( sledderObject.hatToHeadJoint );
			sledderObject.hatToHeadJoint = NULL;
			hatOff = true;
			sledderBodyVectors.hat.prevX = sledderObject.sledderHatBody->GetPosition().x * 10.0f;
			sledderBodyVectors.hat.prevY = sledderObject.sledderHatBody->GetPosition().y * 10.0f;
		}
	}

	if ( fallenOff && !headOff && sledderObject.headToBodyJoint != NULL )
	{
		float force = ( sledderObject.headToBodyJoint->GetReactionForce( 10.0f ).x + sledderObject.headToBodyJoint->GetReactionForce( 10.0f ).y ) / 2.0f;

		if ( force >= 90.0f || force <= -90.0f )
		{
			physicsCore.DeleteJoint( sledderObject.headToBodyJoint );

			sledderObject.headToBodyJoint = NULL;
			headOff = true;

			sledderBodyVectors.head.prevX = sledderObject.sledderHeadBody->GetPosition().x * 10.0f;
			sledderBodyVectors.head.prevY = sledderObject.sledderHeadBody->GetPosition().y * 10.0f;
		}
	}

	if ( fallenOff && !armOff && sledderObject.armToBodyJoint != NULL )
	{
		float force = ( sledderObject.armToBodyJoint->GetReactionForce( 10.0f ).x + sledderObject.armToBodyJoint->GetReactionForce( 10.0f ).y ) / 2.0f;

		if ( force >= 90.0f || force <= -90.0f )
		{
			physicsCore.DeleteJoint( sledderObject.armToBodyJoint );

			sledderObject.armToBodyJoint = NULL;
			armOff = true;

			sledderBodyVectors.arm.prevX = sledderObject.sledderArmBody->GetPosition().x * 10.0f;
			sledderBodyVectors.arm.prevY = sledderObject.sledderArmBody->GetPosition().y * 10.0f;
		}
	}

	if ( fallenOff && !legOff && sledderObject.legToBodyJoint != NULL )
	{
		float force = ( sledderObject.legToBodyJoint->GetReactionForce( 10.0f ).x + sledderObject.legToBodyJoint->GetReactionForce( 10.0f ).y ) / 2.0f;

		if ( force >= 90.0f || force <= -90.0f )
		{
			physicsCore.DeleteJoint( sledderObject.legToBodyJoint );

			sledderObject.legToBodyJoint = NULL;
			legOff = true;

			sledderBodyVectors.leg.prevX = sledderObject.sledderLegBody->GetPosition().x * 10.0f;
			sledderBodyVectors.leg.prevY = sledderObject.sledderLegBody->GetPosition().y * 10.0f;
		}
	}


	if ( sledderObject.sledderBodyBody )
	{
		float centerX = sledderObject.sledderBodyBody->GetPosition().x * 10.0f;
		float centerY = sledderObject.sledderBodyBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.armToBodyJoint );
			sledderObject.armToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.headToBodyJoint );
			sledderObject.headToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.legToBodyJoint );
			sledderObject.legToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.bodyToSledJoint );
			sledderObject.bodyToSledJoint = NULL;
			physicsCore.DestroyBody( sledderObject.sledderBodyBody );
			sledderObject.sledderBodyBody = NULL;

			sledderObject.bodyImage.SetVisible(false);
		}
	}

	if ( sledObject.sledBody )
	{
		float centerX = sledObject.sledBody->GetPosition().x * 10.0f;
		float centerY = sledObject.sledBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.armToSledJoint );
			physicsCore.DeleteJoint( sledderObject.legToSledJoint );
			physicsCore.DeleteJoint( sledderObject.bodyToSledJoint );

			sledderObject.bodyToSledJoint = NULL;
			sledderObject.armToSledJoint = NULL;
			sledderObject.legToSledJoint = NULL;
			physicsCore.DestroyBody( sledObject.sledBody );
			sledObject.sledBody = NULL;
			sledObject.sledImage.SetVisible(false);
			sledObject.sledCrumpledImage.SetVisible(false);
		}
	}

	if ( sledderObject.sledderHatBody )
	{
		float centerX = sledderObject.sledderHatBody->GetPosition().x * 10.0f;
		float centerY = sledderObject.sledderHatBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.hatToHeadJoint );
			sledderObject.hatToHeadJoint = NULL;
			physicsCore.DestroyBody( sledderObject.sledderHatBody );
			sledderObject.sledderHatBody  = NULL;
			sledderObject.hatImage.SetVisible(false);
		}
	}

	if ( sledderObject.sledderHeadBody )
	{
		float centerX = sledderObject.sledderHeadBody->GetPosition().x * 10.0f;
		float centerY = sledderObject.sledderHeadBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.headToBodyJoint );
			sledderObject.headToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.hatToHeadJoint );
			sledderObject.hatToHeadJoint = NULL;
			physicsCore.DestroyBody( sledderObject.sledderHeadBody );
			sledderObject.sledderHeadBody = NULL;
			sledderObject.headImage.SetVisible(false);
		}
	}

	if ( sledderObject.sledderArmBody )
	{
		float centerX = sledderObject.sledderArmBody->GetPosition().x * 10.0f;
		float centerY = sledderObject.sledderArmBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.armToBodyJoint );
			sledderObject.armToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.armToSledJoint );
			sledderObject.armToSledJoint = NULL;
			physicsCore.DestroyBody( sledderObject.sledderArmBody );
			sledderObject.sledderArmBody = NULL;
			sledderObject.armImage.SetVisible(false);
		}
	}

	if ( sledderObject.sledderLegBody )
	{
		float centerX = sledderObject.sledderLegBody->GetPosition().x * 10.0f;
		float centerY = sledderObject.sledderLegBody->GetPosition().y * 10.0f;
		if(centerX > 49800.0f || centerX < -49800.0f || centerY > 49800.0f || centerY < -49800.0f)
		{
			physicsCore.DeleteJoint( sledderObject.legToBodyJoint );
			sledderObject.legToBodyJoint = NULL;
			physicsCore.DeleteJoint( sledderObject.legToSledJoint );
			sledderObject.legToSledJoint = NULL;
			physicsCore.DestroyBody( sledderObject.sledderLegBody );
			sledderObject.sledderLegBody = NULL;
			sledderObject.legImage.SetVisible(false);
		}
	}
}

void SlysicsCore::ProcessLines()
{
	if ( !stopped && !paused )
	{
		if ( physicsCore.GetIfAccelerate( 0 ) )
		{
			float rotatedLinearVel = ( ( sledObject.sledBody->GetLinearVelocity().x * cos( double( sledObject.sledBody->GetAngle() ) ) ) - ( sin( double( sledObject.sledBody->GetAngle() ) ) ) );
			if ( rotatedLinearVel < RED_LINE_MAX_AVERAGE_LINEAR_VELOCITY && rotatedLinearVel > -RED_LINE_MAX_AVERAGE_LINEAR_VELOCITY )
			{
				if ( rotatedLinearVel > 0.0f )
				{
					float rotatedVelX = ( ( RED_LINE_ACCELERATION_IMPULSE * cos( double( sledObject.sledBody->GetAngle() ) ) ) - ( sin( double( sledObject.sledBody->GetAngle() ) ) ) );
					float rotatedVelY = ( ( RED_LINE_ACCELERATION_IMPULSE * sin( double( sledObject.sledBody->GetAngle() ) ) ) + ( cos( double( sledObject.sledBody->GetAngle() ) ) ) );
					sledObject.ApplyImpulse( b2Vec2( rotatedVelX, rotatedVelY ) );
				}
				else if ( rotatedLinearVel < 0.0f )
				{
					float rotatedVelX = ( ( -RED_LINE_ACCELERATION_IMPULSE * cos( double( sledObject.sledBody->GetAngle() ) ) ) - ( sin( double( sledObject.sledBody->GetAngle() ) ) ) );
					float rotatedVelY = ( ( -RED_LINE_ACCELERATION_IMPULSE * sin( double( sledObject.sledBody->GetAngle() ) ) ) + ( cos( double( sledObject.sledBody->GetAngle() ) ) ) );
					sledObject.ApplyImpulse( b2Vec2( rotatedVelX, rotatedVelY ) );
				}
			}

		}
		else if ( physicsCore.GetIfDecelerate( 0 ) )
		{
			float rotatedLinearVel = ( ( sledObject.sledBody->GetLinearVelocity().x * cos( double( sledObject.sledBody->GetAngle() ) ) ) - ( sin( double( sledObject.sledBody->GetAngle() ) ) ) );
			if ( rotatedLinearVel > TAN_LINE_MIN_AVERAGE_LINEAR_VELOCITY || rotatedLinearVel < -TAN_LINE_MIN_AVERAGE_LINEAR_VELOCITY )
			{
				rotatedLinearVel *= TAN_LINE_DECELERATION_FACTOR;

				float rotatedVelX = ( ( rotatedLinearVel * cos( double( sledObject.sledBody->GetAngle() ) ) ) - ( sin( double( sledObject.sledBody->GetAngle() ) ) ) );
				float rotatedVelY = ( ( rotatedLinearVel * sin( double( sledObject.sledBody->GetAngle() ) ) ) + ( cos( double( sledObject.sledBody->GetAngle() ) ) ) );

				sledObject.SetVelocity( b2Vec2( rotatedVelX, rotatedVelY ), 0.0f );
			}
		}
	}
}
