#!/usr/bin/env python2.5
# -*- coding: utf-8 -*-
#
# 
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#
# ============================================================================
# Name        : sleeppy.py
# Author      : Vadim Karpusenko < vadikus gmail com >
# Version     : 0.2
# Description : Python Hildon SleepPy Patterns
# ============================================================================
#
# ToDo:
# - show current statustics, plot, etc.
# - time interval sec, min...
# - Options change.
# - reverse linear threshold at the alarm window
# - start counting duration only after deep phase sleep (optional)
# - bugfix: gst custom mp3 alarm play
# - make smart snooze
#
# ToDo 15 apr:
# - Translate in the propper way, use Russian as default 
# - Statistics window:
#   - plot
#   - deep phase sleep analysis
#   - how long was felling asleep
#   - total time
# - main window
#   - time intervals before start
#   - current statistics while running
#   - plotting accelerometr
# - options window
#
# - combine video presentation
# - Write about SleepPy on habrahabr.ru
# - wtite on fruict.ru
# - register program on garage
# - register in contest
# - send email with links

from __future__ import with_statement
import gtk 
import hildon
import ConfigParser
import os, os.path
import datetime
import time
import alarm
import gobject
import gettext
import locale
import math
import sys

defdir = '/home/user/MyDocs/sleeppy'
config = ConfigParser.SafeConfigParser()
if os.path.exists(defdir):
    if os.path.exists(defdir+'/sleeppy.cfg'):
        config.read(defdir+'/sleeppy.cfg')
else:
    config.add_section('user')
    #config.set('user', 'mp3', '/home/user/MyDocs/al.mp3')
    config.set('user', 'fileformat', defdir+'/sleeppy-$04d.log')
    config.set('user', 'index', '2')
    config.set('user', 'alarm', '8:00')
    config.set('user', 'window', '15')
    config.set('user', 'duration',  '7.5')
    config.set('user', 'used_durations',  '7.5 7.0')
    config.set('user', 'duration_or_alarm', 'duration')
    config.set('user', 'language', 'rus')
    config.set('user', 'alarm_threshold', '100')
    config.set('user', 'alarm_play_time_sec', '60')
    #config.set('user', 'snooze_time_min', '10')
    config.set('user', 'accelerometr_threshold', '100')
    config.set('user', 'deep_sleep', '10.0')
    
    config.add_section('tech')
    config.set('tech', 'average', '100')
    config.set('tech', 'accelerometr', "/sys/class/i2c-adapter/i2c-3/3-001d/coord")
    config.set('tech', 'pause', '0.01')      

    os.mkdir(defdir)
    
border = 2

APP_NAME = "SleepPy"
gettext.install(APP_NAME)

local_path = os.path.realpath(os.path.dirname(sys.argv[0]))
langs = []
lc, encoding = locale.getdefaultlocale()
if (lc):
    langs = [lc]
language = os.environ.get('LANGUAGE', None)
if (language):
    langs += language.split(":")
langs += ["en_US", "ru_RU"]
        
if config.get('user', 'language') == 'rus':
    langs = ["ru_RU"]

gettext.bindtextdomain(APP_NAME, local_path)
gettext.textdomain(APP_NAME)
lang = gettext.translation(APP_NAME, local_path
    , languages=langs, fallback = True)
_ = lang.gettext
    
class SleepPyPatterns(hildon.Program):
    def __init__(self):
        hildon.Program.__init__(self)

        self.window = hildon.StackableWindow()
        self.window.connect("delete_event", self.quit)  
        self.add_window(self.window)
                    
        gtk.set_application_name(_('SleepPy Patterns'))
        
        vbox = gtk.VBox(False, border)
        vbox.set_border_width(border)
        self.window.add(vbox)
        vbox.show()
        
        box1 = gtk.HBox(True, border)
        box1.set_border_width(border)
        vbox.pack_start(box1, True, False, border)
        box1.show()
        
        separator = gtk.HSeparator()
        vbox.pack_start(separator, False, False, border)
        separator.show()
        
        box2 = gtk.HBox(True, border)
        box2.set_border_width(border)
        vbox.pack_start(box2, True, False, border)
        box2.show()
        
        self.button_alarm = hildon.Button(gtk.HILDON_SIZE_AUTO, 
                               hildon.BUTTON_ARRANGEMENT_VERTICAL, 
                               _('Alarm for easier wake up'),
                               _('Click here to start gethering statistics') 
                               )
        #button_alarm.set_style(hildon.BUTTON_STYLE_PICKER)
        self.button_alarm.connect('pressed', self.main)
        
        box1.pack_start(self.button_alarm, False, False, 0)
        self.button_alarm.show()

        self.button_duration = hildon.Button(gtk.HILDON_SIZE_THUMB_HEIGHT, 
                               hildon.BUTTON_ARRANGEMENT_VERTICAL, 
                               config.get('user', 'duration')+' '+_('hours'), 
                               _('Set sleep duration'))
        self.button_duration.connect('clicked', self.duration_dialog)
        
        self.button_time = hildon.Button(gtk.HILDON_SIZE_THUMB_HEIGHT, 
                               hildon.BUTTON_ARRANGEMENT_VERTICAL, 
                               config.get('user', 'alarm'), 
                               _('Set alarm time'))
        self.button_time.connect('clicked', self.time_dialog)

        self.button_statistics = hildon.Button(gtk.HILDON_SIZE_THUMB_HEIGHT, 
                               hildon.BUTTON_ARRANGEMENT_VERTICAL, 
                               _('Statistics'),
                               _('Sleep quality data'))
        self.button_statistics.connect("clicked", self.test_win)#self.show_stat_window)
        
        image_statistics = gtk.image_new_from_stock(gtk.STOCK_EXECUTE, gtk.ICON_SIZE_BUTTON)
        self.button_statistics.set_image(image_statistics)
        self.button_statistics.set_image_position(gtk.POS_RIGHT)
    
        
        box2.pack_start(self.button_duration, False, True, 0)
        box2.pack_start(self.button_time, False, True, 0)
        box2.pack_start(self.button_statistics, False, True, 0)
        self.button_duration.show()
        self.button_time.show()
        self.button_statistics.show()
        
        self.alarm_running = False
    
    def test_win(self, wiget):
        # Create a new backing pixmap of the appropriate size
        def configure_event(widget, event):

            gc = widget.window.new_gc()
            gc.set_rgb_fg_color(gtk.gdk.color_parse("#222222"))
            x, y, width, height = widget.get_allocation()
            self.pixmap[widget] = gtk.gdk.Pixmap(widget.window, width, height)
            self.pixmap[widget].draw_rectangle(gc, #widget.get_style().white_gc,
                                  True, 0, 0, width, height)
        
            return True
        
        # Redraw the screen from the backing pixmap
        def expose_event(widget, event):
            x , y, width, height = event.area
            widget.window.draw_drawable(widget.get_style().fg_gc[gtk.STATE_NORMAL],
                                        self.pixmap[widget], x, y, x, y, 
                                        width, height)
            return False
        
        # Draw a rectangle on the screen
        def draw_brush(widget, x, y):
            
            gc = widget.window.new_gc()
            gc.set_rgb_fg_color(gtk.gdk.color_parse("#AAAAAA"))

            rect = (int(x-5), int(y-5), 10, 10)
            self.pixmap[widget].draw_rectangle(gc, #widget.get_style().bg_gc, 
                                  True,
                                  rect[0], rect[1], rect[2], rect[3])
            widget.queue_draw_area(rect[0], rect[1], rect[2], rect[3])

        def draw_stripe(widget, width, x, y):
            
            gc = widget.window.new_gc()
            gc.set_rgb_fg_color(gtk.gdk.Color(y, y, y, 0))

            rect = (int(x-width), 0, int(width)+1, 500)
            self.pixmap[widget].draw_rectangle(gc, #widget.get_style().bg_gc, 
                                  True,
                                  rect[0], rect[1], rect[2], rect[3])
            widget.queue_draw_area(rect[0], rect[1], rect[2], rect[3])

        def draw_line(widget, x, y):
            
            gc = widget.window.new_gc()
            gc.set_rgb_fg_color(gtk.gdk.color_parse("#555555"))

            rect = (int(x-1), int(y), 3, 500)
            self.pixmap[widget].draw_rectangle(gc, #widget.get_style().bg_gc, 
                                  True,
                                  rect[0], rect[1], rect[2], rect[3])
            widget.queue_draw_area(rect[0], rect[1], rect[2], rect[3])

#            gc = widget.window.new_gc()
#            gc.set_rgb_fg_color(gtk.gdk.color_parse("#AAAAAA"))
#
#            rect = (int(x-5), int(y-5), 10, 10)
#            pixmap.draw_rectangle(gc, #widget.get_style().bg_gc, 
#                                  True,
#                                  rect[0], rect[1], rect[2], rect[3])
#            widget.queue_draw_area(rect[0], rect[1], rect[2], rect[3])
        
#        def button_press_event(widget, event):
#            if event.button == 1 and pixmap != None:
#                draw_brush(widget, event.x, event.y)
#            return True
#        
#        def motion_notify_event(widget, event):
#            if event.is_hint:
#                x, y, state = event.window.get_pointer()
#            else:
#                x = event.x
#                y = event.y
#                state = event.state
#        
#            if state & gtk.gdk.BUTTON1_MASK and pixmap != None:
#                draw_brush(widget, x, y)
#        
#            return True            

        window = hildon.StackableWindow()
        pannable_area = hildon.PannableArea()
        self.pixmap = dict()
        window.set_title(_('Statistics'))
        hildon.hildon_gtk_window_set_progress_indicator(window, 1)
        label = gtk.Label(_('Sleep quality data'))
        vbox = gtk.VBox(False, 0)
        vbox.pack_start(label, True, True, 0)
        vbox.pack_start(gtk.HSeparator(), False, False, border)
        pannable_area.add_with_viewport(vbox)
        window.add(pannable_area)
        vbox.show()
        plots, labels, data, times = [], [], [], []
        plot_max = 0
        filename = config.get('user', 'fileformat').replace('$', '%')
        for i in range(config.getint('user', 'index')):
            if os.path.exists(filename%i):
                plot_max += 1
        plot_max = min(7, plot_max)
        for i in range(plot_max):
            data.append([])
            times.append([])
            
            label = gtk.Label()
            label.set_alignment(0, 0)
            vbox.pack_start(label, False, False, 0)
            labels.append(label)
            
            drawing_area = gtk.DrawingArea()
            drawing_area.set_size_request(800, 30)
            vbox.pack_start(drawing_area, False, True, 0)
            plots.append(drawing_area)
        
            drawing_area.show()

            # Signals used to handle backing pixmap
            drawing_area.connect("expose_event", expose_event)
            drawing_area.connect("configure_event", configure_event)
        
            # Event signals
#            drawing_area.connect("motion_notify_event", motion_notify_event)
#            drawing_area.connect("button_press_event", button_press_event)
        
            drawing_area.set_events(gtk.gdk.EXPOSURE_MASK
                                    | gtk.gdk.LEAVE_NOTIFY_MASK
                                    | gtk.gdk.BUTTON_PRESS_MASK
                                    | gtk.gdk.POINTER_MOTION_MASK
                                    | gtk.gdk.POINTER_MOTION_HINT_MASK)
    
            vbox.pack_start(gtk.HSeparator(), False, False, border)        
        window.show_all()

        num = config.getint('user', 'index')
        for i in range(plot_max):
            while gtk.events_pending():
                gtk.main_iteration(False)
            while not os.path.exists(filename%num):
                num -= 1
            x, y, width, height = plots[i].get_allocation()
            with open(filename%num, 'r') as file:
                for line in file:
                    if line[0] == '#':
                        labels[i].set_text(labels[i].get_text()+' '+line[1:].strip())
                    else:
                        try:
                            t, shake = [float(x) for x in line.split()]
                        except:
                            pass
                        data[i].append(shake)
                        times[i].append(t)
                        
            deep_sleep_time = 0.0
            fell_asleep = 0.0
            ds = config.getfloat('user', 'deep_sleep')/60.0
            if len(times[i]) >2:
                for j in range(len(times[i])-1):
                    dt = times[i][j+1]-times[i][j]                
                    if dt>ds:
                        deep_sleep_time += dt - ds
                        if not fell_asleep:
                            fell_asleep = times[i][j]
            
                scalet, scaled = [], []
                for t in times[i]:
                    scalet.append(t/max(times[i])*(width-1))
                for dat in data[i]:
                    #scaled.append((1-dat/max(data[i]))*height)
                    scaled.append(int(dat/max(data[i])*65535*3/4 + 65535/4))
                    
                #for td in zip(scalet, scaled):
                #    draw_line(plots[i], *td)
                for td in zip(scalet, scaled):
                    draw_stripe(plots[i], width/max(times[i])/3600+1, *td)
                
    
                labels[i].set_text(labels[i].get_text()+
                                   _(' Total: ')+timed(max(times[i]))+
                                   _(' Deep: ')+timed(deep_sleep_time)+
                                   _(' Falling asleep: ')+timed(fell_asleep)
                                   )            
            num -= 1
            
        
        
        hildon.hildon_gtk_window_set_progress_indicator(window, 0)

    def on_history_append(self, toolbar, user_data):
        # Get last added index
        index = toolbar.get_last_index()
        # Get the inner list
        list = toolbar.get("list")    
        # Get the item
        iter = list.get_iter_from_string("%d" % index)    
        item, = list.get(iter, 0)
        return item

    def duration_dialog(self, widget):
        dialog = hildon.PickerDialog(self.window)
        dialog.set_transient_for(self.window)

        dialog.set_title(_('Required sleep duration (hours)'))
        selector = hildon.TouchSelectorEntry(text=True)
        for val in config.get('user', 'used_durations').split():
            selector.append_text(str(val))
        dialog.set_selector(selector)
        selector.set_name(config.get('user', 'duration'))
        
        dialog.show_all()
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            try:
                duration = float(selector.get_current_text().strip())
                config.set('user', 'duration', selector.get_current_text().strip())
                if selector.get_current_text().strip() not in config.get('user', 'used_durations').split():
                    config.set('user', 'used_durations', selector.get_current_text().strip() +' '+
                                                config.get('user', 'used_durations').strip())
                widget.set_title(config.get('user', 'duration')+' '+_('hours'))
                config.set('user', 'duration_or_alarm', 'duration')
                duration = float(selector.get_current_text().strip())
                nownum = time2num(str(datetime.datetime.now())[11:16])
                self.button_time.set_title(timed((nownum + duration)%24))
            except:
                banner = hildon.hildon_banner_show_information(widget, gtk.STOCK_CLOSE, _('Enter a number, please'))
                banner.set_timeout(9000)

    #            image_statistics = gtk.image_new_from_stock(gtk.STOCK_CLOSE, gtk.ICON_SIZE_BUTTON)
    #            self.button_duration.set_image(image_statistics)
    #            #self.button_time.set_image(None)
    #            self.button_duration.set_image_position(gtk.POS_LEFT)
            dialog.hide()
        
    def time_dialog(self, widget):
        dialog = hildon.Dialog()
        dialog.set_title(_('Set alarm time'))
        dialog.set_transient_for(self.window)
        time_button = hildon.TimeButton(gtk.HILDON_SIZE_THUMB_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        h, m = [int(x) for x in config.get('user', 'alarm').split(':')]
        time_button.set_time(h, m)
        dialog.vbox.pack_start(time_button, True, True, border)
        dialog.vbox.pack_start(gtk.HSeparator(), False, False, border)
        dialog.vbox.pack_start(gtk.Label(_('Wake up time interval (minutes)')), False, False, border)
        bar = hildon.Seekbar()
        bar.set_total_time(60)
        bar.set_position(config.getint('user', 'window'))
        dialog.vbox.pack_start(bar, True, True, border)
        interval = gtk.Label()
        self.control_changed(bar, interval)
        dialog.vbox.pack_start(interval, False, False, border)
        bar.connect('value-changed', self.control_changed, interval)        
        dialog.add_buttons("Done", gtk.RESPONSE_OK)
        dialog.show_all()
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            h, m = time_button.get_time()
            str_alarm = '%d:%02d'%(h,m)
            config.set('user', 'alarm', str_alarm)
            config.set('user', 'window', str(bar.get_position()))
            widget.set_title(str_alarm)
            config.set('user', 'duration_or_alarm', 'alarm')
            alarmnum = h+m/60.0
            nownum = time2num(str(datetime.datetime.now())[11:16])
            if nownum>alarmnum:
                alarm_time = 24.0 - nownum + alarmnum
            else:
                alarm_time = alarmnum - nownum
            self.button_duration.set_title('%.2f '%alarm_time+_('hours'))
            dialog.hide()

    def control_changed(self, widget, label): 
        label.set_text(" %d %s" % (int(widget.get_value()), _('minutes')))

    def quit(self, *args):
        self.alarm_running = False
        with open(defdir+'/sleeppy.cfg', 'wb') as configfile:
            config.write(configfile)
        gtk.main_quit()
        
    def run(self):     
        self.window.show_all()
        gtk.main() 

    def add_two_button_alarm(self, widget=None):
        event = alarm.Event()
        event.appid = 'SleepPy'
        event.message = _('Wake Up! Go, get them, Tiger!')
        event.alarm_time = time.time()
        action_stop, action_snooze = event.add_actions(2)
        action_stop.label = config.get(self.lang, 'stop')
        action_stop.flags |= alarm.ACTION_WHEN_RESPONDED | alarm.ACTION_TYPE_NOP
        action_snooze.label = config.get(self.lang, 'snooze')
        action_snooze.flags |= alarm.ACTION_WHEN_RESPONDED | alarm.ACTION_TYPE_SNOOZE
        cookie = alarm.add_event(event)
        return cookie
    
    def get_rotation(self):
        file = open(config.get('tech', 'accelerometr'), 'r' )
        coords = [int(w) for w in file.readline().split()]
        file.close()
        return coords
    
    def main(self, wiget):
        if not self.alarm_running:
            self.alarm_running = True
            average = [0.0, 0.0, 0.0]
            dif = [0.0, 0.0, 0.0]
            begining_time = datetime.datetime.now()
            t_old = begining_time
            filename = config.get('user', 'fileformat').replace('$', 
                                    '%')%config.getint('user', 'index')
            self.button_alarm.set_value(_('Press to stop statistics gethering\nLoggin to the file:\n')+filename)
            config.set('user', 'index', str(config.getint('user', 'index')+1))
            alarm = -config.getfloat('user', 'alarm_play_time_sec')
            diff_old = 0.0
            const_average = config.getfloat('tech', 'average')
            
            alarmnum = time2num(config.get('user', 'alarm'))
            windownum = config.getfloat('user', 'window')/60.0
            nownum = time2num(str(datetime.datetime.now())[11:16])
            
            pause = config.getfloat('tech', 'pause')
            threshold = config.getint('user', 'accelerometr_threshold')
            
            if nownum>alarmnum:
                alarm_time = 24.0 - nownum + alarmnum
            else:
                alarm_time = alarmnum - nownum 
            
            with open(filename, 'w') as file:
                file.write('# '+ str(begining_time)[:19])
                while self.alarm_running:
                    
                    value = self.get_rotation()
                    
                    while gtk.events_pending():
                        gtk.main_iteration(False)
                        
                    time.sleep(pause)
                    t = datetime.datetime.now()
                    dt = (t - begining_time).seconds
                    dth = dt/3600.0
                    
                    # required alarm
                    if dth > alarm_time: 
                        if config.get('user', 'duration_or_alarm') == 'alarm':
                            self.set_alarm_up()
                            # TODO use: alarm.delete_event(num)
                            break
                    
                    for i in range(3):
                        average[i] = (value[i] + (const_average-1)*average[i])/const_average
                        dif[i] = abs(value[i] - average[i])
                    if max(dif) > threshold:
                        if t.second == t_old.second:
                            diff_old = max(diff_old, max(dif))
                        else:
                            file.write('\n%8.4f %7.3f'%(dth, diff_old))
                            t_old = t
                            diff_old = max(dif)
                        if config.get('user', 'duration_or_alarm') == 'duration':
                            if dth > config.getfloat('user', 'duration') and max(dif
                                ) > config.getint('user', 'alarm_threshold'):
                                #button_alarm.set_title(str((dt-alarm, config.getint('user', 'alarm_play_time_sec'))))
                                if dt-alarm > config.getfloat('user', 'alarm_play_time_sec'):
                                    self.set_alarm_up()
                        else:
                            if dth > alarm_time-windownum and max(dif
                                ) > config.getint('user', 'alarm_threshold'):
                                if dt-alarm > config.getfloat('user', 'alarm_play_time_sec'):
                                    self.set_alarm_up()
                                    
        else:
            self.alarm_running = False
            self.button_alarm.set_value(_('Click here to start gethering statistics'))

    def set_alarm_up(self):
        self.add_two_button_alarm()
        self.alarm_running = False
        self.button_alarm.set_value(config.get(self.lang, 'button_alarm'))    

            
def timed(t):
        min = (t - math.floor(t))*60
        return '%d:%02d'%(math.floor(t), min)

def time2num(t):
        h, m = [int(x) for x in t.split(':')]
        return h+m/60.0

if __name__ == "__main__":
    app = SleepPyPatterns() 
    app.run()         
