#!/usr/bin/env python2.5
# -*- coding: utf-8 -*-
###########################################################################
## Name: 	       SleepAnalyser
## Description:   Sleep Analyser records your movement during your sleep. It is able to visualise it on a graph to show
##                      how much you move during your sleep. This can help to indicate how you sleep. It also has an alarm
##                      function. You can set the alarm and a time window (ex. 30 minutes). The Alarm will then go off
##                      sometimes during the time window (as soon as you move more), but latest at the set alarm.
##                      Old records can be visualised and you can load records from your friends. SleepAnalyser also has a
##                      test function. It will record and visualise much faster. Also it will make a beep when ever your
##                      movement goes over the trigger level.
## Copyright:     George Ruinelli, george@ruinelli.ch
## Licence:        GPL
###########################################################################

# TODO:
# update documentation
#make volume louder on wakeup


from PyQt4.QtCore import *
from PyQt4.QtGui import *
import os, sys

import dbus
from dbus.mainloop.qt import DBusQtMainLoop
from time import strftime

import data #own data
import records #record data


## INIT ###################################
#app path has to be set first
data.app_path=os.path.dirname(__file__)
if(data.app_path==""): data.app_path="./"
else: data.app_path=data.app_path+"/"

print strftime("%A %x %X") #for log file

print "SleepAnalyser"

#Load version file
try:
    file = open(data.app_path+"version", 'r')
    data.version = file.readline()
    data.version=data.version[:-1]
    data.build = file.readline()
    print "Version: "+str(data.version)+ "-"+str(data.build)
except:
    print "Version file not found, please check your installation!"

import kht_report  #Used for error reporting


#check parameters
if(len(sys.argv)>1):  #at least one parameter
    print "Parameters:", sys.argv
    if(sys.argv[1]=="debug"):
        print "Enable Debug mode"
        data.debug=True
    elif(sys.argv[1]=="lang"):
        if(len(sys.argv)>2):
            data.language=sys.argv[2]
            print "Set language to:", data.language
        else:
            print "Wrong parameter list"
            print "Please set 2-letter language code"
            exit()

    else: #show command line interface data
        print "sleepanalyser [options]"
        print "OPTIONS:"
        print "  debug: Shows debug information"
        print "  lang COUNTRYCODE: Use specific language file. (Use 2-letter country code like de, en, fr, nl, ..."
        exit()

#load language file
import i18n
#_ = i18n.language.gettext
_ = i18n.language.ugettext

########################################

#own functions
from functions_general import *
from functions_file import *
from functions_record import *
from functions_alarm import *

#own c cfunctions (for speed improvements)
try:
    from lib import  mylib
    MyLib_Imported=True
except:
    MyLib_Imported=False


#GUIs
from Ui_StartWindow import *
from Ui_RecordWindow import *
from Ui_ViewWindow import *
from Ui_ConfigWindow import *
from Ui_HelpWindow import *
from Ui_DateWindow import *
from Ui_SetRecordingWindow import *
from Ui_WakeUpSettingsWindow import *
from Ui_LucidDreamSettingsWindow import *
from Ui_AlarmSettingsWindow import *
from Ui_AskForKeepWindow import *
from Ui_AdditionalInfoWindow import *


###########################################################################
## Main function
##############o#############################################################
def main():
    global app,  StartWindow,  RecordWindow,  ViewWindow ,  ConfigWindow,  DateWindow,  SetRecordingWindow, LucidDreamSettingsWindow, WakeUpSettingsWindow, AlarmSettingsWindow,  AskForKeepWindow,  AdditionalInfoWindow,  HelpWindow
    global config

    #if not on the N900, a acceleration sensor has to be simulated
    if(os.path.exists("/sys/class/i2c-adapter/i2c-3/3-001d/coord")): data.DeviceIsN900=True	#runs on real N900 hardware
    else: data.DeviceIsN900=False	#runs on other hardware/OS


    app = QApplication(sys.argv)

    kht_report.install_excepthook("SleepAnalyser", data.version +"-"+ data.build) #register error reporter

    if(data.debug==True): print "Application path:",  data.app_path

    if(data.DeviceIsN900==False):
        print "Runs not on a N900 => Use randomizer for acceleration sensor data"
        data.configFolder=data.developmentconfigFolder
        data.RecordFolder=data.developmentRecordFolder
    else: data.RecordFolder=data.defaultRecordFolder

    config=LoadConfig()


    #check if module is available and can get loaded
    if(MyLib_Imported==True): #module is loaded
        print "Mylib is compiled:", mylib.IsCompiled() #check if module is compiled
        if(mylib.IsCompiled()==False): #module is not compiled
            txt= "The compiled module mylib.so could not get loaded! Please check if there is a never version "+ \
                     "or inform the maintainer about it! SleepAnalyser will still work, " + \
                     "but some functions will work slow as we have to fall back to the unoptimised python code!"
            print txt
            cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog " + \
                     "string:\""+"SleepAnalyser:\n"+txt+"\" uint32:0 string:\"OK\""
            os.popen(cmd)
    else: #module is available but broken
        txt="SleepAnalyser could not load the module mylib.so\n" + \
                 "in "+data.app_path+"lib. It might be corrupted. Please check if there is a never version "+ \
                 "or inform the maintainer about it! You can get SleepAnalyser working by removing the file "+data.app_path+"lib/mylib.so, " + \
                 "how ever some functions will work slow as we have to fall back to the unoptimised python code!"
        print txt
        cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog " + \
                 "string:\""+"SleepAnalyser:\n"+txt+"\" uint32:0 string:\"OK\""
        os.popen(cmd)
        exit()


    #Load Stylesheet
    try:
        file = open(data.configFolder+"UIStyle.css_user.css", 'r')
        data.UIStyle=""
        for line in file: data.UIStyle=data.UIStyle+line+"\n"
        file.close()
        print "Loaded user defined style sheet"
    except:
        print "User defined style sheet not existing or not readable, load default one"
        print "If you want to use a user defined style sheet, place it under", data.configFolder+"UIStyle_user.css"
        try:
            file = open(data.app_path+"UIStyle", 'r')
            data.UIStyle=""
            for line in file: data.UIStyle=data.UIStyle+line+"\n"
            file.close()
        except:
            pass
#        try:
#            print "Create user defined style sheet in",  data.configFolder
#            file= open(data.configFolder+"UIStyle.css", 'w')
#            file.write(data.UIStyle)
#            file.close()
#        except:
#            print "Could not create user defined style sheet"

    StartWindow = frmStartWindow()
    StartWindow.show()
    StartWindow.UpdateScreenStart()

    # Create subwindows
    RecordWindow = frmRecordWindow(StartWindow)
    ConfigWindow = frmConfigWindow(StartWindow)
    ViewWindow = frmViewWindow(StartWindow)
    HelpWindow = frmHelpWindow(StartWindow)
    SetRecordingWindow = frmSetRecordingWindow(StartWindow)
    AlarmSettingsWindow = frmAlarmSettingsWindow(SetRecordingWindow)
    LucidDreamSettingsWindow = frmLucidDreamSettingsWindow(SetRecordingWindow)
    WakeUpSettingsWindow = frmWakeUpSettingsWindow(SetRecordingWindow)
    DateWindow = frmDateWindow(ViewWindow)
    AdditionalInfoWindow = frmAdditionalInfoWindow(ViewWindow)
    AskForKeepWindow = frmAskForKeepWindow(RecordWindow)


    # Hide SubWindows at first
    RecordWindow.hide()
    ViewWindow.hide()
    ConfigWindow.hide()
    SetRecordingWindow.hide()
    AlarmSettingsWindow.hide()
    LucidDreamSettingsWindow.hide()
    WakeUpSettingsWindow.hide()
    DateWindow.hide()
    AdditionalInfoWindow.hide()
    AskForKeepWindow.hide()

    if(data.user==""):#get user name for in record files
        EditUserName()

    DisplayStatusDaemon()

    delete_all_preexisting_Alarms() #Check if there are old alarms. If SleepAnalyser crashed, they might still be set

    sys.exit(app.exec_())





###########################################################################
##
###########################################################################
def EditUserName():
#      r=QInputDialog.getText(None,  "Add Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
#        if(r[1]==True): #OK pressed
    r = QInputDialog.getText(None, _("SleepAnalyser"), _("Please enter your name.\nIt will be used in the records file, this makes it easier\nwhen you want to share your sleep pattern with others."), 0,  data.user)
#    if(r != ""):
    if(r[1]==True): #OK pressed
        data.user=unicode(r[0])
        if(data.user==""): data.user="-"
        config.set('main', 'user', str(data.user.encode( "utf-8" )))

    try:
        print "User name:",  data.user
    except:
        pass





###########################################################################
##  Gets called every second
###########################################################################
def MeasurementTimerLoop(): #(interval: sampling*sample_smoothing*value_smoothing)
    r=GetSensorData()
    if(r==1): #new data got added to array
        ProcessData(False,  False)
        GenerateSleepPattern() #new: generate blue graph always
        ProcessAlarm()

        WriteRecord()

#        GraphLen=records.Index/data.arrXscale[data.Xscale]
#        data.GraphStart=max(GraphLen-779, 0)

        data.ScrollToEnd=True
        if(data.DisplayIsOn==True): #when display is on, update graph
            GenerateStatisticData()
#            GenerateSleepPattern()
            Generate_and_show_GraphBitmap(RecordWindow, True) #New: show blue graph
#            Generate_and_show_GraphBitmap(RecordWindow, False)

        if(records.Index==records.MaxArrayDataLen-1):
            print "Memory is full!"
            txt=_('Memory is full!')
            data.Status=txt
            QtGui.QMessageBox.warning(RecordWindow, txt, _("You reached the end of the recording capacity!\nThe recording stops now."), QtGui.QMessageBox.Ok)
#            RecordWindow.StopMeasurement()
            StopMeasurement() #stop but keep alarm active!
            SwitchModes(False) #change back to normal mode and general profile if requested

    ProcessAlarm()

    ProcessLucid_Dream()

    if(data.DisplayIsOn==True):#when display is on, update clock animation
        RecordWindow.UpdateAnimation() #show turning clock
        if(data.debug==True): print "-> o Update screen"
        RecordWindow.UpdateScreen()
    else:
        if(data.debug==True): print "-> x Do not update screen, Display is off!"


#    print data.AlarmIsActive,  data.StopAfterAlarm
    if(data.AlarmIsActive==True): #alarm got activated
#        print "Alarm is active"
        if(data.StopAfterAlarm==True): #stop recording

            data.DisplayIsOn=True

#            print "Stop measurement, alarm is active"
            print "Alarm got activated, stopped recording"
            txt=_("Alarm got activated, stopped recording")
            data.Status=txt
            StopMeasurement() #stop but keep alarm active!

            #change directly to view screen
#            DeleteAlarm() #delete Alarm if existing
#            records.AlarmIsSet=False
            data.LastRecord=data.RecordFilename #set to corrent file for viewwindow
            config.set('main', 'LastRecord', str(data.LastRecord))
            print "Show record in ViewWindow:",  data.LastRecord
            RecordWindow.hide()
            StartWindow.ShowViewWindow()

        else: #continue recording
            pass

        SwitchModes(False) #change back to normal mode and general profile if requested




###########################################################################
## Start the measurement
###########################################################################
def InitMeasurement():
    global app

    data.sample_index=0
    data.last_beep=0 #for quick mode

    data.WakeUp_SongIsActive=False
    data.Lucid_Dream_SongIsActive=False

    data.Animation=7
    RecordWindow.UpdateAnimation() #show turning clock

    if(data.QuickMode==False): data.Status=_("Recording...")
    else: data.Status=_("Recording in Test Mode...")

    records.User=data.user
    records.SWversion=data.version
    records.Sampling_interval=data.Sampling_interval
    records.Sample_smoothing=data.Sample_smoothing
    records.Value_smoothing=data.Value_smoothing
    records.Trigger=data.Trigger #save Trigger level on start
    records.statusID="OK"

    if(data.debug==True): PrintRecordData()

    #set scale and sampling interval
    if(data.QuickMode==False):
        data.interval=data.Sampling_interval #normal
        data.Xscale=1 #set to default zoom level
    else:
        data.interval=10 #Fast mode for testing
        data.Xscale=0 #set to max zoom level

#    print "#################### USE MAXIMUM SPEED FOR DEBUGGING!!!!!!!!!!!"
#    data.interval=1 #Debugging


    data.run=True #change to recording
    app.timer_GetData = QtCore.QTimer()
    QtCore.QObject.connect(app.timer_GetData, QtCore.SIGNAL("timeout()"), MeasurementTimerLoop)

    data.StartTime=time.localtime()
    data.StartTimeSec=Tuple2Seconds(data.StartTime)

#    if(data.debug==False): app.timer_GetData.start(data.interval)
#    else: app.timer_GetData.start(1)    #run very fast for testing
    app.timer_GetData.start(data.interval)

    CreateRecordFile(data.interval)
    GetFirstEntry()
    WriteRecord()

    Generate_and_show_GraphBitmap(RecordWindow, True)
#    Generate_and_show_GraphBitmap(RecordWindow, False)
#    RecordWindow.UpdateScreen()

    if(records.AlarmIsSet==True or data.QuickMode==True): #alarm time window is set, show it
        RecordWindow.ui.bRaiseTrigger.show()
        RecordWindow.ui.bLowerTrigger.show()
    else:
        RecordWindow.ui.bRaiseTrigger.hide()
        RecordWindow.ui.bLowerTrigger.hide()


    RecordWindow.ui.lblStatus.hide()
    RecordWindow.ui.CanvasWidget.show()
    RecordWindow.UpdateScreen()
    app.processEvents() #force ui update


    if(data.debug==True):
        print "Start timers (Interval:", data.interval ,"ms)..."
        print "Sampling interval:", data.interval, "ms"
        print "sample_smoothing:", data.Sample_smoothing, "x"
        print "value_smoothing:", data.Value_smoothing, "x"
        print "=> record interval:", data.interval*data.Value_smoothing, "ms"
        print "Derivation:", data.Derivation, "(substracts value", data.Derivation, "- value 0)"

    SwitchModes(True) #change to offline mode and silent profile if requested


###########################################################################
##
###########################################################################
def StopMeasurement():
    global app

    GenerateStatisticData()
    Generate_and_show_GraphBitmap(RecordWindow, True)
#    Generate_and_show_GraphBitmap(RecordWindow, False)

    RecordWindow.Canvas.update()
    RecordWindow.UpdateScreen()
    app.processEvents() #force ui update

#    DeleteAlarm() #delete Alarm if existing

#    print "Stop Measurement",  data.run
    if(data.run==True): #in recording
        if(data.debug==True): print "Stop Timer"
        app.timer_GetData.stop()
        StopRecordFile()
        #                    PauseSong() #stop playing song, in case it was playing
        CloseMediaPlayer() #close media player
        UpdateFileHeader() #rewriterecord file (only do when recording is stopped!)

    data.run=False

    RecordWindow.UpdateScreenStop()







############################################################################
## Stretch graph to show more details
############################################################################
def Zoom(direction):
    if(direction==+1): #zoom in
        if(data.Xscale==0): return
        ViewWindow.ui.bZoomOut.setEnabled(True)
        data.Xscale=(data.Xscale-1) % data.arrXscaleLen
    else: # -1, zoom out
        if(data.Xscale==data.arrXscaleLen-1): return
        ViewWindow.ui.bZoomIn.setEnabled(True)
        data.Xscale=(data.Xscale+1) % data.arrXscaleLen





###########################################################################
## Raise or lower the Trigger in the graph + sleep phases
###########################################################################
def MoveTrigger(direction):
    if(direction==+1): #up
        data.Trigger=data.Trigger+1
        if(data.Trigger>data.maxTrigger): data.Trigger=data.maxTrigger
    else: #-1, down
        data.Trigger=data.Trigger-1
        if(data.Trigger<data.minTrigger): data.Trigger=data.minTrigger
    if(data.run==True): records.Trigger=data.Trigger




def About(ui):
    QtGui.QMessageBox.about(ui, _("About")+" "+_("SleepAnalyser")+" "+str(data.version)+"-"+str(data.build), _("Copyright 2010 by George Ruinelli with great support from the MAEMO community")+"\n"+\
    _("Contact:")+" george@ruinelli.ch\n" +\
    _("Website:")+" http://www.ruinelli.ch/sleepanalyser/\n\n"+\
    _("Translations:")+"\n" +\
    data.translation_copyrights)













###########################################################################
##  Class for Main Window
###########################################################################
class frmStartWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # This attribute makes the whole Stacked Window thing work
        self.ui = Ui_StartWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bStartRecording, QtCore.SIGNAL("clicked()"), self.StartRecording)
        QtCore.QObject.connect(self.ui.bView, QtCore.SIGNAL("clicked()"), self.ShowViewWindow)
#        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)

        self.setStyleSheet(data.UIStyle) #Add styles


        QtCore.QObject.connect(self.ui.actionTest_recording, QtCore.SIGNAL("triggered()"), self.StartMeasurementQuick)
        QtCore.QObject.connect(self.ui.actionSettings, QtCore.SIGNAL("triggered()"), self.ShowConfigWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser2.png"))
#            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))
#            self.ui.bQuick.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bStartRecording.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bView.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
#            self.ui.bConfig.setIcon(QIcon(data.app_path+"img/"+"configure.png"))
#            self.ui.actionTest_recording.setIcon(QIcon(data.app_path+"img/"+"configure.png"))


        #set texts
        self.setWindowTitle(_("SleepAnalyser"))
        self.ui.bView.setText(_("View records"))
        self.ui.bStartRecording.setText(_("Start recording"))
#        self.ui.bSetAlarm.setText(_("Set alarm and\nstart recording"))
        self.ui.actionAbout.setText(_("About"))
        self.ui.actionSettings.setText(_("Settings"))
        self.ui.actionHelp.setText(_("Help"))
        self.ui.actionTest_recording.setText(_("Test recording"))

#        self.ui.lblNote.setText(_("Development version, might be unstable!"))





#    ###########################################################################
#    ## Show the collecting window (her the data gets visualised during the logging)
#    ###########################################################################
#    def StartMeasurementNormal(self): ##(Class StartWindow)
#        if(data.debug==True): print "Measure in normal Mode"
#        data.QuickMode=False #run in normal mode
##        data.UseAlarm=False
#        self.StartMeasurement()





    ###########################################################################
    ## Show the collecting window (her the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementQuick(self): ##(Class StartWindow)
        if(data.debug==True): print "Measure in QuickMode"
        data.QuickMode=True #run in quick mode (for testing only)
        data.Xscale=0 #default zoom level
#        data.UseAlarm=False
        self.StartMeasurement()





    ###########################################################################
    ##
    ###########################################################################
    def StartMeasurement(self): ##(Class StartWindow)
        RecordWindow.ui.lblStatus.hide()
        RecordWindow.show()
        app.processEvents() #force ui update

        InitRecord() #clear all old data
        InitMeasurement()
        RecordWindow.UpdateScreenStart()
        RecordWindow.Canvas.show()


    ###########################################################################
    ## Show the log window (where the data from a file gets visualised)
    ###########################################################################
    def ShowViewWindow(self): ##(Class StartWindow)
        ViewWindow.ui.lblStart.setText("")
        ViewWindow.ui.lblEnd.setText("")
        ViewWindow.ui.lblNote.setText("")
        ViewWindow.Canvas.hide()
        ViewWindow.StatisticsCanvas.hide()
        ViewWindow.show()
        app.processEvents() #force ui update

        #reset to default record folder
        if(data.DeviceIsN900==False): data.RecordFolder=data.developmentRecordFolder #folder on devel PC
        else: data.RecordFolder=data.defaultRecordFolder

        LoadFileList()
        data.arrRecordFilesIndex=len(data.arrRecordFiles)-1 #no file selected, show latest

        if(os.path.exists(data.RecordFolder+data.LastRecord)):
            data.RecordFilename=data.LastRecord
        else:
            l=len(data.arrRecordFiles)
            if(l>0):
                data.RecordFilename=data.arrRecordFiles[l-1] #open last (youngest)record file
            else: #no RecordFiles in RecordFolder, show "open file" dialog
                tmp=ViewWindow.ShowFileDialog()
                print "## ShowViewWindow", tmp
                if(not os.path.exists(tmp)): #only continue if valid record file selected
                    StartWindow.show()
                    ViewWindow.hide()
                    return #only continue if valid record file selected
                tmp2=tmp.split("/")
                l=len(tmp2)
                data.RecordFilename=tmp2[l-1]
                data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
                print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename


        ViewWindow.ShowRecord()

#        if(data.RecordFilename!=""):
#            ViewWindow.show()
#        else:
        if(data.RecordFilename==""):
            print "No valid file selected"
            ViewWindow.hide()






    def ShowConfigWindow(self): ##(Class StartWindow)
        ConfigWindow.UpdateScreen()
        ConfigWindow.show()





    def ShowHelpWindow(self): ##(Class StartWindow)
        HelpWindow.load_data()
        HelpWindow.show()




    def StartRecording(self): ##(Class StartWindow)
        SetRecordingWindow.UpdateScreenStart()
        SetRecordingWindow.UpdateScreen()
        SetRecordingWindow.show()



    def UpdateScreenStart(self): ##(Class StartWindow)
        pass



    def About(self): ##(Class StartWindow)
        About(self)



    def closeEvent(self, event): ##(Class StartWindow)
        global config
        WriteConfig()
        delete_all_preexisting_Alarms() #Check if there are old alarms. If SleepAnalyser crashed, they might still be set

















###########################################################################
#  Class for Collecting Window
###########################################################################
class frmRecordWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_RecordWindow()
        self.ui.setupUi(self)
        app.processEvents() #force ui update

#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)

        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)

        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?


        QtCore.QObject.connect(self.ui.bStop, QtCore.SIGNAL("clicked()"), self.StopMeasurement)
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)

        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)

        self.setStyleSheet(data.UIStyle) #Add styles

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            self.ui.lblAnimation.setPixmap(QtGui.QPixmap(data.app_path+"img/anim/0.png"))

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Recording")+")")
        self.ui.lblElapsedTime.setText(_("Elapsed time:")+" 0")
        self.ui.bStop.setText(_("Stop"))
        self.ui.lblAlarmTime.setText(_("Alarm time:")+ " "+_("(no alarm set)"))
        self.ui.lblStatus2.setText(_("Status:"))
        self.ui.lblStatus.setText(_("Loading record..."))
        self.ui.lblstate1.setText(_("Awake"))
        self.ui.lblstate2.setText(_("Light sleep"))
        self.ui.lblstate3.setText(_("Deep sleep"))
        self.ui.actionHelp.setText(_("Help"))
        self.ui.actionAbout.setText(_("About"))
#        app.processEvents() #force ui update



    def UpdateScreenStart(self): ##(Class RecordWindow)
#        t=strftime("%d. %b. %Y %H:%M:%S", data.StartTime)
        t=strftime("%a. %x %X", data.StartTime)
#        self.ui.lblStarttime.setText("Start: "+t)
        self.ui.lblElapsedTime.setText(_("Elapsed time:"))
#        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        self.ui.bStop.setIcon(QIcon(data.app_path+"img/"+"krec2.png"))
        self.ui.bStop.setText(_("Stop"))

        if(records.AlarmIsSet==True): #alarm active
#            m=str(data.AlarmMinute)
#            if(len(m)==1): m="0"+m
#            self.ui.lblAlarmTime.setText(str(data.AlarmHour)+":"+m+" or up to "+str(data.AlarmTimeWindow)+" minutes before that")
            self.ui.lblAlarmTime.setText(_("Alarm time:")+" "+strftime("%H:%M", records.AlarmTimeWindowStart_tuple)+" - "+strftime("%H:%M", records.AlarmTimeWindowEnd_tuple))
        else: #Do only recording, no alarm
            self.ui.lblAlarmTime.setText(_("Alarm time:")+ " "+_("(no alarm set)"))

        self.ui.lblNote.setText(_("Note:"))

        if(records.Use_Lucid_Dream_Song==True): self.ui.lblIcon1.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"music_green.png"))
        else: self.ui.lblIcon1.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"no_music_green.png"))
        if(records.Use_WakeUp_Song==True): self.ui.lblIcon2.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"music_red.png"))
        else: self.ui.lblIcon2.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"no_music_red.png"))
        if(records.AlarmIsSet==True):
            if(data.UseAlternativeAlarm==False):
                self.ui.lblIcon3.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
            else:
                self.ui.lblIcon3.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"External_Alarm.png"))
        else: self.ui.lblIcon3.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"Alarm_off.png"))

        self.UpdateScreen()




    def UpdateScreen(self): ##(Class RecordWindow)
        if(data.run==True): #recording
            t=time.mktime(time.localtime())
            time_difference=t-data.StartTimeSec
            d=time.strftime("%Hh %Mm", time.gmtime(time_difference))
#            self.ui.lblElapsedTime.setText("Elapsed time: "+d + " ("+str(records.Index)+" entries)")
            self.ui.lblElapsedTime.setText(_("Elapsed time:")+" "+d)
        else: self.UpdateScreenStop()

#        self.ui.lblStatus2.setText("Status: "+data.Status)
        self.ui.lblStatus2.setText(data.Status)

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==data.minTrigger): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)





    def UpdateScreenStop(self): ##(Class RecordWindow)
        self.ui.bStop.setText(_("Close"))
#        self.ui.lblStatus2.setText("Status: "+data.Status)
        self.ui.lblStatus2.setText(data.Status)



    def UpdateAnimation(self):
        data.Animation=(data.Animation+1) % 8
        self.ui.lblAnimation.setPixmap(QtGui.QPixmap(data.app_path+"img/anim/"+str(data.Animation)+".png"))
#        print data.Animation


    def StopMeasurement(self): ##(Class RecordWindow)
            self.closeEvent(0)



    def ZoomIn(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Zoom in..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(+1)
        Generate_and_show_GraphBitmap(self, True)
#        Generate_and_show_GraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def ZoomOut(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Zoom out..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(-1)
        Generate_and_show_GraphBitmap(self, True)
#        Generate_and_show_GraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def RaiseTrigger(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Raise trigger level..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        MoveTrigger(+1)
        self.ui.bLowerTrigger.setEnabled(True)
        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)

        Generate_and_show_GraphBitmap(self, True)
#        Generate_and_show_GraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def LowerTrigger(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Lower trigger level..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        MoveTrigger(-1)
        self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==data.minTrigger): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)

        Generate_and_show_GraphBitmap(self, True)
#        Generate_and_show_GraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def closeEvent(self, event): ##(Class RecordWindow)
        global AskForKeepWindow,  StartWindow
#        close=True
        if(data.QuickMode==False ): #running in normal mode
            try:
                event.ignore()  #ignore closing
            except:
                pass
            AskForKeepWindow.UpdateScreen()
            AskForKeepWindow.show()
#            RecordWindow.hide()
#        else:
#            pass #close
        else: #quick mode, delete file anyway and close
            StopMeasurement()
            print "Delete RecordFile: "+data.RecordFilename
            os.remove(data.RecordFolder+data.RecordFilename)
            data.QuickMode=False
            StartWindow.show()
            RecordWindow.hide()
            self.hide()
            SwitchModes(False) #change back to normal mode and general profile if requested



    def EditNote(self): ##(Class RecordWindow)
        r=QInputDialog.getText(None,  _("Note about this record"),  _("Add a note to this record\n(leave empty for no note):"),  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            records.Note=unicode(r[0])
            self.ui.lblNote.setText(_("Note:")+" "+records.Note)
            if(data.run==True): UpdateFileHeader() #rewrite record file (only do when recording is stopped!)



    def About(self): ##(Class RecordWindow)
        About(self)


    def ShowHelpWindow(self): ##(Class RecordWindow)
        HelpWindow.load_data()
        HelpWindow.show()








###########################################################################
## Class for Log Window
###########################################################################
class frmViewWindow(QMainWindow):
    def __init__(self, parent=None):
        global app
#        global Canvas
        QMainWindow.__init__(self, parent)
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_ViewWindow()
        self.ui.setupUi(self)

#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)

        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)
        self.StatisticsCanvas = MyStatCanvas(self)

        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?


#        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
#        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bNext, QtCore.SIGNAL("clicked()"), self.NextRecord)
        QtCore.QObject.connect(self.ui.bLast, QtCore.SIGNAL("clicked()"), self.LastRecord)
        QtCore.QObject.connect(self.ui.bDelete, QtCore.SIGNAL("clicked()"), self.DeleteRecord)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
        QtCore.QObject.connect(self.ui.bOpenDate, QtCore.SIGNAL("clicked()"), self.OpenDate)
        QtCore.QObject.connect(self.ui.bStatistics, QtCore.SIGNAL("clicked()"), self.ShowRecordData)

        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionShowRecordData, QtCore.SIGNAL("triggered()"), self.ShowRecordData)
        QtCore.QObject.connect(self.ui.actionExport, QtCore.SIGNAL("triggered()"), self.Export)

        self.setStyleSheet(data.UIStyle) #Add styles

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bLast.setIcon(QIcon(data.app_path+"img/"+"arrow-left-double.png"))
            self.ui.bNext.setIcon(QIcon(data.app_path+"img/"+"arrow-right-double.png"))
            self.ui.bDelete.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"document-open-folder.png"))
            self.ui.bOpenDate.setIcon(QIcon(data.app_path+"img/"+"date.png"))
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
#            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
#            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            self.ui.bStatistics.setIcon(QIcon(data.app_path+"img/"+"klipper_dock.png"))


        self.setWindowTitle(_("SleepAnalyser")+" ("+_("View")+")")
        self.ui.lblstate1.setText(_("Awake"))
        self.ui.lblstate2.setText(_("Light sleep"))
        self.ui.lblstate3.setText(_("Deep sleep"))
        self.ui.actionShowRecordData.setText(_("Additional information"))
        self.ui.actionHelp.setText(_("Help"))
        self.ui.actionAbout.setText(_("About"))
        self.ui.actionExport.setText(_("Export as image"))




    def UpdateScreen(self): ##(Class ViewWindow)
#        d0=strftime("%a. %x %X", records.StartTime_tuple)
        day=strftime("%A %d. %B %Y", records.StartTime_tuple)
        d0=strftime("%H:%M", records.StartTime_tuple)
#        d1=strftime("%a. %x %X", records.EndTime_tuple)
        d1=strftime("%H:%M", records.EndTime_tuple)
        diff_sec=records.EndTime_seconds-records.StartTime_seconds
#        diff=time.strftime("%H:%M:%S", time.gmtime(diff_sec))
        diff=time.strftime("%Hh %Mm", time.gmtime(diff_sec))
##        The following lines are disabled, currently i dont care if it could record for more than 24h
#        if(diff_sec>86400): d="1 day, " #longer than 24h
#        else: d=""
#        print day
#        self.ui.lblStart.setText(str(day))
        self.ui.lblStart.setText(QtGui.QApplication.translate("ViewWindow", day, None, QtGui.QApplication.UnicodeUTF8))
#        self.ui.lblEnd.setText(str(d1)+" (Duration: "+d+str(diff)+")")
#        self.ui.lblEnd.setText(_("Start:")+" "+d0+", "+_("End:")+" "+d1+", "+_("Duration:")+" "+diff)
#        print d0+" - "+d1+"   ("+diff+")"
        self.ui.lblEnd.setText(d0+" - "+d1+"   ("+diff+")")

        self.ui.lblNote.setText(_("Note:")+" "+records.Note)
#        self.ui.lblNote.setText(records.Note)

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

#        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
#        else: self.ui.bRaiseTrigger.setEnabled(True)
#        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
#        else: self.ui.bLowerTrigger.setEnabled(True)

        if(len(data.arrRecordFiles)<2): #disable last/next buttons if only one file in folder
            self.ui.bNext.setEnabled(False)
            self.ui.bLast.setEnabled(False)
        else:
            self.ui.bNext.setEnabled(True)
            self.ui.bLast.setEnabled(True)

#        ScrollbarStartPercent,ScrollbarEndPercent= ScrollbarData()
#        ScrollbarStartPercent=round(ScrollbarStartPercent,  0)
#        ScrollbarEndPercent=round(ScrollbarEndPercent,  0)
##        print "UpdateScreen,  Scrollbar percent:",  ScrollbarStartPercent,  ScrollbarEndPercent
#        if(ScrollbarStartPercent>0): self.ui.bLeft.setEnabled(True)
#        else: self.ui.bLeft.setEnabled(False)
#        if(ScrollbarEndPercent<100): self.ui.bRight.setEnabled(True)
#        else: self.ui.bRight.setEnabled(False)




    ###########################################################################
    ##
    ###########################################################################
    def ShowRecord(self): ##(Class ViewWindow)
        global app

        self.ui.lblStatus.setText("")
        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()

        self.ui.lblStatus.setText(_("Loading record..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update

        r=LoadRecord()
        if(r==1): #file too long
            QtGui.QMessageBox.warning(self, _("Record too long!"), _("The record is too long. The end will be cut off!"), QtGui.QMessageBox.Ok)
        elif(r==2): #file does not exist
            self.OpenRecord()
            return
#        elif(r==3): #file is corrupted => draw empty graph
#            QtGui.QMessageBox.warning(self, 'File corrupded!', "The data in the record file\n"+data.RecordFilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==8): #file header is corrupted
#            QtGui.QMessageBox.warning(self, 'File header corrupted!', "The header data in the record file\n"+data.RecordFilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==4): #file is empty
#            QtGui.QMessageBox.warning(self, 'File empty!', "The data in the record file\n"+data.RecordFilename+"\ncontains no data!", QtGui.QMessageBox.Ok)
#        elif(r==9): #Record too old
#            QtGui.QMessageBox.warning(self, 'File version too old!', "Can not load record \n"+data.RecordFilename+"\nas it got created from SleepAnalyser version <1.6 and is not compatible with the current SleepAnalyser version.", QtGui.QMessageBox.Ok)
#        LoadFileList()

        if(data.DeviceIsN900==True): #We are on a N900
            #It gives miss behavior on PC, so only do on N900
            self.UpdateScreen()
            app.processEvents() #force ui update

        self.ui.lblStatus.setText(_("Processing data..."))
        app.processEvents() #force ui update
        print "Processing data..."
        ProcessData(True, True)

        self.ui.lblStatus.setText(_("Rendering data..."))
        app.processEvents() #force ui update
        print "Rendering data..."
        GenerateSleepPattern()

        self.ui.lblStatus.setText(_("Generating statistic..."))
        app.processEvents() #force ui update
        print "Generating statistic..."

        GenerateStatisticData()
#        data.Xscale=1 #default zoom level
        data.Xscale=2
#        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)

#        data.ChangesInGraph=True #Make sure it only gets refreshed once

        self.ui.lblStatus.hide()
        ViewWindow.Canvas.show()  #ViewWindow instead of self, as it gets executed in RecordWindow
        ViewWindow.StatisticsCanvas.show() #ViewWindow instead of self, as it gets executed in RecordWindow
        Generate_and_show_GraphBitmap(self, True)
        GenerateStatisticsBitmap(self)

    #Note: On the PC, the lable texts disappears here again until the window looses the focus, but it works fine on the N900, so we ignore it




    ###########################################################################
    ##
    ###########################################################################
    def NextRecord(self): ##(Class ViewWindow)
        data.arrRecordFilesIndex=(data.arrRecordFilesIndex+1)  % len(data.arrRecordFiles)
        data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
        self.ShowRecord()

    ###########################################################################
    ##
    ###########################################################################
    def LastRecord(self): ##(Class ViewWindow)
        data.arrRecordFilesIndex=(data.arrRecordFilesIndex-1)  % len(data.arrRecordFiles)
        data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
        self.ShowRecord()


    ###########################################################################
    ##
    ###########################################################################
    def ZoomIn(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Zoom in..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(+1)

        self.ui.bZoomOut.setEnabled(True)
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)

        Generate_and_show_GraphBitmap(self, True)

        self.ui.lblStatus.hide()




    ###########################################################################
    ##
    ###########################################################################
    def ZoomOut(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Zoom out..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(-1)
        self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        Generate_and_show_GraphBitmap(self, True)

        self.ui.lblStatus.hide()





    ###########################################################################
    ##
    ###########################################################################
    def DeleteRecord(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Delete record..."))
        self.ui.lblStatus.show()
        reply = QtGui.QMessageBox.question(self, _("Delete record?"), _("Are you sure you want to delete this record:")+"\n"+data.RecordFilename, QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)
        if reply == QtGui.QMessageBox.Yes:
            print "Delete RecordFile: "+data.RecordFilename
            os.remove(data.RecordFolder+data.RecordFilename)

            index=data.arrRecordFilesIndex
            LoadFileList()
            if(len(data.arrRecordFiles)>0):
#                data.RecordFilename=data.arrRecordFiles[l-1] #open last (youngest) record file
                data.arrRecordFilesIndex=index-1
                data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
            else: #no RecordFiles in RecordFolder, show "open file" dialog
                tmp=self.ShowFileDialog()
                print "## DeleteRecord", tmp
                if(tmp==""): #only continue if valid record file selected
                    self.closeEvent("") #no file selected, close
                    return
                tmp2=tmp.split("/")
                l=len(tmp2)
                data.RecordFilename=tmp2[l-1]
                data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
                print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
            self.ShowRecord()
            self.UpdateScreen()
        self.ui.lblStatus.hide()
        self.StatisticsCanvas.show()



    ###########################################################################
    ##
    ###########################################################################
    def OpenRecord(self): ##(Class ViewWindow)

        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()
        self.ui.lblStatus.setText(_("Opening file dialog..."))
        self.ui.lblStatus.show()


        tmp=self.ShowFileDialog()
        print "## OpenRecord", tmp
        if(tmp==""): #if user presses cancel tmp will be empty
            self.hide()
            StartWindow.show()
        if(not os.path.exists(tmp)): #only continue if valid record file selected
            self.ui.lblStatus.hide()
            print "file does not exist"
            return
        tmp2=tmp.split("/") #tmp includes whole path + file name!
        l=len(tmp2)
        data.RecordFilename=tmp2[l-1] #extract file name
        data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
        print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
        LoadFileList()
        self.ShowRecord()
        self.ui.lblStatus.hide()
        ViewWindow.Canvas.show()
        ViewWindow.StatisticsCanvas.show()


    ###########################################################################
    ##
    ###########################################################################
    def ShowFileDialog(self): ##(Class ViewWindow)
        data.RecordFolder=string.replace(data.RecordFolder, "~",  data.homedir) #workaround
        if(data.debug==True): print "Open file dialog in:",  data.RecordFolder
        tmp= QFileDialog.getOpenFileName(None,_("Open SleepAnalyser record"), data.RecordFolder, _("Record Files")+" (*.csv)")
        print "Selected file:",  tmp
        return tmp


    ###########################################################################
    ##
    ###########################################################################
    def OpenDate(self): ##(Class ViewWindow)
        DateWindow.show()
        DateWindow.DateChanged()


    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class ViewWindow)
        self.close()
        StartWindow.show()


    ###########################################################################
    ##
    ###########################################################################
    def EditNote(self): ##(Class ViewWindow)
        r=QInputDialog.getText(None,  _("Note about this record"),  _("Add a note to this record\n(leave empty for no note):"),  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            tmp=r[0]
            tmp=unicode(tmp)
            records.Note=tmp
            self.ui.lblNote.setText(_("Note:")+" "+records.Note)
#            self.ui.lblNote.setText(records.Note)
            UpdateFileHeader() #rewrite record file (only do when recording is stopped!)



    ###########################################################################
    ##
    ###########################################################################
    def About(self): ##(Class ViewWindow)
        About(self)


    ###########################################################################
    ##
    ###########################################################################
    def ShowRecordData(self): ##(Class ViewWindow)
        AdditionalInfoWindow.UpdateScreen()
        self.hide()
        AdditionalInfoWindow.show()

    def ShowHelpWindow(self): ##(Class ViewWindow)
        HelpWindow.load_data()
        HelpWindow.show()



    def Export(self): ##(Class ViewWindow)
        filename=string.replace(data.RecordFilename, ".csv",  "")
        tmp=QFileDialog.getSaveFileName(None,_("Export graph as"), data.defaultRecordFolder+filename, _("PNG-Image")+" (*.png)") #;;JPEG-Image (*.jpg)
        if(tmp!=""):
            if(tmp[-4:]!=".png"): tmp=tmp+".png" #add file extention if missing
            print "Export graph as :",  tmp
            r=data.GraphBitMap.save(tmp,  "png",  -1)
            if(r==True): print "Export was successful"
            else: QtGui.QMessageBox.warning(self, _("Could not export file"), _("Failed to save graph as image:")+"\n"+tmp, QtGui.QMessageBox.Ok)
#        print "beep2"""
#        self.show()
#        HelpWindow.show()
#        ViewWindow.show()




#    def UpdateStatus(self, text): ##(Class ViewWindow)
#        self.ui.lblStatus.setText(text)
#        self.ui.lblStatus.show()
#        app.processEvents() #force ui update







###########################################################################
## Class for Wake Up SettingsWindow
###########################################################################
class frmWakeUpSettingsWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_WakeUpSettingsWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSet_Music.setIcon(QIcon(data.app_path+"img/"+"music_red.png"))
            self.ui.bPlay.setIcon(QIcon(data.app_path+"img/"+"media-playback-start.png"))
#
        QtCore.QObject.connect(self.ui.bSet_Music, QtCore.SIGNAL("clicked()"), self.Set_Music)
        QtCore.QObject.connect(self.ui.bSelect_Song, QtCore.SIGNAL("clicked()"), self.Select_Song)
        QtCore.QObject.connect(self.ui.bPlay, QtCore.SIGNAL("clicked()"), self.Play)

        QtCore.QObject.connect(self.ui.SlrVolume,SIGNAL("valueChanged(int)"),self.VolumeSliderMoved)
#        QtCore.QObject.connect(self.ui.SlrDuration,SIGNAL("valueChanged(int)"),self.DurationSliderMoved)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Wake up option")+")")
        self.ui.bSet_Music.setText(_("Set Wake up music"))
        self.ui.lblSong.setText(_("Song:"))
        self.ui.lblDuration.setText(_("Duration:"))
        self.ui.lblVolume.setText(_("Volume:"))
        self.ui.bSelect_Song.setText("...")
        self.ui.lblNote.setText(_("The WakeUp song will be played once with the set volume when the alarm gets triggered. The real alarm gets delayed for the duration of the song."))




    def UpdateScreen(self): ## Class Wake Up SettingsWindow
        self.ui.lblVolume.setText(_("Volume:")+" "+str(data.WakeUp_Song_Volume)+"%")
        self.ui.SlrVolume.setValue(int(data.WakeUp_Song_Volume))

#        m=int(math.floor(data.WakeUp_Song_Duration/60))
#        s=data.WakeUp_Song_Duration-m*60
#        self.ui.lblDuration.setText(_("Duration:")+" "+str(m)+"m "+str(s)+"s")
        self.ui.SlrVolume.setValue(int(data.WakeUp_Song_Volume))

        tmp2=data.WakeUp_Song.split("/")
        l=len(tmp2)
        self.ui.lblSong.setText(_("Song:")+" "+str(tmp2[l-1]))

        self.ui.lblDuration.setText(_("Duration:")+" "+time.strftime("%Hh %Mm %Ss", time.gmtime(data.WakeUp_Song_Duration)))




    def Set_Music(self):## Class Wake Up SettingsWindow
        data.Use_WakeUp_Song=True
        SetRecordingWindow.ui.cbWakeUpSongEnable.setChecked(True)
        SetRecordingWindow.UpdateScreen()
        SetRecordingWindow.show()
        WakeUpSettingsWindow.hide()




    def Select_Song(self):## Class Wake Up SettingsWindow
        tmp= QFileDialog.getOpenFileName(None,_("Select song"), data.WakeUp_Song, "*.mp3 (*.mp3)")
        if(tmp!="" and os.path.exists(tmp)):
            data.WakeUp_Song=tmp
            print "Selected file:",  data.WakeUp_Song
            data.WakeUp_Song_Duration=GetSongLength(data.WakeUp_Song)
        else:
            print "No song selected or not valid, keep last one"
        self.UpdateScreen()



    def Play(self):## Class Wake Up SettingsWindow
        if(not os.path.exists(data.WakeUp_Song)):
            print "Sound file not valid:", data.WakeUp_Song
            return
        Notify(_("Open Song in media player, set volume to ")+ str(data.WakeUp_Song_Volume) + "%")
        PlaySong(data.WakeUp_Song_Volume, data.WakeUp_Song)




    def VolumeSliderMoved(self):## Class Wake Up SettingsWindow
        x=self.ui.SlrVolume.value()
        data.WakeUp_Song_Volume=x
        self.UpdateScreen()


    def closeEvent(self, event): ## Class Wake Up SettingsWindow
        self.Set_Music()






###########################################################################
## Class for Lucid Dream Window
###########################################################################
class frmLucidDreamSettingsWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_LucidDreamSettingsWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSet_Music.setIcon(QIcon(data.app_path+"img/"+"music_green.png"))
            self.ui.bPlay.setIcon(QIcon(data.app_path+"img/"+"media-playback-start.png"))
#
        QtCore.QObject.connect(self.ui.bSet_Music, QtCore.SIGNAL("clicked()"), self.Set_Music)
        QtCore.QObject.connect(self.ui.bSelect_Song, QtCore.SIGNAL("clicked()"), self.Select_Song)
        QtCore.QObject.connect(self.ui.bPlay, QtCore.SIGNAL("clicked()"), self.Play)

        QtCore.QObject.connect(self.ui.SlrVolume,SIGNAL("valueChanged(int)"),self.VolumeSliderMoved)
        QtCore.QObject.connect(self.ui.SlrDelay,SIGNAL("valueChanged(int)"),self.DelaySliderMoved)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Lucid dream option")+")")
        self.ui.bSet_Music.setText(_("Set lucid dream music"))
        self.ui.lblSong.setText(_("Song:"))
        self.ui.lblDelay.setText(_("Delay:"))
        self.ui.lblVolume.setText(_("Volume:"))
        self.ui.bSelect_Song.setText("...")
        self.ui.lblDuration.setText(_("Duration:"))
        self.ui.lblNote.setText(_("The lucid dream song will be played once with the set volume after the set delay time in light or deep sleep."))



    def UpdateScreen(self):## Class Lucid Dream SettingsWindow
        self.ui.lblVolume.setText(_("Volume:")+" "+str(data.Lucid_Dream_Song_Volume)+"%")
        self.ui.SlrVolume.setValue(int(data.Lucid_Dream_Song_Volume))

        m=int(math.floor(data.Lucid_Dream_Song_Delay/60))
        self.ui.lblDelay.setText(_("Delay:")+" "+str(m)+" "+ _("minutes"))
        self.ui.SlrDelay.setValue(int(m))

        tmp2=data.Lucid_Dream_Song.split("/")
        l=len(tmp2)
        self.ui.lblSong.setText(_("Song:")+" "+str(tmp2[l-1]))

        self.ui.lblDuration.setText(_("Duration:")+" "+time.strftime("%Hh %Mm %Ss", time.gmtime(data.Lucid_Dream_Song_Duration)))





    def Set_Music(self):## Class Lucid Dream SettingsWindow
        data.Use_Lucid_Dream_Song=True
        SetRecordingWindow.ui.cbLucidDreamSongEnable.setChecked(True)
        SetRecordingWindow.UpdateScreen()
        SetRecordingWindow.show()
        LucidDreamSettingsWindow.hide()




    def Select_Song(self):## Class Lucid Dream SettingsWindow
        tmp= QFileDialog.getOpenFileName(None,_("Select song"), data.Lucid_Dream_Song, "*.mp3 (*.mp3)")
        if(tmp!="" and os.path.exists(tmp)):
            data.Lucid_Dream_Song=tmp
            print "Selected file:",  data.Lucid_Dream_Song
            data.Lucid_Dream_Song_Duration=GetSongLength(data.Lucid_Dream_Song)
        else:
            print "No song selected or not valid, keep last one"
        self.UpdateScreen()



    def Play(self):## Class Lucid Dream SettingsWindow
        if(not os.path.exists(data.Lucid_Dream_Song)):
            print "Sound file not valid:", data.Lucid_Dream_Song
            return
        Notify(_("Open Song in media player, set volume to ")+ str(data.Lucid_Dream_Song_Volume) + "%")
        PlaySong(data.Lucid_Dream_Song_Volume, data.Lucid_Dream_Song)



    def VolumeSliderMoved(self):## Class Lucid Dream SettingsWindow
        x=self.ui.SlrVolume.value()
        data.Lucid_Dream_Song_Volume=x
        self.UpdateScreen()


    def DelaySliderMoved(self):## Class Lucid Dream SettingsWindow
        x=self.ui.SlrDelay.value()
        data.Lucid_Dream_Song_Delay=x*60
        self.UpdateScreen()



    def closeEvent(self, event): ## Class Lucid Dream SettingsWindow
        self.Set_Music()








###########################################################################
## Class for Set Recording Window
###########################################################################
class frmSetRecordingWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_SetRecordingWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
            self.ui.lblLucid_Dream_Icon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"music_green.png"))
            self.ui.lblAlarm_Time_Icon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
            self.ui.lblWakeUp_Icon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"music_red.png"))
            self.ui.bStart.setIcon(QIcon(data.app_path+"img/"+"krec.png"))

        QtCore.QObject.connect(self.ui.bSet_Lucid_Dream_Options, QtCore.SIGNAL("clicked()"), self.Set_Lucid_Dream_Options)
        QtCore.QObject.connect(self.ui.cbLucidDreamSongEnable, QtCore.SIGNAL("toggled(bool)"), self.Lucid_DreamSongEnable)

        QtCore.QObject.connect(self.ui.bSet_Wake_Up_Options, QtCore.SIGNAL("clicked()"), self.Set_Wake_Up_Options)
        QtCore.QObject.connect(self.ui.cbWakeUpSongEnable, QtCore.SIGNAL("toggled(bool)"), self.WakeUpSongEnable)

        QtCore.QObject.connect(self.ui.bSet_Alarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
        QtCore.QObject.connect(self.ui.cbAlarmEnable, QtCore.SIGNAL("toggled(bool)"), self.AlarmEnable)

        QtCore.QObject.connect(self.ui.bStart, QtCore.SIGNAL("clicked()"), self.Start)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Setup recording")+")")
        self.ui.bStart.setText(_("Start"))
        self.ui.lblLucid_Dream.setText(_("Lucid dream option"))
        self.ui.lblLucid_Dream_Volume.setText(_("Volume:"))
        self.ui.lblAlarmTime2.setText(_("Alarm option"))
        self.ui.bSet_Alarm.setText("...")
        self.ui.lblLucid_Dream_Song.setText(_("Song:"))
        self.ui.bSet_Lucid_Dream_Options.setText("...")
        self.ui.lblAlarmTime.setText(_("Alarm time:"))
        self.ui.lblWakeUp_Song.setText(_("Song:"))
        self.ui.lblWakeUp.setText(_("Wake up option"))
        self.ui.bSet_Wake_Up_Options.setText("...")
        self.ui.lblWakeUp_Volume.setText(_("Volume:"))
        self.ui.lblAlarmType.setText(_("Alarm type:"))






    def UpdateScreenStart(self):
        if(data.Use_Lucid_Dream_Song==True): self.ui.cbLucidDreamSongEnable.setChecked(True)
        else: self.ui.cbLucidDreamSongEnable.setChecked(False)

        if(data.Use_WakeUp_Song==True): self.ui.cbWakeUpSongEnable.setChecked(True)
        else: self.ui.cbWakeUpSongEnable.setChecked(False)

        if(data.UseAlarm==True): self.ui.cbAlarmEnable.setChecked(True)
        else: self.ui.cbAlarmEnable.setChecked(False)

        if(data.Lucid_Dream_Song=="" or not os.path.exists(data.Lucid_Dream_Song)): data.Lucid_Dream_Song_Duration=0
        if(data.WakeUp_Song=="" or not os.path.exists(data.WakeUp_Song)): data.WakeUp_Song_Duration=0



    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class SetRecordingWindow)
        if(data.Use_Lucid_Dream_Song==True):
            self.ui.lblLucid_Dream_Song.setEnabled(True)
            self.ui.lblLucid_Dream_Volume.setEnabled(True)
        else:
            self.ui.lblLucid_Dream_Song.setEnabled(False)
            self.ui.lblLucid_Dream_Volume.setEnabled(False)

        if(data.Lucid_Dream_Song!=""):
            tmp2=data.Lucid_Dream_Song.split("/")
            l=len(tmp2)
            txt=tmp2[l-1]
        else:
            txt=""
        self.ui.lblLucid_Dream_Song.setText(_("Song:")+" "+str(txt))
        m=int(math.floor(data.Lucid_Dream_Song_Delay/60))
        txt=_("start after $ minutes in light or deep sleep")
        txt=string.replace(txt, "$", str(m)) + ", "+_("Duration:")+" "+time.strftime("%Hh %Mm %Ss", time.gmtime(data.Lucid_Dream_Song_Duration))
        self.ui.lblLucid_Dream_Volume.setText(_("Volume:")+" "+str(data.Lucid_Dream_Song_Volume)+"%, "+ txt)


        ######################
        if(data.Use_WakeUp_Song==True):
            self.ui.lblWakeUp_Song.setEnabled(True)
            self.ui.lblWakeUp_Volume.setEnabled(True)
        else:
            self.ui.lblWakeUp_Song.setEnabled(False)
            self.ui.lblWakeUp_Volume.setEnabled(False)

        if(data.WakeUp_Song!=""):
            tmp2=data.WakeUp_Song.split("/")
            l=len(tmp2)
            txt=tmp2[l-1]
        else:
            txt=""
        self.ui.lblWakeUp_Song.setText(_("Song:")+" "+str(txt))
        self.ui.lblWakeUp_Volume.setText(_("Volume:")+" "+str(data.WakeUp_Song_Volume)+"%, "+_("Duration:")+" "+time.strftime("%Hh %Mm %Ss", time.gmtime(data.WakeUp_Song_Duration)))

        ########################
        if(data.UseAlarm==True):
            self.ui.lblAlarmTime.setEnabled(True)
            self.ui.lblAlarm_Time_Icon.setEnabled(True)
            self.ui.lblAlarmTime.setEnabled(True)
            self.ui.cbWakeUpSongEnable.setEnabled(True)
            self.ui.bSet_Wake_Up_Options.setEnabled(True)
            self.ui.lblWakeUp_Song.setEnabled(True)
            self.ui.lblWakeUp_Icon.setEnabled(True)
            self.ui.lblWakeUp.setEnabled(True)
            self.ui.lblAlarmType.setEnabled(True)
            if(data.Use_WakeUp_Song==True):
                self.ui.lblWakeUp_Song.setEnabled(True)
                self.ui.lblWakeUp_Volume.setEnabled(True)
            else:
                self.ui.lblWakeUp_Song.setEnabled(False)
                self.ui.lblWakeUp_Volume.setEnabled(False)

        else:
            self.ui.lblAlarmTime.setEnabled(False)
            self.ui.lblAlarm_Time_Icon.setEnabled(False)
            self.ui.lblAlarmTime.setEnabled(False)
            self.ui.cbWakeUpSongEnable.setEnabled(False)
            self.ui.bSet_Wake_Up_Options.setEnabled(False)
            self.ui.lblWakeUp_Icon.setEnabled(False)
            self.ui.lblWakeUp.setEnabled(False)
            self.ui.lblWakeUp_Song.setEnabled(False)
            self.ui.lblWakeUp_Volume.setEnabled(False)
            self.ui.lblAlarmType.setEnabled(False)

        if(data.AlarmHour==-1 or data.AlarmMinute==-1):
            data.AlarmHour=7
            data.AlarmMinute=0
            data.AlarmTimeWindow=30
        al0=int(data.AlarmHour)*60+int(data.AlarmMinute)-data.AlarmTimeWindow
        h0=int(math.floor(al0/60))
        m0=al0-(h0*60)
        m1=data.AlarmMinute
        if(m0<10): m0="0"+str(m0)
        if(m1<10): m1="0"+str(m1)
        txt=str(h0)+":"+str(m0) + "-" + str(data.AlarmHour)+":"+str(m1)
        self.ui.lblAlarmTime.setText(_("Alarm time:")+" "+ txt)

        if(data.UseAlternativeAlarm==False):
            self.ui.lblAlarmType.setText(_("Alarm type:")+" "+ _("Normal"))
        else:
            self.ui.lblAlarmType.setText(_("Alarm type:")+" "+_("External alarm")+" "+str(data.AlternativeAlarm))

#        print "data.AlternativeAlarm", data.AlternativeAlarm

    def Lucid_DreamSongEnable(self, state):
        if(state==True):
            data.Use_Lucid_Dream_Song=True
        else:
            data.Use_Lucid_Dream_Song=False
        self.UpdateScreen()



    def Set_Lucid_Dream_Options(self):
        LucidDreamSettingsWindow.UpdateScreen()
        LucidDreamSettingsWindow.show()
        SetRecordingWindow.hide()



    def WakeUpSongEnable(self, state):
        if(state==True):
            data.Use_WakeUp_Song=True
        else:
            data.Use_WakeUp_Song=False
        self.UpdateScreen()


    def Set_Wake_Up_Options(self):
        WakeUpSettingsWindow.UpdateScreen()
        WakeUpSettingsWindow.show()
        SetRecordingWindow.hide()



    def AlarmEnable(self, state):
        if(state==True):
            data.UseAlarm=True
        else:
            data.UseAlarm=False
        self.UpdateScreen()



    def SetAlarm(self):
        AlarmSettingsWindow.UpdateScreen()
        AlarmSettingsWindow.show()
        SetRecordingWindow.hide()




    def Start(self):
        InitRecord() #clear all old data, initialise next measurement

        records.Use_Lucid_Dream_Song = data.Use_Lucid_Dream_Song
        records.Lucid_Dream_Song = data.Lucid_Dream_Song
        records.Lucid_Dream_Song_Delay = data.Lucid_Dream_Song_Delay
        records.Lucid_Dream_Song_Duration = data.Lucid_Dream_Song_Duration
        records.Lucid_Dream_Song_Volume = data.Lucid_Dream_Song_Volume

        if(data.UseAlarm==False):#its not possible to have a wake up song without alarm
            records.Use_WakeUp_Song=False
        else:
            records.Use_WakeUp_Song = data.Use_WakeUp_Song
            records.WakeUp_Song = data.WakeUp_Song
            records.WakeUp_Song_Duration = data.WakeUp_Song_Duration
            records. WakeUp_Song_Volume= data.WakeUp_Song_Volume


        if(data.UseAlarm==True): #Use alarm
            try:
                now=int(strftime("%H", time.localtime()))*3600+int(strftime("%M", time.localtime()))*60
                al=int(data.AlarmHour)*3600+int(data.AlarmMinute)*60
                alarmdiff=al-now
                if(alarmdiff<0):  #set to tomorrow if time already is elapsed today
                    alarmdiff=alarmdiff+86400

                records.AlarmTimeWindow_minutes=data.AlarmTimeWindow

                records.AlarmTime_seconds=int((records.StartTime_seconds+alarmdiff)/60)*60 #set seconds to zero
                records.AlarmTime_tuple=Seconds2Tuple(records.AlarmTime_seconds)

                records.AlarmTimeWindowEnd_tuple=records.AlarmTime_tuple
                records.AlarmTimeWindowEnd_seconds=records.AlarmTime_seconds

                records.AlarmTimeWindowStart_seconds=records.AlarmTimeWindowEnd_seconds-records.AlarmTimeWindow_minutes*60
                records.AlarmTimeWindowStart_tuple=Seconds2Tuple(records.AlarmTimeWindowStart_seconds)

            except: #If nothing selected (workaround)
                print "ERROR: Selected time not valid!"

        records.UseAlternativeAlarm= data.UseAlternativeAlarm
        records.AlternativeAlarm=data.AlternativeAlarm

        data.QuickMode=False #run in normal mode

        print "*", strftime("%x %X"), "SetAlarm(Class AlarmTimeWindow) => DeleteAlarm" #for log file
        DeleteAlarm() #delete old alarm if existing
        delete_all_preexisting_Alarms() #Check if there are old alarms. If SleepAnalyser crashed, they might still be set

        if(data.UseAlarm==True): #Use alarm
            records.AlarmIsSet=True
            print "*", strftime("%x %X"), "SetAlarm(Class AlarmTimeWindow) => SetAlarm" #for log file
            SetAlarm() #set alarm for the last possible time

        config.set('main', 'Use_Lucid_Dream_Song', str(data.Use_Lucid_Dream_Song))
        config.set('main', 'Lucid_Dream_Song', str(data.Lucid_Dream_Song))
        config.set('main', 'Lucid_Dream_Song_Volume', str(data.Lucid_Dream_Song_Volume))
        config.set('main', 'Lucid_Dream_Song_Delay', str(data.Lucid_Dream_Song_Delay))
        config.set('main', 'Lucid_Dream_Song_Duration', str(data.Lucid_Dream_Song_Duration))

        config.set('main', 'Use_WakeUp_Song', str(data.Use_WakeUp_Song))
        config.set('main', 'WakeUp_Song', str(data.WakeUp_Song))
        config.set('main', 'WakeUp_Song_Volume', str(data.WakeUp_Song_Volume))
        config.set('main', 'WakeUp_Song_Duration', str(data.WakeUp_Song_Duration))

        config.set('main', 'UseAlarm', str(data.UseAlarm))
        config.set('main', 'AlarmHour', str(data.AlarmHour))
        config.set('main', 'AlarmMinute', str(data.AlarmMinute))
        config.set('main', 'AlarmTimeWindow', str(data.AlarmTimeWindow))

        config.set('main', 'UseAlternativeAlarm', str(data.UseAlternativeAlarm))

        WriteConfig() #good point to save config

        InitMeasurement()

        RecordWindow.show()
        SetRecordingWindow.hide()
        RecordWindow.UpdateScreenStart()



    def closeEvent(self, event): ##(Class SetRecordingWindow)
        self.close()
        StartWindow.show()













###########################################################################
## Class for Help Window
###########################################################################
class frmHelpWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_HelpWindow()
        self.ui.setupUi(self)

        self.setStyleSheet(data.UIStyle) #Add styles



    def load_data(self):
        filename=data.app_path+"help_"+data.language+".htm"
        if(os.path.exists(filename)):
            print "Open help file:", filename
        else:
            print "Help file", filename, "does not exist"
            filename=data.app_path+"help_en.htm"
            print "Open default help file:", filename

        file = open(filename, 'r')
        txt=""
        for line in file: txt=txt+line+"\n"
        file.close()
        txt=QtGui.QApplication.translate("", txt, None, QtGui.QApplication.UnicodeUTF8)
        self.ui.textBrowser.setHtml(txt)

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Help")+")")





    def closeEvent(self, event): ##(Class HelpWindow)
        self.close()
        StartWindow.show()





###########################################################################
## Class for Configuration Window
###########################################################################
class frmConfigWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_ConfigWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        self.ui.imgYscale_Movement.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"Yscale_Movement.png"))
        self.ui.imgYscale_Sleep_Pattern.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"Yscale_Sleep_Pattern.png"))
        self.ui.imgYoffset_Sleep_Pattern.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"Yoffset_Sleep_Pattern.png"))
        self.ui.imgDelayStart.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"DelayStart.png"))

        self.ui.bYscale_Movement_Reset.setIcon(QIcon(data.app_path+"img/"+"edit-undo.png"))
        self.ui.bYscale_Sleep_Pattern_Reset.setIcon(QIcon(data.app_path+"img/"+"edit-undo.png"))
        self.ui.bYoffset_Sleep_Pattern_Reset.setIcon(QIcon(data.app_path+"img/"+"edit-undo.png"))
        self.ui.bDelayStart_Reset.setIcon(QIcon(data.app_path+"img/"+"edit-undo.png"))

        QtCore.QObject.connect(self.ui.bEditUserName, QtCore.SIGNAL("clicked()"), self.EditUserName)
        QtCore.QObject.connect(self.ui.SlrDelayStart,SIGNAL("valueChanged(int)"),self.DelaySliderMoved)
        QtCore.QObject.connect(self.ui.SlrYscale_Movement,SIGNAL("valueChanged(int)"),self.YscaleSliderMoved_Movement)
        QtCore.QObject.connect(self.ui.SlrYscale_Sleep_Pattern,SIGNAL("valueChanged(int)"),self.YscaleSliderMoved_Sleep_Pattern)
        QtCore.QObject.connect(self.ui.SlrYoffset_Sleep_Pattern,SIGNAL("valueChanged(int)"),self.YoffsetSliderMoved_Sleep_Pattern)
        QtCore.QObject.connect(self.ui.bYscale_Movement_Reset, QtCore.SIGNAL("clicked()"), self.Yscale_Movement_Reset)
        QtCore.QObject.connect(self.ui.bYscale_Sleep_Pattern_Reset, QtCore.SIGNAL("clicked()"), self.Yscale_Sleep_Pattern_Reset)
        QtCore.QObject.connect(self.ui.bYoffset_Sleep_Pattern_Reset, QtCore.SIGNAL("clicked()"), self.Yoffset_Sleep_Pattern_Reset)
        QtCore.QObject.connect(self.ui.bDelayStart_Reset, QtCore.SIGNAL("clicked()"), self.DelayStart_Reset)


        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bAlternativeAlarmPlay.setIcon(QIcon(data.app_path+"img/"+"media-playback-start.png"))

        QtCore.QObject.connect(self.ui.bOK, QtCore.SIGNAL("clicked()"), self.OK)
        QtCore.QObject.connect(self.ui.bAlternativeAlarmPlay, QtCore.SIGNAL("clicked()"), self.AlternativeAlarmPlay)
#        QtCore.QObject.connect(self.ui.bCancel, QtCore.SIGNAL("clicked()"), self.Cancel)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_User), _("Name"))
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_General), _("General"))
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_Advanced_1), _("Advanced") +" 1")
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_Advanced_2), _("Advanced") +" 2")
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_Alarm), _("Alarm"))

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Settings")+")")
        self.ui.cOfflineMode.setText(_("Change to Offline Mode while recording"))
        self.ui.cSilentProfile.setText(_("Change to Silent Profile while recording"))
        self.ui.cStopAfterAlarm.setText(_("Stop recording after alarm occured"))
        self.ui.lblUser.setText(_("User name:"))
        self.ui.bEditUserName.setText(_("Change"))
        self.ui.lblDelayStart.setText(_("Start delay:"))
        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):"))
        self.ui.lblYscale_Sleep_Pattern.setText(_("Y scaling sleep graph (blue):"))
        self.ui.lblYoffset_Sleep_Pattern.setText(_("Y offset sleep graph (blue):"))
        self.ui.lblAlternativeAlarm.setText(_("Alternative alarm:"))
        self.ui.lblAlternativeAlarmNote.setText(_("Type in any program name to be run instead the normal alarm.\nI.e. evilalarm --wakeup."))

        if(data.AlternativeAlarm!="evilalarm --wakeup"):
            self.ui.cbAlternativeAlarm.addItem(data.AlternativeAlarm)
            self.ui.cbAlternativeAlarm.setCurrentIndex(1)

#        self.ui.lblMore.setText(_("More settings can be changed at your own risk in the file\n~/.SleepAnalyer/SleepAnalyser.conf."))
#        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_More), _("More"))
        self.ui.bOK.setText(_("OK"))

        self.ui.tabWidget.setCurrentIndex(0)  #change to first page


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class ConfigWindow)
        if(data.OfflineMode==True): self.ui.cOfflineMode.setChecked(True)
        else: self.ui.cOfflineMode.setChecked(False)
        if(data.SilentProfile==True): self.ui.cSilentProfile.setChecked(True)
        else: self.ui.cSilentProfile.setChecked(False)
        if(data.StopAfterAlarm==True): self.ui.cStopAfterAlarm.setChecked(True)
        else: self.ui.cStopAfterAlarm.setChecked(False)
        self.ui.lblUser.setText(_("User name:")+" "+data.user)
        self.ui.lblDelayStart.setText(_("Start delay:")+" "+str(data.DelayStartTime)+" seconds")
        self.ui.SlrDelayStart.setValue(int(data.DelayStartTime))

        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):")+" "+str(data.Yscale_Movement))
        self.ui.SlrYscale_Movement.setValue(int(data.Yscale_Movement))

        self.ui.lblYscale_Sleep_Pattern.setText(_("Y scaling sleep graph (blue):")+" "+str(data.Yscale_Sleep_Pattern))
        self.ui.SlrYscale_Sleep_Pattern.setValue(int(data.Yscale_Sleep_Pattern))

        self.ui.lblYoffset_Sleep_Pattern.setText(_("Y offset sleep graph (blue):")+" "+str(data.Yoffset_Sleep_Pattern))
        self.ui.SlrYoffset_Sleep_Pattern.setValue(int(data.Yoffset_Sleep_Pattern))



#    def Cancel(self):
#        self.hide()
#        StartWindow.show()



    def OK(self):
#        self.save()
        self.close()
#        StartWindow.show()



    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event):  ##(Class ConfigWindow)
        self.save() #closing the window is equal to saving
#        self.close()
#        event.ignore() #do not close
        StartWindow.show()



    def save(self): ##(Class ConfigWindow)
#        print "save"
        if(self.ui.cOfflineMode.isChecked()==True):
            data.OfflineMode=True
        else:
            data.OfflineMode=False
        config.set('main', 'OfflineMode', str(data.OfflineMode))

        if(self.ui.cSilentProfile.isChecked()==True):
            data.SilentProfile=True
        else:
            data.SilentProfile=False
        config.set('main', 'SilentProfile', str(data.SilentProfile))

        if(self.ui.cStopAfterAlarm.isChecked()==True):
            data.StopAfterAlarm=True
        else:
            data.StopAfterAlarm=False
        config.set('main', 'StopAfterAlarm', str(data.StopAfterAlarm))

        data.AlternativeAlarm=str(self.ui.cbAlternativeAlarm.currentText())

        config.set('main', 'user', str(data.user.encode( "utf-8" )))
        config.set('main', 'DelayStartTime', str(data.DelayStartTime))
        config.set('main', 'Yscale_Movement', str(data.Yscale_Movement))
        config.set('main', 'Yscale_Sleep_Pattern', str(data.Yscale_Sleep_Pattern))
        config.set('main', 'Yoffset_Sleep_Pattern', str(data.Yoffset_Sleep_Pattern))
        config.set('main', 'AlternativeAlarm', str(data.AlternativeAlarm))



    ###########################################################################
    ##
    ###########################################################################
    def EditUserName(self): ##(Class ConfigWindow)
        EditUserName()
        self.ui.lblUser.setText(_("User name:")+" "+data.user)


    ###########################################################################
    ##
    ###########################################################################
    def DelaySliderMoved(self): ##(Class ConfigWindow)
        x=self.ui.SlrDelayStart.value()
        data.DelayStartTime=x
        self.ui.lblDelayStart.setText(_("Start delay:")+" "+str(data.DelayStartTime)+" "+_("seconds"))


    ###########################################################################
    ##
    ###########################################################################
    def YscaleSliderMoved_Movement(self): ##(Class ConfigWindow)
        x=self.ui.SlrYscale_Movement.value()
        data.Yscale_Movement=int(x)
        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):")+" "+str(data.Yscale_Movement))


    ###########################################################################
    ##
    ###########################################################################
    def YscaleSliderMoved_Sleep_Pattern(self): ##(Class ConfigWindow)
        x=self.ui.SlrYscale_Sleep_Pattern.value()
        data.Yscale_Sleep_Pattern=int(x)
        self.ui.lblYscale_Sleep_Pattern.setText(_("Y scaling sleep graph (blue):")+" "+str(data.Yscale_Sleep_Pattern))


    ###########################################################################
    ##
    ###########################################################################
    def YoffsetSliderMoved_Sleep_Pattern(self): ##(Class ConfigWindow)
        x=self.ui.SlrYoffset_Sleep_Pattern.value()
        data.Yoffset_Sleep_Pattern=int(x)
        self.ui.lblYoffset_Sleep_Pattern.setText(_("Y offset sleep graph (blue):")+" "+str(data.Yoffset_Sleep_Pattern))



    ###########################################################################
    ##
    ###########################################################################
    def Yscale_Movement_Reset(self): ##(Class ConfigWindow)
        self.ui.SlrYscale_Movement.setValue(int(data.Yscale_Movement_default))
        self.YscaleSliderMoved_Movement()

    ###########################################################################
    ##
    ###########################################################################
    def Yscale_Sleep_Pattern_Reset(self): ##(Class ConfigWindow)
        self.ui.SlrYscale_Sleep_Pattern.setValue(int(data.Yscale_Sleep_Pattern_default))
        self.YscaleSliderMoved_Sleep_Pattern()

    ###########################################################################
    ##
    ###########################################################################
    def Yoffset_Sleep_Pattern_Reset(self): ##(Class ConfigWindow)
        self.ui.SlrYoffset_Sleep_Pattern.setValue(int(data.Yoffset_Sleep_Pattern_default))
        self.YoffsetSliderMoved_Sleep_Pattern()

    ###########################################################################
    ##
    ###########################################################################
    def DelayStart_Reset(self): ##(Class ConfigWindow)
        self.ui.SlrDelayStart.setValue(int(data.DelayStartTime_default))
        self.DelaySliderMoved()



    def AlternativeAlarmPlay(self):
        cmd=str(self.ui.cbAlternativeAlarm.currentText())
        Notify(_("Run command:")+" "+cmd)
        print "Run command: ", cmd
        b = os.popen(cmd)
        b.close()






###########################################################################
## Class for Date Window
###########################################################################
class frmDateWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_DateWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))

        QtCore.QObject.connect(self.ui.cal, QtCore.SIGNAL('selectionChanged()'), self.DateChanged)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.ui.cal.showToday()

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Select date")+")")


    ###########################################################################
    ##
    ###########################################################################
    def DateChanged(self): ##(Class ConfigWindow)
        date = self.ui.cal.selectedDate()
        date=str(date.toPyDate())

#        print "Load list of all csv files in folder:",  data.RecordFolder + " with date " + date
        output = os.popen("cd "+str(data.RecordFolder)+"; ls -C1 *"+date+"*.csv").read() # run ls and make the output look like a file I can read
        arr=output.split("\n")

        amount=len(arr)-1

#TODO: visualise dates with records (=> not easely possible)

        txt=string.replace(_("Found % Record(s) for this date."),"%", str(amount))
        self.ui.lblResult.setText(txt)
        if(amount>0):
            self.ui.bOpen.setEnabled(True)
            data.RecordFileSelectedDate=arr[0] #select first file of that date for opening

            for i in range(0,  len(data.arrRecordFiles)-1):
                if(data.arrRecordFiles[i]==data.RecordFileSelectedDate):
#                    print "found:",  i
                    data.arrRecordFilesIndex=i
                    break

        else: self.ui.bOpen.setEnabled(False)


    ###########################################################################
    ##
    ###########################################################################
    def OpenRecord(self): ##(Class ConfigWindow)
        data.RecordFilename=data.RecordFileSelectedDate
        print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
        DateWindow.hide()
        ViewWindow.ShowRecord()








###########################################################################
## Class for Date Window
###########################################################################
class frmAdditionalInfoWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_AdditionalInfoWindow()
        self.ui.setupUi(self)

        self.setStyleSheet(data.UIStyle) #Add styles

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Record Info\'s")+")")


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AdditionalInfoWindow)
        shortfolder=string.replace(data.RecordFolder, data.homedir, "~/")
        shortfolder=string.replace(shortfolder, "//", "/") #workaround

        d0=strftime("%A %d. %B %Y  %H:%M", records.StartTime_tuple)
#        EndTime_seconds=Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        EndTime_tuple=Seconds2Tuple(records.EndTime_seconds)
        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", records.EndTime_tuple)):
            d1=strftime("%H:%M", records.EndTime_tuple) #on same day, only show date once
        else:
            d1=strftime("%a. %d.%m.%Y  %H:%M", records.EndTime_tuple)

        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        diff=time.strftime("%X", time.gmtime(diff_sec))
        diff=time.strftime("%Hh %Mm", time.gmtime(diff_sec))

        if(diff_sec>86400): diff="1 day, "+diff #longer than 24h

        if(records.AlarmIsSet==True):
            a0=strftime("%A %d. %B %Y  %H:%M", records.AlarmTimeWindowStart_tuple)
            if(strftime("%d.%m.%Y", records.AlarmTimeWindowStart_tuple)==strftime("%d.%m.%Y", records.AlarmTimeWindowEnd_tuple)):
                a1=strftime("%H:%M", records.AlarmTimeWindowEnd_tuple) #on same day, only show date once
            else:
                a1=strftime("%A %d. %B %Y  %H:%M", records.AlarmTimeWindowEnd_tuple)
            alarm="<b>"+_("Alarm Time Window:")+"</b> "+QtGui.QApplication.translate("", a0, None, QtGui.QApplication.UnicodeUTF8)+" - "+QtGui.QApplication.translate("", a1, None, QtGui.QApplication.UnicodeUTF8) + "<br>"
            if(records.AlarmTime_seconds!=-1): #alarm was set
                if(records.EndTime_seconds>=records.AlarmTime_seconds-60): #alarm got set off during recording (-60 is workaround)
                    alarm=alarm+"<b>"+_("Alarm:")+"</b> "+QtGui.QApplication.translate("", strftime("%A %d. %B %Y  %H:%M", records.AlarmTime_tuple), None, QtGui.QApplication.UnicodeUTF8)+"<br>"
                    if(records.AlarmTimeWindowEnd_seconds<=records.AlarmTime_seconds):
                        alarm=alarm+"<b>"+_("Alarm reason:")+"</b> "+_("End of alarm time window")+"<br>"
                    else:
                        alarm=alarm+"<b>"+_("Alarm reason:")+"</b> "+_("Movements over trigger level")+"<br>"
                else:
                    alarm=alarm+"<b>"+_("Alarm:")+"</b> "+_("Never went off")+"<br>"
            else:
                alarm=alarm+"<b>"+_("Alarm:")+"</b> "+_("Never went off")+"<br>"
            alarm= alarm+"<b>"+_("Alarm trigger level:")+"</b> "+str(records.Trigger)+"<br>"
            if(records.UseAlternativeAlarm==False):
                alarm= alarm+"<b>"+_("Alarm type:")+"</b> "+ _("Normal")
            else:
                alarm= alarm+"<b>"+_("Alarm type:")+"</b> "+_("External alarm")+" "+str(records.AlternativeAlarm)
        else:
            alarm="<b>"+_("Alarm:")+"</b> "+_("Not used")

        txt= "<b>"+_("File:")+"</b> "+str(shortfolder)+str(data.RecordFilename+"<br>") + \
        "<b>"+_("Date/Time:")+"</b> "+QtGui.QApplication.translate("", d0, None, QtGui.QApplication.UnicodeUTF8)+" - "+d1+"<br>" + \
        "<b>"+_("Duration:")+"</b> "+ diff+"<br>" + \
        "<b>"+_("Note:")+"</b> "+QtGui.QApplication.translate("", records.Note, None, QtGui.QApplication.UnicodeUTF8)+"<br>" + \
        "<b>"+_("User name:")+"</b> "+QtGui.QApplication.translate("", records.User, None, QtGui.QApplication.UnicodeUTF8) +"<br>" + \
        "<b>"+_("Record entries:")+"</b> "+str(records.Index)+" (1/"+str(int(records.Sampling_interval*records.Value_smoothing/1000))+" "+_("seconds")+")<br>" +\
        "<b>"+_("Record version:")+"</b> "+str(records.SWversion) + "<br>"

        txt= txt+"<b>"+_("Condition of record:")+"</b> "+QtGui.QApplication.translate("", records.status, None, QtGui.QApplication.UnicodeUTF8)

        txt=txt+"<hr>"

        txt= txt+ alarm

        txt=txt+"<hr>"

        if(records.Use_Lucid_Dream_Song==True):
            if(records.Lucid_Dream_Song_Start_tuple==-1):
                txt= txt+"<b>" + _("Lucid dream song:") +"</b> " + _("Never went off") + "<br>"
            else:
                txt= txt+"<b>" + _("Lucid dream song start:") +"</b> " + strftime("%A %d. %B %Y", records.Lucid_Dream_Song_Start_tuple) + "<br>" + \
                "<b>" + _("Lucid dream song:")+"</b> "+str(records.Lucid_Dream_Song) + "<br>" + \
                "<b>" + _("Lucid dream song duration:")+"</b> "+str(int(records.Lucid_Dream_Song_Duration/60)) +" "+ _("minutes")+"<br>" + \
                "<b>" + _("Lucid dream song volume:")+"</b> "+str(records.Lucid_Dream_Song_Volume) + "%"+"<br>"
            txt= txt+"<b>" + _("Lucid dream song delay:")+"</b> "+str(int(records.Lucid_Dream_Song_Delay/60)) +" "+ _("minutes")
        else:
            txt= txt+"<b>" + _("Lucid dream song:") + "</b> " + _("Not used")

        txt=txt+"<hr>"

        if(records.Use_WakeUp_Song==True):
            if(records.WakeUp_Song_Start_tuple==-1):
                txt= txt+"<b>" + _("Wake up song:") +"</b> " + _("Never went off")
            else:
                txt= txt+"<b>" + _("Wake up song start:") +"</b> " + strftime("%A %d. %B %Y", records.WakeUp_Song_Start_tuple) + "<br>" + \
                "<b>" + _("Wake up song:")+"</b> "+str(records.WakeUp_Song) + "<br>" + \
                "<b>" + _("Wake up song duration:")+"</b> "+str(int(records.WakeUp_Song_Duration/60)) +" "+ _("minutes")+"<br>" + \
                "<b>" + _("Wake up song volume:")+"</b> "+str(records.WakeUp_Song_Volume) + "%"
        else:
            txt= txt+"<b>" + _("Wake up song:") + "</b> " + _("Not used")



        txt=txt+"<hr>"
        txt=txt+"<b>"+_("Statistics:")+"</b><table CELLPADDING=2>\n" + \
                "<tr bgcolor=#DCDCDC><td></td>                                                               <td><b>"+_("Total:")+" </b></td>                                     <td><b>"+_("Maximum:")+"</b></td>                                    <td><b>"+_("Count:")+" </b></td>                     <td colspan=2><b>"+_("Time percent:")+"</b></td>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+_("Awake")+":</b></td>         <td><nobr>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[0][0]))+ "</nobr></td>    <td><nobr>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[0][2]))+"</nobr></td>    <td>"+str(records.arrStatData[0][1])+"</td>    <td>"+str(records.arrStatData[0][3])+"%</td><td><img src=\""+data.app_path+"img/"+"Awake.png"+"\" width="+str(records.arrStatData[0][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+_("Light sleep")+":</b></td>         <td><nobr>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[1][0]))+ "</nobr></td>    <td><nobr>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[1][2]))+"</nobr></td>    <td>"+str(records.arrStatData[1][1])+"</td>    <td>"+str(records.arrStatData[1][3])+"%</td><td><img src=\""+data.app_path+"img/"+"Light_Sleep.png"+"\" width="+str(records.arrStatData[1][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+_("Deep sleep")+":</b></td>        <td><nobr>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[2][0]))+"</nobr></td>    <td><nobr>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[2][2]))+"</nobr></td>    <td>"+str(records.arrStatData[2][1])+"</td>    <td>"+str(records.arrStatData[2][3])+"%</td><td><img src=\""+data.app_path+"img/"+"Deep_Sleep.png"+"\" width="+str(records.arrStatData[2][3]*2)+" height=25</td></tr>\n" + \
              "</table>\n"

        txt=txt+"<hr>"
        txt=txt+"<b>"+_("Sampling interval:")+"</b> "+str(records.Sampling_interval) + " ms<br>" + \
        "<b>"+_("Sampling smoothing:")+"</b> "+str(records.Sample_smoothing) + "<br>" + \
        "<b>"+_("Value smoothing:")+"</b> "+str(records.Value_smoothing)

        self.ui.textBrowser.setHtml("<html><body>"+txt+"</html></body>")

#        print records.arrStatData






    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class AdditionalInfoWindow)
        self.close()
        ViewWindow.show()







###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAlarmSettingsWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor

        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
#        self.setWindowFlags(Qt.SplashScreen)
        self.ui = Ui_AlarmSettingsWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
        QtCore.QObject.connect(self.ui.cbExternalAlarm, QtCore.SIGNAL("toggled(bool)"), self.ExternalAlarm)

        self.setStyleSheet(data.UIStyle) #Add styles

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))
#            self.ui.bNo_Alarm.setIcon(QIcon(data.app_path+"img/"+"Alarm_off.png"))

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Set alarm")+")")
        txt=_("The Alarm will go off sometimes during the set time window (as soon as your movement is over the trigger level), but latest at the set alarm time.")
        self.ui.lblNote.setText(txt)
        self.ui.bSetAlarm.setText(_("Set alarm"))
        self.ui.label_7.setText(_("Alarm Time:"))
        self.ui.label_9.setText(_("Time Window:"))
        self.ui.label_8.setText(_("Hour:"))
        self.ui.label_10.setText(_("Minutes:"))
        self.ui.label_11.setText(_("Minutes:"))

        if(data.UseAlternativeAlarm==True): self.ui.cbExternalAlarm.setChecked(True)
        else: self.ui.cbExternalAlarm.setChecked(False)


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AlarmSettingsWindow)
        if(data.AlarmHour==-1 or data.AlarmMinute==-1):
            data.AlarmHour=7
            data.AlarmMinute=0
            data.AlarmTimeWindow=30
        if(data.debug==True): print "Alarm time:",  data.AlarmHour, ":",  data.AlarmMinute
        self.ui.lstAlarmHour.setCurrentRow(int(data.AlarmHour))
        self.ui.lstAlarmMinute.setCurrentRow(int(data.AlarmMinute)/5)
        self.ui.lstAlarmTimeWindow.setCurrentRow(int(data.AlarmTimeWindow)/5-1)




    def ExternalAlarm(self, state):
        if(state==True):
            if(data.AlternativeAlarm==""):
                self.ui.cbExternalAlarm.setChecked(False)
                QtGui.QMessageBox.warning(self, _("Alarm command missing"), _("Please first add an alternative alarm command in the settings menu!"), QtGui.QMessageBox.Ok)
            else:
                data.UseAlternativeAlarm=True
        else:
            data.UseAlternativeAlarm=False
        self.UpdateScreen()




    ###########################################################################
    ##
    ###########################################################################
    def SetAlarm(self): ##(Class AlarmSettingsWindow)
        data.UseAlarm=True
        SetRecordingWindow.ui.cbAlarmEnable.setChecked(True)
        data.AlarmHour=int(self.ui.lstAlarmHour.currentItem().text())
        data.AlarmMinute=int(self.ui.lstAlarmMinute.currentItem().text())
        data.AlarmTimeWindow=int(self.ui.lstAlarmTimeWindow.currentItem().text())
#        print data.AlarmHour, data.AlarmMinute, data.AlarmTimeWindow

        SetRecordingWindow.UpdateScreen()
        SetRecordingWindow.show()
        AlarmSettingsWindow.hide()



#    def No_Alarm(self): ##(Class AlarmSettingsWindow)
#        data.UseAlarm= False
#        SetRecordingWindow.UpdateScreen()
#        SetRecordingWindow.show()
#        AlarmSettingsWindow.hide()


    def closeEvent(self, event): ##(Class AlarmSettingsWindow)
#        records.AlarmIsSet=False
        data.UseAlarm=False
        self.close()
        SetRecordingWindow.show()














###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAskForKeepWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor

        if(data.DeviceIsN900==True): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_AskForKeepWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bstop_keepRecord, QtCore.SIGNAL("clicked()"), self.KeepRecord)
        QtCore.QObject.connect(self.ui.bstop_removeRecord, QtCore.SIGNAL("clicked()"), self.RemoveRecord)
        QtCore.QObject.connect(self.ui.bcontinue, QtCore.SIGNAL("clicked()"), self.ContinueRecording)

        self.setStyleSheet(data.UIStyle) #Add styles

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bstop_keepRecord.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
            self.ui.bstop_removeRecord.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bcontinue.setIcon(QIcon(data.app_path+"img/"+"krec.png"))

        self.setWindowTitle(_("SleepAnalyser")+" ("+_("Recording")+")")
        self.ui.bcontinue.setText(_("Continue recording"))
        self.ui.bstop_keepRecord.setText(_("Stop recording and keep the record"))
        self.ui.bstop_removeRecord.setText(_("Stop recording and delete the record"))



    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AskForKeepWindow)
#        s="Do you really want to stop recording?"
#        if(records.AlarmIsSet==True): s=s+"\nThis will also remove the set alarm!"
#        s=s+"\n\nAnd do you want to keep the record?"
#        self.ui.lblNote.setText(s)
        pass


    ###########################################################################
    ##
    ###########################################################################
    def KeepRecord(self): ##(Class AskForKeepWindow)
        print "Stop recording and keep record"
#        global ViewWindow
        StopMeasurement()
        print "*", strftime("%x %X"), "KeepRecord => DeleteAlarm" #for log file
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        data.LastRecord=data.RecordFilename #set to corrent file for viewwindow
        config.set('main', 'LastRecord', str(data.LastRecord))
        print "Show record in ViewWindow:",  data.LastRecord
        RecordWindow.hide()
        self.hide()
        StartWindow.ShowViewWindow()
        app.processEvents() #force ui update
        SwitchModes(False) #change back to normal mode and general profile if requested



    ###########################################################################
    ##
    ###########################################################################
    def RemoveRecord(self): ##(Class AskForKeepWindow)
        print "Stop recording and remove record"
#        global StartWindow,  RecordWindow
        StopMeasurement()
        print "*", strftime("%x %X"), "RemoveRecord => DeleteAlarm" #for log file
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        print "Delete RecordFile: "+data.RecordFilename
        os.remove(data.RecordFolder+data.RecordFilename)
        RecordWindow.hide()
        self.hide()
        StartWindow.show()
        app.processEvents() #force ui update
        SwitchModes(False) #change back to normal mode and general profile if requested



    ###########################################################################
    ##
    ###########################################################################
    def ContinueRecording(self): ##(Class AskForKeepWindow)
        print "continue recording"
        self.hide()
        RecordWindow.show()
        return


    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class AskForKeepWindow)
        event.ignore() #do not close
        self.KeepRecord() #interprete closing as "stop and keep file"

















###########################################################################
##
###########################################################################
class DisplayStatusDaemon:
    def __init__(self):
        if(data.DeviceIsN900==True): #We are on a N900
            loop = DBusQtMainLoop(set_as_default=True)
            system_bus = dbus.SystemBus()
            system_bus.add_signal_receiver(self.DisplayStatus,
                path='/com/nokia/mce/signal',
                signal_name='display_status_ind',
                dbus_interface='com.nokia.mce.signal')

            if(data.debug==True): print "DisplayStatusDaemon started"




    ###########################################################################
    ##
    ###########################################################################
    def DisplayStatus(self, *params):
        global RecordWindow
#        print "1", len(params)
#        print params
        tmp=str(params)
        tmp=string.split(tmp, "'")
#        print tmp
        if(tmp[1]=="off"): data.DisplayIsOn=False
        elif(tmp[1]=="on"):
            data.DisplayIsOn=True
#            if(data.run==True):
            if(RecordWindow.isHidden()==False):
                if(data.debug==True): print "Update RecordWindow screen & graph"
#                data.ChangesInGraph=True

                GenerateStatisticData()
                Generate_and_show_GraphBitmap(RecordWindow, True)
#                Generate_and_show_GraphBitmap(RecordWindow, False)
                RecordWindow.Canvas.update()
                RecordWindow.UpdateScreen()








###########################################################################
##
###########################################################################
class MyGraphCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)
        self.setGeometry(0, 0, 770, data.GraphBitMapHeight)

#        print "Create canvas with:",  770,  data.Ysize,  "(Size will change depending on graph len)"


    ###########################################################################
    ##
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False and data.run==True):
            if(data.debug==True): print "-> o Do not paint event (Graph), Display is off!"
            return
        else:
            if(data.debug==True): print "-> x paint event (Graph)"

        paint = QtGui.QPainter()
        paint.begin(self)
#        paint.setBackgroundMode(QtCore.Qt.TransparentMode)
        paint.drawPixmap(0, 0, data.GraphBitMap) #load graph from Bitmap

        paint.end()

        #Scroll graph to end, as there is the most interesting part
        if(data.ScrollToEnd==True):
            if(data.run==True): RecordWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            else:
                ViewWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            data.ScrollToEnd=False






###########################################################################
##
###########################################################################
class MyStatCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)
        self.setGeometry(406, 280, data.StatisticsBitMapLen,  data.StatisticsBitMapHeight)


    ###########################################################################
    ##
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False and data.run==True):
            if(data.debug==True): print "-> o Do not paint event (Statistics), Display is off!"
            return
        else:
            if(data.debug==True): print "-> x paint event (Statistics)"

        paint = QtGui.QPainter()
        paint.begin(self)
        paint.setBackgroundMode(QtCore.Qt.TransparentMode)
        paint.drawPixmap(0, 0, data.StatisticsBitMap) #load statistics from Bitmap
        paint.end()










###########################################################################
##
###########################################################################
def Generate_and_show_GraphBitmap(self, ShowSleepPattern=False):
    if(data.debug==True): print "Generate Graph Bitmap"

    Xscale=data.arrXscale[data.Xscale]
    data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  649) #628 minimum graph width
    data.GraphBitMapHeight=data.Ysize+34

#    try:
    data.GraphBitMap = QtGui.QPixmap(data.GraphBitMapLen, data.GraphBitMapHeight) #create bitmap for graph
    DrawGraph(data.GraphBitMap,  QtGui, ShowSleepPattern) # draw graph
#    except:
#        print "Error while creating the graph bitmap"

    self.Canvas.setGeometry(0, 0, data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas to graph

    self.ui.CanvasWidget.setMinimumSize(data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas widget to graph

    data.ScrollToEnd=True

    self.ui.CanvasWidget.hide()
    self.ui.CanvasWidget.show()
    self.UpdateScreen()






###########################################################################
##
###########################################################################
def GenerateStatisticsBitmap(self):
    if(data.debug==True): print "Generate Statistics Bitmap"

#    try: #TESTING
    data.StatisticsBitMap = QtGui.QPixmap(data.StatisticsBitMapLen,  data.StatisticsBitMapHeight) #create bitmap for graph
    DrawStatistic(data.StatisticsBitMap,  QtGui, QtCore) #draw statistics
#    except:
#        print "Error while creating the statistics bitmap"

















###########################################################################
main() # Now we can start it.
