#include <QtGui>
#include <QtCore/qstate.h>
#include <QtDBus/QtDBus>

#include "gamescene.h"
#include "boxwidget.h"
#include "maekerbutton.h"
#include "digitwidget.h"

#define SHIFT_LEFT 333
#define SHIFT_TOP 197
#define BOX_SIZE 40

#define STATE_FILE "gamestate.dat"
#define SCORE_FILE "scores.dat"
#if defined(Q_WS_MAEMO_5)
    #define CFG_DIR "/home/user/.skidstone/"
#else
    #define CFG_DIR
#endif

gameScene::gameScene(bool newGame, QObject *parent)
    : QGraphicsScene(parent)
{
    // Ui
    this->setSceneRect(-400, -240, 800, 480);
    QPixmap bgPix(":/img/game_back.png");
    QGraphicsItem *item = this->addPixmap(bgPix);
    item->setPos(-400,-240);

    qsrand(QTime(0,0,0).secsTo(QTime::currentTime())); // init randomize
    m_levelDat.resize(81);
    m_initState = true;
    m_score = 0;
    m_xBox = 0;
    m_prevDel = 1;

    m_datFileName = QString(CFG_DIR) + QString(STATE_FILE); // last game state file

    QPixmap buttonPix(":/img/back.png");
    MaekerButton *mb = new MaekerButton(buttonPix);
    this->addItem(mb);
    mb->setPos(358,-208);
    connect(mb,SIGNAL(clicked()),this,SLOT(backCommand()));

    QPixmap buttonPix1(":/img/level.png");
    item = this->addPixmap(buttonPix1);
    int ix = buttonPix1.width()/2;
    item->setZValue(5);
    item->setPos(250 - ix, 100);

    _wLevel = new DigitWidget();
    this->addItem(_wLevel);
    _wLevel->setZValue(5);
    _wLevel->setPos(250, 160);
    _wLevel->setNumber(1);

    loadScores();
    QPixmap buttonPix2(":/img/score.png");
    item = this->addPixmap(buttonPix2);
    item->setZValue(5);
    ix = buttonPix2.width()/2;
    item->setPos(250 - ix, -60);

    _wScore = new DigitWidget();
    this->addItem(_wScore);
    _wScore->setZValue(5);
    _wScore->setPos(250, 0);
    _wScore->setNumber(0);

    QPixmap buttonPix3(":/img/t_score.png");
    item = this->addPixmap(buttonPix3);
    item->setZValue(5);
    ix = buttonPix3.width()/2;
    item->setPos(250 - ix, -160);

    _wTScore = new DigitWidget();
    this->addItem(_wTScore);
    _wTScore->setZValue(5);
    _wTScore->setPos(250, -100);
    _wTScore->setNumber(m_topScore);

    #if defined(Q_WS_MAEMO_5)
        QPixmap buttonPix6(":/img/desk.png");
        mb = new MaekerButton(buttonPix6);
        this->addItem(mb);
        mb->setPos(-368,-211);
        connect(mb,SIGNAL(clicked()),this,SLOT(gotoDektop()));
    #endif

    if (newGame)
        gnerateField();
    else
        loadField();

    sceneInit();
}

// minimize application
void gameScene::gotoDektop()
{
//    emit hideApp();
#if defined(Q_WS_MAEMO_5)
    QDBusConnection c = QDBusConnection::sessionBus();
    QDBusMessage m = QDBusMessage::createSignal("/","com.nokia.hildon_desktop","exit_app_view");
    c.send(m);
#endif
}

void gameScene::loadScores()
{
    QString scoresFile = QString(CFG_DIR) + QString(SCORE_FILE);
    QFile file(scoresFile);
    if (file.exists())
    {
        QDataStream in(&file);
        if (file.open(QIODevice::ReadOnly))
        {
            in >> m_topScore;
        }
        file.close();
    }
    else
      m_topScore = 10000;
}

void gameScene::saveScores()
{
    QString scoresFile = QString(CFG_DIR) + QString(SCORE_FILE);
    QFile file(scoresFile);
    QDataStream out(&file);
    if (file.open(QIODevice::WriteOnly))
    {
        out << m_topScore;
    }
    file.close();
}

void gameScene::gnerateField()
{
    currentLevel = 1;
    readLevel();     // read 1st level data

//    m_colorNumber = 7; //TODO comment this line

    for (int i = 0; i < 121; ++i) {
        BoxWidget *item;

        // corner tiles
        if ((i == 0) || (i == 10) || (i == 110) || (i == 120))
        {
            item = new BoxWidget(Qt::white);
            item->setWayTile(true);
        }
        // box tiles
        else if ((i < 10) || (i > 110) || (i%11 == 0) || ((i-10)%11 == 0))
        {
            item = new BoxWidget(qrand() % m_colorNumber + 1);
            item->setBoxTile(true);
            item->setColorNumber(m_colorNumber);
        }
        // field tiles
        else
        {
            item = new BoxWidget(Qt::white);
            item->setWayTile(false);
        }
        m_items << item;
    }
}

void gameScene::loadField()
{
    QFile file(m_datFileName);
    QDataStream in(&file);
    BoxWidget *item;
    if (file.open(QIODevice::ReadOnly))
    {
        in >> currentLevel;
        _wLevel->setNumber(currentLevel);
        in >> m_colorNumber;
        in >> m_score;
        _wScore->setNumber(m_score);
        for (int i = 0; i < 121; ++i) {
            item = new BoxWidget();
            in >> item;
            m_items << item;
        }
    }
    file.close();
}

void gameScene::sceneInit()
{
    // main state
    QState *s1 = new QState(QState::ParallelStates);

    for (int i = 0; i < 121; ++i) {
        BoxWidget *item = m_items.at(i);

        // corner tiles
        if ((i == 0) || (i == 10) || (i == 110) || (i == 120))
        {
            item->setOpacity(0);
        }
        // box tiles
        else if ((i < 10) || (i > 110) || (i%11 == 0) || ((i-10)%11 == 0))
        {
            QState *c1 = new QState(s1);
            QState *c11 = new QState(c1); // initial state
            QState *c12 = new QState(c1); // state on field
            QState *c121 = new QState(c1); // state on field
            QState *c13 = new QState(c1); // returning state
            c1->setInitialState(c11);
            c11->assignProperty(item,"opacity",qreal(1));
            c11->assignProperty(item,"pos",QPointF((i % 11) * BOX_SIZE - SHIFT_LEFT, (i/11)*BOX_SIZE - SHIFT_TOP));
            c12->assignProperty(item,"opacity",qreal(0));
            c121->assignProperty(item,"opacity",qreal(0));
            c13->assignProperty(item,"opacity",qreal(1));
//            c13->assignProperty(item,"color",Qt::transparent);
            c13->assignProperty(item,"color",0);

            if (i < 10) // first line
            {
                c13->assignProperty(item,"pos",QPointF((i % 11) * BOX_SIZE - SHIFT_LEFT,
                                                       (i/11)*BOX_SIZE - SHIFT_TOP - BOX_SIZE));
            }
            if (i > 110) // last line
            {
                c13->assignProperty(item,"pos",QPointF((i % 11) * BOX_SIZE - SHIFT_LEFT,
                                                       (i/11)*BOX_SIZE - SHIFT_TOP + BOX_SIZE));
            }
            if (i%11 == 0) // first row
            {
                c13->assignProperty(item,"pos",QPointF((i % 11) * BOX_SIZE - SHIFT_LEFT - BOX_SIZE,
                                                       (i/11)*BOX_SIZE - SHIFT_TOP));
            }
            if ((i-10)%11 == 0) // last row
            {
                c13->assignProperty(item,"pos",QPointF((i % 11) * BOX_SIZE - SHIFT_LEFT + BOX_SIZE,
                                                       (i/11)*BOX_SIZE - SHIFT_TOP));;
            }

            QAbstractTransition *trans = c11->addTransition(item, SIGNAL(pressed()), c121);
            trans = c121->addTransition(c121, SIGNAL(propertiesAssigned()), c12);
            QPropertyAnimation *anim = new QPropertyAnimation(item, "opacity");
            anim->setDuration(300);
            trans->addAnimation(anim);
            trans = c12->addTransition(c12, SIGNAL(propertiesAssigned()), c13);
            trans = c13->addTransition(c13, SIGNAL(propertiesAssigned()), c11);
            anim = new QPropertyAnimation(item, "pos");
            anim->setEasingCurve(QEasingCurve::OutBounce);
            anim->setDuration(600); // was 750
            trans->addAnimation(anim);

            connect(item, SIGNAL(fired(QPointF, QPointF,int)), this, SLOT(boxFired(QPointF, QPointF,int)));
        }
        // field tiles
        else
        {
            QState *c1 = new QState(s1);
            QState *c11 = new QState(c1); // tile state
            c11->assignProperty(item,"opacity",qreal(1));

            QState *c12 = new QState(c1); // box state
            c12->assignProperty(item,"opacity",qreal(1));

            QState *c13 = new QState(c1); // way state
            c13->assignProperty(item,"opacity",qreal(0));

            item->setStateBox(c12);
            item->setStateBack(c11);
            item->setStateWay(c13);

            connect(c11,SIGNAL(propertiesAssigned()),item,SLOT(transitionFinished1()));
            connect(c12,SIGNAL(propertiesAssigned()),item,SLOT(transitionFinished2()));
            connect(c13,SIGNAL(propertiesAssigned()),item,SLOT(transitionFinished3()));

            if (item->isBoxTile())
                c1->setInitialState(c12);
            else
                c1->setInitialState(c11);
        }

        this->addItem(item);
        if (item->isBoxTile())
            item->setZValue(2);
        else
            item->setZValue(1);
        item->setPos((i % 11) * BOX_SIZE - SHIFT_LEFT, (i/11)*BOX_SIZE - SHIFT_TOP);
    }

    m_timer.setSingleShot(true);
    m_timerColor.setSingleShot(true);
    m_timerField.setSingleShot(true);
    m_timerFieldCheck.setSingleShot(true);
    m_timerOver.setSingleShot(true);
    connect(&m_timerOver, SIGNAL(timeout()), this,SLOT(signalGameOver()));
    connect(&m_timer, SIGNAL(timeout()), this,SLOT(cleanField()));
    connect(&m_timerColor, SIGNAL(timeout()), this,SLOT(checkColor()));
    connect(&m_timerField, SIGNAL(timeout()), this,SLOT(setFieldBoxes2()));
    connect(&m_timerFieldCheck, SIGNAL(timeout()), this,SLOT(checkBuddyColors()));
    connect(s1,SIGNAL(propertiesAssigned()),this,SLOT(setFieldBoxes()));

    // init moving block
    m_movingBox = new BoxWidget(Qt::black);
    m_movingBox->setBoxTile(true);
    QState *c1 = new QState(s1);
    m_mb1 = new QState(c1);
    m_mb2 = new QState(c1);
    m_mb3 = new QState(c1);
    m_mb1->assignProperty(m_movingBox,"opacity",qreal(0));
    m_mb2->assignProperty(m_movingBox,"opacity",qreal(1));
    m_mb3->assignProperty(m_movingBox,"opacity",qreal(1));
    QAbstractTransition *trans = m_mb1->addTransition(this, SIGNAL(moveTile()), m_mb2);
    trans = m_mb2->addTransition(m_mb2, SIGNAL(propertiesAssigned()), m_mb3);
    QPropertyAnimation *anim = new QPropertyAnimation(m_movingBox, "pos");
    anim->setDuration(600); // was 750
    anim->setEasingCurve(QEasingCurve::InOutCubic);
    trans->addAnimation(anim);
    trans = m_mb3->addTransition(m_mb3, SIGNAL(propertiesAssigned()), m_mb1);
    connect(m_mb3, SIGNAL(propertiesAssigned()),this,SLOT(tileTransform()));
    c1->setInitialState(m_mb1);
    this->addItem(m_movingBox);
    m_movingBox->setZValue(3);

    // state machine
    m_states.addState(s1);
    m_states.setInitialState(s1);
    m_states.start();
}

// calculate coord in matrix using Box Coordinate
void gameScene::boxFired(QPointF aPos, QPointF aPoint, int aColor)
{
    disableMove();
    m_xBox = (aPoint.x() + SHIFT_LEFT) / BOX_SIZE +
             (aPoint.y() + SHIFT_TOP) / BOX_SIZE * 11;
    m_xColor = aColor;
    m_movingBox->setIColor(m_xColor);
    m_mb2->assignProperty(m_movingBox,"pos",aPos);
    m_mb3->assignProperty(m_movingBox,"pos",aPoint);
    emit moveTile();
}

void gameScene::tile2boxAssign(BoxWidget *aTile)
{
    QState *s1 = aTile->stateWay();
    QState *s2 = aTile->stateBox();
    s2->assignProperty(aTile,"color",m_xColor);
    QAbstractTransition *trans = s1->addTransition(this, SIGNAL(tile2box()), s2);
    aTile->setTransition(trans,1,2);
}

// activate tile on a field
void gameScene::tileTransform()
{
    if (m_xBox > 0)
    {
        BoxWidget *tile = m_items.at(m_xBox);
        tile2boxAssign(tile);
    }
    emit tile2box();
    if (m_xBox > 0)
        m_timerColor.start(20); // was 50
    else
    {
        m_initState = false;
        m_timerFieldCheck.start(20); //was 50 // goto check buddy colors
    }
 }

//bool gameScene::checkBuddy(int aPos, QColor aColor)
bool gameScene::checkBuddy(int aPos, int aColor)
{
    BoxWidget *byddyTile = m_items.at(aPos);
    if (byddyTile->isBoxTile())
    {
        int buddyColor = byddyTile->iColor();
        if (aColor == buddyColor)
        {
            QState *s1 = byddyTile->stateBox();
            QState *s2 = byddyTile->stateWay();
            QAbstractTransition *trans = s1->addTransition(this, SIGNAL(box2tile()), s2);
            QPropertyAnimation *anim = new QPropertyAnimation(byddyTile, "opacity");
            anim->setDuration(500); // was 750
            trans->addAnimation(anim);
            byddyTile->setTransition(trans,2,1);
            return true;
        }
    }
    return false;
}

// check field for possible coincidences
void gameScene::checkColor()
{
    m_initState = false;
    bool coincidence = false;
    BoxWidget *boxTile = m_items.at(m_xBox);
    int boxColor = boxTile->iColor();
    int ix = m_xBox - 11;
    int dels = 0; // count deleted boxes for score
    if (ix > 10) //check upper box
    {
        if (checkBuddy(ix,boxColor))
        {
            coincidence = true;
            dels ++;
        }
    }

    ix = m_xBox - 1;
    if (ix%11 != 0 ) // check left buddy
    {
        if (checkBuddy(ix,boxColor))
        {
            coincidence = true;
            dels ++;
        }
    }

    ix = m_xBox + 1;
    if ((ix-10)%11 != 0 ) // check right buddy
    {
        if (checkBuddy(ix,boxColor))
        {
            coincidence = true;
            dels ++;
        }
    }

    ix = m_xBox + 11;
    if (ix < 110 ) // check bottom buddy
    {
        if (checkBuddy(ix,boxColor))
        {
            coincidence = true;
            dels ++;
        }
    }

    if (coincidence)
    {
        dels ++;
        QState *s1 = boxTile->stateBox();
        QState *s2 = boxTile->stateWay();
        QAbstractTransition *trans = s1->addTransition(this, SIGNAL(box2tile()), s2);
        QPropertyAnimation *anim = new QPropertyAnimation(boxTile, "opacity");
        anim->setDuration(500); // was 750
        trans->addAnimation(anim);
        boxTile->setTransition(trans,2,1);
        emit box2tile();
        m_timer.start(520);

        m_score += dels * 5 * m_prevDel;
        _wScore->setNumber(m_score);
        m_prevDel ++;
    }
    else
    {
        cleanField();
        m_prevDel = 1;
    }
}

// check field for possible moves
void gameScene::cleanField()
{
    if (m_initState)
        return;
    for (int i = 1; i < 10; i++)
        for (int j = 1; j < 10; j++)
        {
            BoxWidget *tile = m_items.at(i+j*11);
            if (!(tile->isBoxTile()))// && (tile->isWayTile()))
            {
                tile->setWayTile(false);
                QState *s1 = tile->stateWay();
                QState *s2 = tile->stateBack();
                QAbstractTransition *trans = s1->addTransition(this, SIGNAL(cleantile()), s2);
                tile->setTransition(trans,1,3);
            }
    }
    emit cleantile();
    checkField();
}

// check field for possible moves
void gameScene::checkField()
{
    bool emptyField = true;
    bool final = true;
    for (int i=1;i<120;++i)
    {
        int dx = 0;
        int dy = 0;
        int x = i%11;
        int y = i/11;

        BoxWidget *tile = m_items.at(i);

        if (((i < 10) && (i > 0))
            || ((i > 110) && (i < 120))
            || ((x == 0) && (i != 110))
            || ((x == 10) && (i !=10)))
        {
            if ((i < 10) && (i > 0)) // first line
                dy = 1;
            if ((i > 110) && (i < 120)) // last line
                dy = -1;
            if ((x == 0) && (i != 110)) // first row
                dx = 1;
            if ((x == 10) && (i !=10)) // last row
                dx = -1;

            bool isBox = false;
            int j = 1;
            while ((j<10) && (!isBox))
            {
                x += dx;
                y += dy;
                BoxWidget *tile1 = m_items.at(x+y*11);
                isBox = tile1->isBoxTile();
                if ((isBox) && (j == 1))
                {
                    emptyField = false;
                    isBox = false;
                    j=10;
                }
                j++;
            }
            if (isBox) // found box Tile
            {
                final = false;
                emptyField = false;
                tile->setClickable(true);
                QPointF c12Point;
                c12Point = QPointF((x-2*dx) * BOX_SIZE - (SHIFT_LEFT - BOX_SIZE*dx),
                                       (y-2*dy)*BOX_SIZE - (SHIFT_TOP - BOX_SIZE*dy));
                tile->setPoint(c12Point);
                // paint way tiles and make Box clickable
                x = i%11;
                y = i/11;
                for (int k = 1; k < j-1; k++)
                {
                    x += dx;
                    y += dy;
                    BoxWidget *tile = m_items.at(x+y*11);
                    QState *s1 = tile->stateBack();
                    QState *s2 = tile->stateWay();
                    QAbstractTransition *trans = s1->addTransition(this, SIGNAL(tile2way()), s2);
                    tile->setTransition(trans,3,1);
                }
            }
            else
                tile->setClickable(false);
        }
    }

    if (final && !emptyField) // The END
    {
        startGameOver();
    }
    else
    {

    emit tile2way();
    if (emptyField) // next level
    {
        currentLevel ++;
        m_score +=500;
        _wScore->setNumber(m_score);
        if (currentLevel > 50)
            currentLevel = 1;
        renewField();
    }
    m_initState = true;

    saveState();
}
}

void gameScene::startGameOver()
{
    //TODO make final message (and maybe let enter user name)

    QPixmap pixmap (":/img/the_end.png");
    QGraphicsItem *item = this->addPixmap(pixmap);
    item->setZValue(20);
    item->setPos(-SHIFT_LEFT+BOX_SIZE/2,-SHIFT_TOP+BOX_SIZE/2);
    item->setOpacity(0.85);

    QFile file(m_datFileName);
    file.remove();
    // show message

    if (m_topScore < m_score)
    {
        m_topScore = m_score;
        saveScores();
    }

    m_timerOver.start(4000);
}

void gameScene::signalGameOver()
{
    emit gameOver();
}

// disable moves while one is active
void gameScene::disableMove()
{
    m_initState = false;
    for (int i=1;i<120;++i)
    {
        int x = i%11;
        if (((i < 10) && (i > 0))
            || ((i > 110) && (i < 120))
            || ((x == 0) && (i != 110))
            || ((x == 10) && (i !=10)))
        {
            BoxWidget *tile = m_items.at(i);
            tile->setClickable(false);
        }
    }
}

void gameScene::renewField()
{
    m_prevDel = 1;
    readLevel();
    for (int i=1;i<120;++i)
    {
        int x = i%11;
        if (((i < 10) && (i > 0))
            || ((i > 110) && (i < 120))
            || ((x == 0) && (i != 110))
            || ((x == 10) && (i !=10)))
        {
            BoxWidget *tile = m_items.at(i);
            tile->setColorNumber(m_colorNumber);
            tile->setIColor(0);
        }
    }
    setFieldBoxes();
}

void gameScene::setFieldBoxes()
{
    for (int i=1;i<10;++i)
        for (int j = 1; j < 10; j++)
    {
        int x = j * 11 + i;
        int y = (j-1) * 9 + i-1;
        if (m_levelDat.testBit(y))
        {
            BoxWidget *tile = m_items.at(x);
            QState *s1 = tile->stateBack();
            QState *s2 = tile->stateWay();
            QAbstractTransition *trans = s1->addTransition(this, SIGNAL(tile2way()), s2);
            tile->setTransition(trans,3,1);
        }
    }
    emit tile2way();
    m_timerField.start(20); // was 50
}


void gameScene::setFieldBoxes2()
{
    m_xBox = 0;
    for (int i=1;i<10;++i)
        for (int j = 1; j < 10; j++)
    {
        int x = j * 11 + i;
        int y = (j-1) * 9 + i-1;
        if (m_levelDat.testBit(y))
        {
            BoxWidget *tile = m_items.at(x);
            m_xColor = qrand() % m_colorNumber + 1;
            tile2boxAssign(tile);
        }
    }
    tileTransform();
}

void gameScene::readLevel()
{
    _wLevel->setNumber(currentLevel);

    QString fileName = QString(":/levels/%1.dat").arg(QString::number(currentLevel));
    QFile file(fileName);
    QDataStream in(&file);
    if (file.open(QIODevice::ReadOnly))
    {
         in >> m_levelDat;
         in >> m_colorNumber;
    }
    file.close();
}

void gameScene::checkBuddyColors()
{
    for (int i=1;i<10;++i)
        for (int j = 1; j < 10; j++)
    {
        int x = j * 11 + i;
        int x1;
        int y1;
        BoxWidget *tile = m_items.at(x);
        if (tile->isBoxTile())
        {
            int boxColor = tile->iColor();
            int boxRight;
            BoxWidget *tileR;
            int boxDown;
            BoxWidget *tileD;
            x1 = x + 1;
            if (x1%11 < 10) // check right buddy
            {
                tileR = m_items.at(x1);
                if (tileR->isBoxTile())
                {
                    boxRight = tileR->iColor();
                    while (boxColor == boxRight) // need to change color
                    {
                       boxRight = 0;
                    }
                    tileR->setIColor(boxRight);
                }
            }

            y1 = x + 11;
            if (y1 < 110) // check bottom buddy
            {
                tileD = m_items.at(y1);
                if (tileD->isBoxTile())
                {
                    boxDown = tileD->iColor();
                    while (boxColor == boxDown) // need to change color
                    {
                       boxDown = 0;
                    }
                    tileD->setIColor(boxDown);
                }
            }
        }
    }
    m_timer.start(20); //was 50 // goto CleanField
}

void gameScene::backCommand()
{
    emit backButtonClicked();
}

void gameScene::saveState()
{
    QFile file(m_datFileName);
    QDataStream out(&file);
    BoxWidget *item;
    if (file.open(QIODevice::WriteOnly))
    {
        out << currentLevel;
        out << m_colorNumber;
        out << m_score;
        for (int i = 0; i < 121; ++i) {
            item = m_items.at(i);
            out << item;
        }
    }
    file.close();
}

int gameScene::score()
{
    return m_score;
}

int gameScene::topScore()
{
    return m_topScore;
}

void gameScene::setTopScore(int aScore)
{
    m_topScore = aScore;
    saveScores();
}

void gameScene::a1Updated() // temp test function
{
//    aBox->setBoxTile(false);
  //  aBox->setWayTile(false);
//            QMessageBox::critical(0, qApp->tr("O"),
  //              qApp->tr("fired"), QMessageBox::Cancel);

//    a1->setPos(a1->pos().x()-10, -100);
//    a1->setProperty("pos", QPointF(0,0));
}

