/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtTest/QtTest>

#include "map/mapcommon.h"

#include "map/gpslocationitem.h"

class TestGPSLocationItem: public QObject
{
    Q_OBJECT

private:
    QRectF boundingRect(qreal accuracy, qreal sceneResolution);

private slots:
    void constructor();
    void childPixmap();
    void enable();
    void updateItem();
    void updateItem_data();
};

QRectF TestGPSLocationItem::boundingRect(qreal accuracy, qreal sceneResolution)
{
    const qreal PEN_WIDTH = 1;

    qreal radius = accuracy / sceneResolution;

    return QRectF(-radius - PEN_WIDTH / 2,
                  -radius - PEN_WIDTH / 2,
                  2 * radius + PEN_WIDTH,
                  2 * radius + PEN_WIDTH);
}

void TestGPSLocationItem::constructor()
{
    GPSLocationItem item;

    // position should be UNDEFINED
    QCOMPARE(item.pos(), QPointF(UNDEFINED, UNDEFINED));

    // zValue should be set
    QCOMPARE(item.zValue(), static_cast<qreal>(GPSLocationItemZValue));

    // by default the item should be hidden
    QVERIFY(item.isVisible() == false);
}

/**
* The actual GPS location red spot is in a child QGraphicsPixmapItem
*/
void TestGPSLocationItem::childPixmap()
{
    GPSLocationItem item;

    // there should be one QGraphicsPixmapItem child item
    QList<QGraphicsItem *> childs = item.childItems();
    QVERIFY(childs.count() == 1);
    QGraphicsPixmapItem *childPixmap = dynamic_cast<QGraphicsPixmapItem *>(childs.at(0));
    QVERIFY(childPixmap);

    // pixmap should be set
    QVERIFY(!childPixmap->pixmap().isNull());

    // pixmap offset should be set based on pixmap
    QCOMPARE(childPixmap->offset(),
             QPointF(-childPixmap->pixmap().width() / 2, -childPixmap->pixmap().height() / 2));

    // ItemIgnoresTransformations flag should be set
    QVERIFY(childPixmap->flags() & QGraphicsItem::ItemIgnoresTransformations);
}

/**
* When setEnabled(true) is called, the item should become visible only after the first location
* update from the GPS is received
*/
void TestGPSLocationItem::enable()
{
    const qreal SCENE_RESOLUTION = 0.252006;
    const qreal ACCURACY_100M = 100;
    const QPointF POSITION_IN_SCENE = QPointF(1000, 1000);

    // create item and update it's position so pixmap is set
    GPSLocationItem item;

    // item should not be visible because it's location is not yet set
    QCOMPARE(item.isVisible(), false);

    // setting the location when item is not set enabled should not make the item visible
    item.updateItem(POSITION_IN_SCENE, ACCURACY_100M, SCENE_RESOLUTION);
    QCOMPARE(item.isVisible(), false);

    // setting the item enabled should not make the item visible before first location update
    // is received
    item.setEnabled(true);
    QCOMPARE(item.isVisible(), false);

    // item should become visible after the first location update
    item.updateItem(POSITION_IN_SCENE, ACCURACY_100M, SCENE_RESOLUTION);
    QCOMPARE(item.isVisible(), true);
}

/**
* Item's position and accuracy ring diameter should be updated. Because accuracy ring diameter
* is changed, the bounding rect should also be changed.
*/
void TestGPSLocationItem::updateItem()
{
    QFETCH(QPointF, position);
    QFETCH(qreal, accuracy);
    QFETCH(qreal, sceneResolution);
    QFETCH(QPointF, expectedPosition);
    QFETCH(QRectF, expectedRect);

    GPSLocationItem item;
    item.updateItem(position, accuracy, sceneResolution);

    QCOMPARE(item.pos(), expectedPosition);
    QCOMPARE(item.boundingRect(), expectedRect);
}

void TestGPSLocationItem::updateItem_data()
{
    const QPointF POSITION1 = QPointF(1000, 1000);
    const QPointF POSITION2 = QPointF(2000, 4000);

    const qreal ACCURACY_100M = 100;
    const qreal ACCURACY_15M = 15;

    const qreal SCENE_RESOLUTION1 = 0.250000;
    const qreal SCENE_RESOLUTION2 = 0.500000;

    QTest::addColumn<QPointF>("position");
    QTest::addColumn<qreal>("accuracy");
    QTest::addColumn<qreal>("sceneResolution");
    QTest::addColumn<QPointF>("expectedPosition");
    QTest::addColumn<QRectF>("expectedRect");

    QTest::newRow("1st test") << POSITION1
                              << ACCURACY_100M
                              << SCENE_RESOLUTION1
                              << POSITION1
                              << boundingRect(ACCURACY_100M, SCENE_RESOLUTION1);

    QTest::newRow("change only position") << POSITION2
                                          << ACCURACY_100M
                                          << SCENE_RESOLUTION1
                                          << POSITION2
                                          << boundingRect(ACCURACY_100M, SCENE_RESOLUTION1);

    QTest::newRow("change only accuracy") << POSITION1
                                          << ACCURACY_15M
                                          << SCENE_RESOLUTION1
                                          << POSITION1
                                          << boundingRect(ACCURACY_15M, SCENE_RESOLUTION1);

    QTest::newRow("change only scene resolution") << POSITION1
                                                  << ACCURACY_100M
                                                  << SCENE_RESOLUTION2
                                                  << POSITION1
                                                  << boundingRect(ACCURACY_100M, SCENE_RESOLUTION2);
}

QTEST_MAIN(TestGPSLocationItem)

#include "testgpslocationitem.moc"
