/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef USER_H
#define USER_H

#include <QPixmap>
#include <QString>
#include <QUrl>

#include "coordinates/geocoordinate.h"

/**
* @brief Class to store user information (applies to friends also)
*
* @author Henri Lampela
* @class User user.h "user/user.h"
*/
class User
{
public:

    /**
    * @brief Constructor, initializes member data
    *
    */
    User(const QString &address, const GeoCoordinate &coordinates, const QString &name,
         const QString &note, const QUrl &imageUrl, const QString &timestamp,
         const bool &type, const QString &userId, const QString &units = 0,
         const double &value = 0);

    /**
    * @brief Default constructor, initializes member data as NULL/0
    *
    */
    User();

    /*******************************************************************************
     * MEMBER FUNCTIONS AND SLOTS
     ******************************************************************************/

    /**
    * @brief Set address
    *
    * @param address street address
    */
    void setAddress(const QString &address);

    /**
    * @brief Set coordinates ( x = lon, y = lat )
    *
    * @param coordinates coordinates
    */
    void setCoordinates(const GeoCoordinate &coordinates);

    /**
    * @brief Set distance
    *
    * @param value distance
    * @param units unit type
    */
    void setDistance(const double &value, const QString &units);

    /**
    * @brief Set note
    *
    * @param note note/status message
    */
    void setNote(const QString &note);

    /**
    * @brief Set profile image
    *
    * @param image Image
    */
    void setProfileImage(const QPixmap &image);

    /**
    * @brief Set download address for profile image
    *
    * @param imageUrl image url of big image
    */
    void setProfileImageUrl(const QUrl &imageUrl);

    /**
    * @brief Set timestamp for last status update, timestamp is in literal mode
    *
    * @param timestamp timestamp
    */
    void setTimestamp(const QString &timestamp);

    /**
    * @brief Get address
    *
    * @return QString address
    */
    const QString &address() const;

    /**
    * @brief Get coordinates
    *
    * @return GeoCoordinate coordinates
    */
    const GeoCoordinate &coordinates() const;

    /**
    * @brief Get distance and units
    *
    * @param value distance
    * @param units unit type
    */
    void distance(double &value, QString &units) const;

    /**
    * @brief Get name
    *
    * @return QString profile name
    */
    const QString &name() const;

    /**
    * @brief Get note/status message
    *
    * @return QString note
    */
    const QString &note() const;

    /**
    * @brief Get profile image
    *
    * @return QPixmap image
    */
    const QPixmap &profileImage() const;

    /**
    * @brief Get download address for profile image
    *
    * @return QString url
    */
    const QUrl &profileImageUrl() const;

    /**
    * @brief Get timestamp of last status update
    *
    * @return QString timestamp
    */
    const QString &timestamp() const;

    /**
    * @brief Get user type
    *
    * @return bool user type (true = user, false = friend)
    */
    const bool &type() const;

    /**
    * @brief Get userId
    *
    * @return QString userId
    */
    const QString &userId() const;

    /*******************************************************************************
     * DATA MEMBERS
     ******************************************************************************/

private:
    QString m_address; ///< placeholder for address information
    GeoCoordinate m_coordinates; ///< placeholder for coordinates
    QString m_name; ///< placeholder for name
    QString m_note; ///< placeholder for note
    QUrl m_profileImageUrl; ///< placeholder for image url
    QString m_timestamp; ///< placeholer for timestamp
    bool m_type; ///< placeholder for user type
    QString m_units; ///< placeholder for distance unit type
    QString m_userId; ///< placeholder for userId
    double m_value; ///< placeholder for distance value
    QPixmap m_profileImage; ///< placeholder for image
};


#endif // USER_H
