/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef TABBEDPANEL_H
#define TABBEDPANEL_H

#include <QWidget>

#include "panelcommon.h"

class QPropertyAnimation;
class QSignalTransition;
class QState;
class QStateMachine;

class ListItemContextButtonBar;
class PanelBar;
class PanelContentStack;
class PanelContextButtonBar;
class PanelTabBar;

/**
 * @brief Class for tabbed panels
 *
 * @author Kaj Wallin - kaj.wallin (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
 */
class TabbedPanel : public QWidget
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     *
     * @param parent Parent
     */
    TabbedPanel(QWidget *parent = 0);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
     * @brief Adds a tab to the panel
     *
     * Adds a tab with the given widget and icon into the tabbed panel and returns the index of the
     * inserted tab.
     *
     * @param widget Widget to be added into the tab
     * @param icon Icon of the tab
     */
    int addTab(QWidget *widget, const QIcon& icon);

    /**
     * @brief Inserts a tab to the panel
     *
     * Inserts a tab with the given widget and icon into the tabbed panel at the specified index,
     * and returns the index of the inserted tab.
     *
     * If index is out of range, the tab is simply appended. Otherwise it is inserted at the
     * specified position.
     *
     * @param index Index of the tab
     * @param widget Widget to be inserted into the tab
     * @param icon Icon of the tab
     */
    int insertTab(int index, QWidget *widget, const QIcon& icon);

    /**
     * @brief Removes a tab from the panel
     *
     * Removes a tab and its widget from the panel at index position. The widget itself is not
     * deleted.
     *
     * @param index Index of the tab
     */
    void removeTab(int index);

    /**
    * @brief Sets tabs enabled.
    *
    * If disabled tab is currently selected, panel will be closed also
    * @param tabIndexes tab indexes to set
    * @param enabled true if should be enabled, false otherwise
    */
    void setTabsEnabled(const QList<int> &tabIndexes, bool enabled);

public slots:
    /**
     * @brief Slot that closes the panel
     */
    void closePanel();

    /**
     * @brief Slot that opens the panel
     *
     * If widget pointer is provided the corresponding tab is also set active
     *
     * @param widget Widget
     */
    void openPanel(QWidget *widget = 0);

    /**
     * @brief Slot to redraw the panel after window resize event
     *
     * @param size Size of the new window
     */
    void resizePanel(const QSize &size);

private slots:
    /**
     * @brief Calculates mask for tabbed panel
     *
     * Mask is constructed from PanelTabBar, PanelContextButtonBar and panel content sizes
     */
    void calculateMask();

    /**
     * @brief Repositions context button bar
     */
    void repositionContextButtonBar();

    /**
     * @brief Sets the panel at current index active
     *
     * @param index Index of the panel
     */
    void setCurrentIndex(int index);

    /**
     * @brief Internal slot used to set the panel state
     */
    void stateChanged();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
     * @brief This signal is emitted whenever the current tab page changes
     *
     * @param index Index of the new tab page
     */
    void currentChanged(int index);

    /**
     * @brief Emitted when there is a change in list item selection
     *
     * @param itemIsSelected True if any item is selected.
     */
    void listItemSelectionChanged(bool itemIsSelected);

    /**
     * @brief Signal that is sent when panel is closed
     *
     * @sa openPanel
     * @sa closePanel
     */
    void panelClosed();

    /**
     * @brief Signal that is sent when panel is opened
     *
     * @sa openPanel
     * @sa closePanel
     */
    void panelOpened();

    /**
     * @brief Signal that is sent when the panel state must be changed
     *
     * @sa openPanel
     * @sa closePanel
     */
    void toggleState();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_open;                ///< Current state of the panel
    bool m_closeRequestPending; ///< Indicates wheater the panel is waiting to be closed

    QState *m_stateClosed;      ///< State of the closed panel
    QState *m_stateOpened;      ///< State of the opened panel

    ListItemContextButtonBar *m_itemContextButtonBar;   ///< Widget for list item context button bar
    PanelBar *m_panelBar;                               ///< Widget for panel bar
    PanelContentStack *m_panelContentStack;             ///< Stack for panel widgets
    PanelContextButtonBar * m_panelContextButtonBar;    ///< Widget for panel context button bar
    PanelTabBar *m_panelTabBar;                         ///< Widget for panel tab bar
};

#endif // TABBEDPANEL_H
