/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "coordinates/geocoordinate.h"
#include "extendedlistitemdelegate.h"
#include "imagebutton.h"
#include "panelcommon.h"
#include "routewaypointlistitem.h"
#include "routewaypointlistview.h"
#include "routing/location.h"
#include "routing/route.h"

#include "routingpanel.h"

RoutingPanel::RoutingPanel(QWidget *parent)
    : PanelBase(parent)
{
    qDebug() << __PRETTY_FUNCTION__;

    // --- LAYOUT & ROUTING INSTRUCTIONS VIEW ---
    m_routeWaypointListView = new RouteWaypointListView(this);
    m_routeWaypointListView->setItemDelegate(new ExtendedListItemDelegate(this));

    connect(m_routeWaypointListView, SIGNAL(routeWaypointItemClicked(GeoCoordinate)),
            this, SIGNAL(routeWaypointItemClicked(GeoCoordinate)));

    connect(m_routeWaypointListView, SIGNAL(listItemSelectionChanged()),
            this, SLOT(onListItemSelectionChanged()));

    QVBoxLayout *panelLayout = new QVBoxLayout;
    panelLayout->setContentsMargins(PANEL_MARGIN_LEFT, PANEL_MARGIN_TOP,
                                    PANEL_MARGIN_RIGHT, PANEL_MARGIN_BOTTOM);

    m_noRouteLabel = new QLabel();
    m_noRouteLabel->setText("No route");
    m_noRouteLabel->setAlignment(Qt::AlignCenter);

    QPalette noRoutePalette = palette();
    noRoutePalette.setColor(QPalette::Foreground, Qt::white);
    m_noRouteLabel->setPalette(noRoutePalette);

    panelLayout->addWidget(m_routeWaypointListView);
    panelLayout->addWidget(m_noRouteLabel, Qt::AlignCenter);
    setLayout(panelLayout);

    // --- CONTEXT BUTTONS ---
    ImageButton *routeToCursorButton = new ImageButton(":res/images/route_to_cursor.png",
                                                       ":res/images/route_to_cursor_s.png",
                                                       ":res/images/route_to_cursor_d.png", this);
    connect(routeToCursorButton, SIGNAL(clicked()),
            this, SIGNAL(routeToCursor()));

    m_clearRouteButton = new ImageButton(":/res/images/clear_route.png",
                                         ":/res/images/clear_route_s.png",
                                         ":/res/images/clear_route_d.png", this);
    connect(m_clearRouteButton, SIGNAL(clicked()),
            this, SLOT(clearRouteButtonClicked()));
    m_clearRouteButton->setDisabled(true);

    m_genericButtonsLayout->addWidget(routeToCursorButton);
    m_genericButtonsLayout->addWidget(m_clearRouteButton);

    showEmptyPanel(true);
}

void RoutingPanel::clearListsSelections()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_routeWaypointListView->clearItemSelection();
}

void RoutingPanel::clearRouteButtonClicked()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_clearRouteButton->setDisabled(true);
    m_routeWaypointListView->clearList();
    m_routeWaypointListView->hide();
    showEmptyPanel(true);
    emit clearRoute();
}

void RoutingPanel::hideEvent(QHideEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QWidget::hideEvent(event);

    clearListsSelections();
}

void RoutingPanel::setRoute(Route &route)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_routeWaypointListView->clearList();
    showEmptyPanel(false);

    QList<RouteSegment> segments = route.segments();
    QList<GeoCoordinate> geometryPoints = route.geometryPoints();

    for (int i = 0; i < segments.size(); ++i) {
        RouteWaypointListItem *item = new RouteWaypointListItem();
        RouteSegment routeSegment = segments.at(i);
        item->setRouteWaypointData(routeSegment,
                                   geometryPoints.at(routeSegment.positionIndex()));

        m_routeWaypointListView->addListItem(QString::number(i), item);
        m_routeWaypointListView->show();
    }

    m_routeWaypointListView->scrollToTop();

    m_clearRouteButton->setDisabled(false);

    emit openPanelRequested(this);
}

void RoutingPanel::showEmptyPanel(bool show)
{
    if (show) {
        m_noRouteLabel->show();
        m_routeWaypointListView->hide();
    }
    else {
        m_noRouteLabel->hide();
    }
}
