/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef LISTVIEW_H
#define LISTVIEW_H

#include <QHash>
#include <QListWidget>

class GeoCoordinate;
class ListItem;

/**
* @brief View for ListItems.
*
* ListView is used to show ListItems in list view. Items can be added, removed or
* filtered.
*
* @author Jussi Laitinen - jussi.laitinen (at) ixonos.com
*/
class ListView : public QListWidget
{
    Q_OBJECT

public:
    /**
    * @brief Constructor.
    *
    * @param parent QWidget
    */
    ListView(QWidget *parent = 0);

    /**
    * @brief Destructor.
    *
    * Calls ListView::clearList().
    */
    ~ListView();

/******************************************************************************
* MEMBER FUNCTIONS AND SLOTS
******************************************************************************/
public:
    /**
    * @brief Add item to view and item list.
    *
    * @param key user ID
    * @param item item to add to view and list
    */
    void addListItem(const QString &key, ListItem *item);

    /**
    * @brief Adds item to view.
    *
    * @param item FriendListItem
    */
    void addListItemToView(ListItem *item);

    /**
    * @brief Clear unused items from view.
    *
    * Clears items which are not in item ID's list from the view and items list.
    *
    * @param itemIDs list of item ID's to keep in list view
    */
    void clearUnused(const QStringList &itemIDs);

    /**
    * @brief Clears filtering from list.
    *
    * Clears m_filteredItemIDs and shows all items.
    */
    void clearFilter();

    /**
    * @brief Deselects all selected items.
    */
    void clearItemSelection();

    /**
    * @brief Clears list.
    *
    * Items are removed from view and item list.
    */
    void clearList();

    /**
    * @brief Checks if view contains item with userID.
    *
    * @param userID user's ID
    * @return true if view contains item, false otherwise
    */
    bool contains(const QString &userID);

    /**
    * @brief Filters list by item IDs.
    *
    * Hide all items that are not in the itemIDs list.
    *
    * @param itemIDs item ID's for items that are shown
    */
    void filter(const QList<QString> &itemIDs);

    /**
    * @brief Filters list by text pattern.
    *
    * Filtering uses item names. If filtering by item IDs is on, filters only those items.
    *
    * @param pattern text pattern to filter
    */
    void filter(const QString &pattern);

    /**
    * @brief Takes item from view.
    *
    * Item is not deleted.
    *
    * @param itemID item's ID
    * @return ListItem
    */
    ListItem *takeListItemFromView(const QString &itemID);

    /**
    * @brief Returns ListItem with itemID.
    *
    * @param itemID item's ID
    * @return ListItem
    */
    ListItem *listItem(const QString &itemID);

    /**
    * @brief Returns ListItem by index.
    *
    * @param index item's index
    * @return ListItem
    */
    ListItem *listItemAt(int index);

    /**
    * @brief Prepends item to view and item list.
    *
    * @param key user ID
    * @param item item to prepend to view and list
    */
    void prependListItem(const QString &key, ListItem *item);

    /**
    * @brief Removes last ListItem in list.
    */
    void removeLastItem();

    /**
    * @brief Returns selected ListItem.
    *
    * @return ListItem if there is selected, 0 otherwise
    */
    ListItem *selectedItem();

    /**
    * @brief Sets selected item.
    *
    * @param item ListItem to select
    */
    void setSelectedItem(ListItem *item);

protected slots:
    /**
    * @brief Slot for list item clicked.
    *
    * Toggles items selection state and emits listItemClicked signal.
    * @param clickedItem clicked ListItem
    * @return true if item was selected, false otherwise
    */
    virtual bool listItemClicked(ListItem *clickedItem);

private slots:
    /**
    * @brief Slot for list item clicked.
    *
    * Calls listItemClicked(ListItem *item)
    * @param item QListWidgetItem
    */
    void listItemClicked(QListWidgetItem *item);

/******************************************************************************
* SIGNALS
******************************************************************************/
signals:
    /**
    * @brief Emitted when list item selection changes.
    */
    void listItemSelectionChanged();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QHash<QString, ListItem *> m_listItems; ///< List of items in this view. Key = user ID

    QList<QString> m_filteredItemIDs;       ///< List of filtered item IDs

    ListItem *m_currentItem;               ///< Previously selected item
};

#endif // LISTVIEW_H
