/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>

#include "listitem.h"
#include "listview.h"
#include "friendlistitem.h"

ListView::ListView(QWidget *parent)
    : QListWidget(parent),
      m_currentItem(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    setSelectionMode(QAbstractItemView::SingleSelection);
    setAutoFillBackground(false);
    viewport()->setAutoFillBackground(false);

    connect(this, SIGNAL(itemClicked(QListWidgetItem*)),
            this, SLOT(listItemClicked(QListWidgetItem*)));
}

void ListView::addListItem(const QString &key, ListItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_listItems.contains(key)) {
        addItem(item);
        m_listItems.insert(key, item);
    }
}

void ListView::addListItemToView(ListItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    addItem(item);
}

void ListView::clearItemSelection()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearSelection();

    if (m_currentItem)
        m_currentItem->setSelected(false);

    emit listItemSelectionChanged();
}

void ListView::clearList()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearItemSelection();

    qDeleteAll(m_listItems.begin(), m_listItems.end());
    m_listItems.clear();
    clear();

    m_currentItem = 0;
}

void ListView::clearUnused(const QStringList &itemIDs)
{
    qDebug() << __PRETTY_FUNCTION__;

    foreach (QString key, m_listItems.keys()) {
        if (!itemIDs.contains(key)) {
            ListItem *item = m_listItems.take(key);
            if (item) {
                takeItem(row(item));
                if (m_currentItem == item)
                    m_currentItem = 0;
                delete item;
            }
        }
    }
}

void ListView::clearFilter()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_filteredItemIDs.clear();

    foreach (ListItem *item, m_listItems)
        setItemHidden(item, false);
}

bool ListView::contains(const QString &itemID)
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_listItems.contains(itemID);
}

void ListView::filter(const QList<QString> &itemIDs)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_filteredItemIDs = itemIDs;

    foreach (ListItem *item, m_listItems) {
        if (itemIDs.contains(m_listItems.key(item))) {
            item->setHidden(false);
        } else {
            item->setSelected(false);
            item->setHidden(true);
        }
    }

    emit listItemSelectionChanged();
}

void ListView::filter(const QString &pattern)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_filteredItemIDs.isEmpty()) {
        foreach (ListItem *item, m_listItems) {
            if (item->title().contains(pattern, Qt::CaseInsensitive)) {
                item->setHidden(false);
            } else {
                item->setSelected(false);
                item->setHidden(true);
            }
        }
    } else {
        foreach (QString key, m_filteredItemIDs) {
            ListItem *item = m_listItems.value(key);
            if (item) {
                if (item->title().contains(pattern, Qt::CaseInsensitive)) {
                    item->setHidden(false);
                } else {
                    item->setSelected(false);
                    item->setHidden(true);
                }
            }
        }
    }

    emit listItemSelectionChanged();
}

ListItem *ListView::takeListItemFromView(const QString &itemID)
{
    qDebug() << __PRETTY_FUNCTION__;

    ListItem *item = listItem(itemID);
    takeItem(row(item));
    return item;
}

bool ListView::listItemClicked(ListItem *clickedItem)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_currentItem == clickedItem) {
        clickedItem->toggleSelection();
    } else {
        if (m_currentItem)
            m_currentItem->setSelected(false);

        clickedItem->setSelected(true);
    }
    m_currentItem = clickedItem;

    emit listItemSelectionChanged();

    return clickedItem->isSelected();
}

void ListView::listItemClicked(QListWidgetItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    ListItem *currentItem = dynamic_cast<ListItem*>(item);

    if (currentItem)
        listItemClicked(currentItem);
}

ListItem *ListView::listItem(const QString &itemID)
{
    qDebug() << __PRETTY_FUNCTION__;

    return dynamic_cast<ListItem*>(m_listItems.value(itemID));
}

ListItem *ListView::listItemAt(int index)
{
    qDebug() << __PRETTY_FUNCTION__;

    ListItem *listItem = 0;

    if (index < count())
        listItem = dynamic_cast<ListItem*>(item(index));

    return listItem;
}

void ListView::prependListItem(const QString &key, ListItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_listItems.contains(key)) {
        insertItem(0, item);
        m_listItems.insert(key, item);
    }
}

void ListView::removeLastItem()
{
    qDebug() << __PRETTY_FUNCTION__;

    ListItem *item = listItemAt(count() - 1);

    if (item) {
        if (item) {
            QString key = m_listItems.key(item);
            m_listItems.remove(key);
            takeItem(row(item));
            if (m_currentItem == item)
                m_currentItem = 0;
            delete item;
            item = 0;
        }
    }
}

ListItem *ListView::selectedItem()
{
    qDebug() << __PRETTY_FUNCTION__;

    QList<QListWidgetItem *> selectedListItems = selectedItems();

    if (!selectedListItems.isEmpty())
        return dynamic_cast<ListItem *>(selectedListItems.first());
    else
        return 0;
}

void ListView::setSelectedItem(ListItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    listItemClicked(item);
}

ListView::~ListView()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearList();
}
