/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#ifndef ROUTESEGMENT_H
#define ROUTESEGMENT_H

#include <QtGlobal>
#include <QString>

/**
 * @brief Container for a single route segment data
 *
 * Contains all data for a single route segment.
 *
 * @author Sami Rämö - sami.ramo@ixonos.com
 */
class RouteSegment
{
public:
    /**
    * @brief Constructor
    */
    RouteSegment();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
    /**
    * @brief Getter for azimuth
    *
    * @return Azimuth (in degrees)
    */
    qreal azimuth() const;

    /**
    * @brief Getter for earth direction code of the start of the segment
    *
    * @returns Earth direction (N, NE, E, SE, S, SW, W, NW)
    */
    const QString& earthDirection() const;

    /**
    * @brief Getter for text instruction
    *
    * e.g. Turn left at Oxford Street
    *
    * @returns Instruction text
    */
    const QString& instruction() const;

    /**
    * @brief Getter for segment length
    *
    * @returns Length of the segment in meters
    */
    qreal length() const;

    /**
    * @brief Getter for length caption text
    *
    * e.g. 22m, 23.4 km, 14.4 miles
    *
    * @returns Length of the segment text
    */
    const QString& lengthCaption() const;

    /**
    * @brief Getter for the route geometry position index of the segment
    *
    * @returns Index of the first point of the segment in route geometry
    */
    int positionIndex() const;

    /**
    * @brief Set azimuth
    *
    * @param azimuth Azimuth (in degrees)
    */
    void setAzimuth(qreal azimuth);

    /**
    * @brief Set earth direction code of the start of the segment
    *
    * @param direction Direction code
    */
    void setEarthDirection(const QString &direction);

    /**
    * @brief Set instruction text
    *
    * @param instruction Instructon text
    */
    void setInstruction(const QString &instruction);

    /**
    * @brief Set length
    *
    * @param meters Length in meters
    */
    void setLength(qreal meters);

    /**
    * @brief Set length caption text
    *
    * @param length Length caption text
    */
    void setLengthCaption(const QString &length);

    /**
    * @brief Set position index
    *
    * @param index Position index
    */
    void setPositionIndex(int index);

    /**
    * @brief Set estimated travel time of the segment
    *
    * @param seconds Estimated travel time in seconds
    */
    void setTime(int seconds);

    /**
    * @brief Set turn angle
    *
    * @param degrees Turn angle in degrees
    */
    void setTurnAngle(qreal degrees);

    /**
    * @brief Set turn type code
    *
    * @param type Turn type code
    */
    void setTurnType(const QString &type);

    /**
    * @brief Get street name/number parsed from the text instruction
    *
    * @returns Street name/number, or empty QString if value is missing
    */
    QString street() const;

    /**
    * @brief Getter for estimated travel time of the segment
    *
    * @returns Estimated travel time of the segment in seconds
    */
    int time() const;

    /**
    * @brief Getter for turn angle
    *
    * @returns Turn angle in degrees
    */
    qreal turnAngle() const;

    /**
    * @brief Getter for turn type code
    *
    * e.g.
    *    C  	continue (go straight)
    *    TL 	turn left
    *    TSLL 	turn slight left
    *    TSHL 	turn sharp left
    *    TR 	turn right
    *    TSLR 	turn slight right
    *    TSHR 	turn sharp right
    *    TU     U-turn
    *
    * @returns Turn type code
    */
    const QString& turnType() const;

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    int m_timeSeconds;          ///< estimated time required to travel the segment in seconds
    int m_positionIndex;        ///< index of the first point of the segment in route geometry

    qreal m_azimuth;            ///< azimuth
    qreal m_length;             ///< length of the segment in meters
    qreal m_turnAngle;          ///< angle in degress of the turn between two segments, 0 for straight

    QString m_earthDirection;   ///< direction: N, NE, E, SE, S, SW, W, NW
    QString m_instruction;      ///< text instruction, e.g. Turn left at Oxford Street
    QString m_lengthCaption;    ///< length of the segment e.g. 22m, 23.4 km, 14.4 miles
    QString m_turnType;         ///< code of the turn type, optional, absent for the first segment
};

#endif // ROUTESEGMENT_H
