/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#ifndef ROUTE_H
#define ROUTE_H

#include <QList>
#include <QString>

#include "coordinates/geocoordinate.h"
#include "routesegment.h"

/**
 * @brief Container for a single route
 *
 * Contains all data for a single route including route geometry points (a.k.a track points) and
 * route segments.
 *
 * @author Sami Rämö - sami.ramo@ixonos.com
 */
class Route
{
public:
    /**
    * @brief Constructor
    */
    Route();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
    /**
    * @brief Append geometry point (a.k.a track point) of the route
    *
    * Appending must be done in order starting from the beginning of the route.
    *
    * @param geometryPoint Geometry point
    */
    void appendGeometryPoint(const GeoCoordinate &geometryPoint);

    /**
    * @brief Append a route segment to the route
    *
    * Appending must be done in order starting from the beginning of the route.
    *
    * @param segment Route segment object
    */
    void appendSegment(const RouteSegment &segment);

    /**
    * @brief Getter for route end point name
    *
    * @returns Name of the end point (or empty QString)
    */
    const QString& endPointName() const;

    /**
    * @brief Get list of geometry points
    *
    * @returns Reference to list of geometry points
    */
    const QList<GeoCoordinate>& geometryPoints() const;

    /**
    * @brief Get list of route segments
    *
    * @returns Reference to list of route segments
    */
    const QList<RouteSegment>& segments() const;

    /**
    * @brief Set name of the route end point
    *
    * @param endPoint Name of the end point
    */
    void setEndPointName(const QString &endPoint);

    /**
    * @brief Set name of the route start point
    *
    * @param startPoint Name of the route start point
    */
    void setStartPointName(const QString &startPoint);

    /**
    * @brief Set total distance of the route
    *
    * @param meters Total distance of the route in meters
    */
    void setTotalDistance(int meters);

    /**
    * @brief Set estimated total travel time of the route
    *
    * @param seconds Estimated total travel time
    */
    void setTotalTime(int seconds);

    /**
    * @brief Getter for route start point name
    *
    * @returns Name of the start point (or empty QString)
    */
    const QString& startPointName() const;

    /**
    * @brief Getter for total route distance
    *
    * @returns Total route distance in meters
    */
    int totalDistance() const;

    /**
    * @brief Getter for estimated travel time
    *
    * @returns Estimated travel time in seconds
    */
    int totalTime() const;

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    int m_totalDistance;                ///< route total distance in meters
    int m_totalTime;                    ///< estimated route total time in seconds

    QList<GeoCoordinate> m_geometryPoints;    ///< lat/lon coordinates of the route points
    QList<RouteSegment> m_segments;     ///< route segments

    QString m_endPointName;             ///< name of the route end point
    QString m_startPointName;           ///< name of the route starting point
};

#endif // ROUTE_H
