/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QTransform>

#include "coordinates/scenecoordinate.h"
#include "mapcommon.h"

#include "maptile.h"

MapTile::MapTile()
    : m_zoomLevel(UNDEFINED)
    , m_tileNumber(QPoint(UNDEFINED, UNDEFINED))
{
    qDebug() << __PRETTY_FUNCTION__;

    setPos(UNDEFINED, UNDEFINED);
    setShapeMode(QGraphicsPixmapItem::BoundingRectShape);
}

SceneCoordinate MapTile::convertTileNumberToSceneCoordinate(int zoomLevel, QPoint tileNumber)
{
    qDebug() << __PRETTY_FUNCTION__;

    int pow = 1 << (OSM_MAX_ZOOM_LEVEL - zoomLevel);
    int x = tileNumber.x() * OSM_TILE_SIZE_X * pow;
    int y = tileNumber.y() * OSM_TILE_SIZE_Y * pow;

    return SceneCoordinate(x, y);
}

int MapTile::lastTileIndex(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    // subtract one because first tile index is zero
    return tilesPerSide(zoomLevel) - 1;
}

void MapTile::setSceneLevel(int currentZoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (currentZoomLevel < m_zoomLevel) {
        qreal z = static_cast<qreal>(MAP_SCENE_MIN_NORMAL_LEVEL + currentZoomLevel
                                     - (m_zoomLevel - currentZoomLevel)) + 0.5;
        setZValue(z);
    }
    else
        setZValue(static_cast<qreal>(MAP_SCENE_MIN_NORMAL_LEVEL + m_zoomLevel));
}

void MapTile::setTileNumber(QPoint tileNumber)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_tileNumber = tileNumber;
    setPosition();
}

void MapTile::setPosition()
{
    qDebug() << __PRETTY_FUNCTION__;

    const int MAX_TILE_NUMBER = MapTile::lastTileIndex(m_zoomLevel);

    // calculate min & max horizontal tile index numbers
    const int INDEX_0_TILE = 1;
    const int MIN_TILE_NUMBER_X = -tilesPerSide(m_zoomLevel)
                                  * MAP_SCENE_VERTICAL_OVERSIZE_FACTOR;
    const int MAX_TILE_NUMBER_X = tilesPerSide(m_zoomLevel)
                                  * (1 + MAP_SCENE_VERTICAL_OVERSIZE_FACTOR)
                                  - INDEX_0_TILE;

    if ((m_zoomLevel >= OSM_MIN_ZOOM_LEVEL) && (m_zoomLevel <= OSM_MAX_ZOOM_LEVEL)
        && (m_tileNumber.x() >= MIN_TILE_NUMBER_X) && (m_tileNumber.x() <= MAX_TILE_NUMBER_X)
        && (m_tileNumber.y() >= 0) && (m_tileNumber.y() <= MAX_TILE_NUMBER)) {
            setPos(convertTileNumberToSceneCoordinate(m_zoomLevel, m_tileNumber).toPointF());
    } else {
        setPos(UNDEFINED, UNDEFINED);
    }
}

void MapTile::setZoomLevel(int tileZoomLevel, int currentViewZoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomLevel = tileZoomLevel;
    setPosition();

    setSceneLevel(currentViewZoomLevel);

    qreal zoomFactor = static_cast<qreal>(1 << (OSM_MAX_ZOOM_LEVEL - m_zoomLevel));
    setScale(zoomFactor);
}

QPoint MapTile::tileNumber()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_tileNumber;
}

QString MapTile::tilePath(int zoomLevel, int x, int y)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString tilePathString(QString::number(zoomLevel) + "/");
    tilePathString.append(QString::number(x) + "/");
    tilePathString.append(QString::number(y));

    return tilePathString;
}

int MapTile::tilesPerSide(int zoomLevel)
{
    return (1 << zoomLevel);
}

int MapTile::zoomLevel()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_zoomLevel;
}
