/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QtAlgorithms>
#include <QDebug>
#include <QGraphicsView>
#include <QString>
#include <QStringList>
#include <QUrl>
#include <QHash>
#include <QHashIterator>
#include <QRect>

#include "common.h"
#include "coordinates/geocoordinate.h"
#include "frienditemshandler.h"
#include "gpslocationitem.h"
#include "mapcommon.h"
#include "mapfetcher.h"
#include "maprouteitem.h"
#include "mapscene.h"
#include "mapscroller.h"
#include "maptile.h"
#include "network/networkaccessmanager.h"
#include "ownlocationitem.h"
#include "user/user.h"

#include "mapengine.h"

const int SMOOTH_CENTERING_TIME_MS = 1000;

MapEngine::MapEngine(QObject *parent)
    : QObject(parent),
      m_autoCenteringEnabled(false),
      m_scrollStartedByGps(false),
      m_smoothScrollRunning(false),
      m_zoomedIn(false),
      m_zoomLevel(MAP_DEFAULT_ZOOM_LEVEL),
      m_centerTile(QPoint(UNDEFINED, UNDEFINED)),
      m_sceneCoordinate(SceneCoordinate(GeoCoordinate(MAP_DEFAULT_LATITUDE, MAP_DEFAULT_LONGITUDE))),
      m_tilesGridSize(QSize(0, 0)),
      m_viewSize(QSize(DEFAULT_SCREEN_WIDTH, DEFAULT_SCREEN_HEIGHT)),
      m_mapRouteItem(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapScene = new MapScene(this);

    m_mapFetcher = new MapFetcher(new NetworkAccessManager(this), this);
    connect(this, SIGNAL(fetchImage(int, int, int)),
            m_mapFetcher, SLOT(enqueueFetchMapImage(int, int, int)));
    connect(m_mapFetcher, SIGNAL(mapImageReceived(int, int, int, QPixmap)),
            this, SLOT(mapImageReceived(int, int, int, QPixmap)));
    connect(m_mapFetcher, SIGNAL(error(int, int)),
            this, SIGNAL(error(int, int)));

    m_ownLocation = new OwnLocationItem();
    m_ownLocation->hide(); // hide until first location info is received
    m_mapScene->addItem(m_ownLocation);

    m_gpsLocationItem = new GPSLocationItem();
    m_mapScene->addItem(m_gpsLocationItem);

    m_friendItemsHandler = new FriendItemsHandler(m_mapScene, this);
    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_friendItemsHandler, SLOT(refactorFriendItems(int)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_friendItemsHandler, SLOT(friendListUpdated(QList<User*>&)));

    connect(this, SIGNAL(friendImageReady(User*)),
            m_friendItemsHandler, SLOT(friendImageReady(User*)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            this, SLOT(friendsPositionsUpdated()));

    connect(m_friendItemsHandler, SIGNAL(locationItemClicked(QList<QString>)),
            this, SIGNAL(locationItemClicked(QList<QString>)));

    m_scroller = &MapScroller::getInstance();

    connect(m_scroller, SIGNAL(coordinateUpdated(SceneCoordinate)),
            this, SLOT(setCenterPosition(SceneCoordinate)));

    connect(m_scroller, SIGNAL(stateChanged(QAbstractAnimation::State, QAbstractAnimation::State)),
            this, SLOT(scrollerStateChanged(QAbstractAnimation::State)));
}

MapEngine::~MapEngine()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    settings.setValue(MAP_LAST_POSITION, QVariant::fromValue(centerGeoCoordinate()));
    settings.setValue(MAP_LAST_ZOOMLEVEL, m_zoomLevel);
}

QRect MapEngine::calculateTileGrid(SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPoint tileCoordinate = convertSceneCoordinateToTileNumber(m_zoomLevel, coordinate);

    QPoint topLeft;
    topLeft.setX(tileCoordinate.x() - (m_tilesGridSize.width() / 2));
    topLeft.setY(tileCoordinate.y() - (m_tilesGridSize.height() / 2));

    return QRect(topLeft, m_tilesGridSize);
}

void MapEngine::centerAndZoomTo(QRect rect, bool useMargins)
{
    qDebug() << __PRETTY_FUNCTION__;

    int marginHorizontal = 0;
    int marginVertical = 0;

    if (useMargins) {
        marginHorizontal = 5;
        marginVertical = 5;
    }

    // calculate the usable size of the view
    int viewUsableHeight = m_viewSize.height() - 2 * marginVertical;
    int viewUsableWidth = m_viewSize.width() - 2 * marginHorizontal;

    // calculate how many levels must be zoomed out from the closest zoom level to get the rect
    // fit inside the usable view area
    int shift = 0;
    while ((rect.height() > (viewUsableHeight * (1 << shift)))
           || (rect.width() > (viewUsableWidth * (1 << shift)))) {

        shift++;
    }


    scrollToPosition(SceneCoordinate(double(rect.center().x()), double(rect.center().y())));

    int zoomLevel = qBound(OSM_MIN_ZOOM_LEVEL, OSM_MAX_ZOOM_LEVEL - shift, OSM_MAX_ZOOM_LEVEL);
    setZoomLevel(zoomLevel);
}

GeoCoordinate MapEngine::centerGeoCoordinate()
{
    qDebug() << __PRETTY_FUNCTION__;

    return GeoCoordinate(m_sceneCoordinate);
}

void MapEngine::centerToCoordinates(GeoCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    scrollToPosition(SceneCoordinate(coordinate));
}

void MapEngine::clearRoute()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_mapRouteItem) {
        m_mapScene->removeItem(m_mapRouteItem);
        delete m_mapRouteItem;
        m_mapRouteItem = 0;
    }
}

QPoint MapEngine::convertSceneCoordinateToTileNumber(int zoomLevel, SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    int pow = 1 << (OSM_MAX_ZOOM_LEVEL - zoomLevel);
    int x = static_cast<int>(coordinate.x() / (OSM_TILE_SIZE_X * pow));
    int y = static_cast<int>(coordinate.y() / (OSM_TILE_SIZE_Y * pow));

    return QPoint(x, y);
}

QRectF MapEngine::currentViewSceneRect() const
{
    qDebug() << __PRETTY_FUNCTION__;

    const QPoint ONE_PIXEL = QPoint(1, 1);

    QGraphicsView *view = m_mapScene->views().first();
    QPointF sceneTopLeft = view->mapToScene(0, 0);
    QPoint viewBottomRight = QPoint(view->size().width(), view->size().height()) - ONE_PIXEL;
    QPointF sceneBottomRight = view->mapToScene(viewBottomRight);

    return QRectF(sceneTopLeft, sceneBottomRight);
}

void MapEngine::disableAutoCenteringIfRequired(SceneCoordinate coordinate)
{
    if (isAutoCenteringEnabled()) {
        int zoomFactor = (1 << (OSM_MAX_ZOOM_LEVEL - m_zoomLevel));

        SceneCoordinate oldPixelValue(m_lastAutomaticPosition.x() / zoomFactor,
                                      m_lastAutomaticPosition.y() / zoomFactor);

        SceneCoordinate newPixelValue(coordinate.x() / zoomFactor,
                                      coordinate.y() / zoomFactor);

        if ((abs(oldPixelValue.x() - newPixelValue.x()) > AUTO_CENTERING_DISABLE_DISTANCE)
            || (abs(oldPixelValue.y() - newPixelValue.y()) > AUTO_CENTERING_DISABLE_DISTANCE)) {

            emit mapScrolledManually();
        }
    }
}

void MapEngine::friendsPositionsUpdated()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapScene->spanItems(currentViewSceneRect());
}

void MapEngine::getTiles(SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_viewTilesGrid = calculateTileGrid(coordinate);
    updateViewTilesSceneRect();
    m_mapScene->setTilesGrid(m_viewTilesGrid);

    int topLeftX = m_viewTilesGrid.topLeft().x();
    int topLeftY = m_viewTilesGrid.topLeft().y();
    int bottomRightX = m_viewTilesGrid.bottomRight().x();
    int bottomRightY = m_viewTilesGrid.bottomRight().y();

    int tileMaxVal = MapTile::lastTileIndex(m_zoomLevel);

    for (int x = topLeftX; x <= bottomRightX; ++x) {
        for (int y = topLeftY; y <= bottomRightY; ++y) {

            // map doesn't span in vertical direction, so y index must be inside the limits
            if (y >= MAP_TILE_MIN_INDEX && y <= tileMaxVal) {
                if (!m_mapScene->tileInScene(MapTile::tilePath(m_zoomLevel, x, y)))
                    emit fetchImage(m_zoomLevel, normalize(x, MAP_TILE_MIN_INDEX, tileMaxVal), y);
            }
        }
    }
}

void MapEngine::gpsPositionUpdate(GeoCoordinate position, qreal accuracy)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gpsPosition = position;

    // update GPS location item (but only if accuracy is a valid number)
    if (!isnan(accuracy)) {
        qreal resolution = MapScene::horizontalResolutionAtLatitude(position.latitude());
        m_gpsLocationItem->updateItem(SceneCoordinate(position).toPointF(), accuracy, resolution);
    }

    m_mapScene->spanItems(currentViewSceneRect());

    // do automatic centering (if enabled)
    if (m_autoCenteringEnabled) {
        m_lastAutomaticPosition = SceneCoordinate(position);
        m_scrollStartedByGps = true;
        scrollToPosition(m_lastAutomaticPosition);
    }

    updateDirectionIndicator();
}

void MapEngine::init()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    // init can be only done if both values exists in the settings
    if (settings.contains(MAP_LAST_POSITION) && settings.contains(MAP_LAST_ZOOMLEVEL)) {
        QVariant zoomLevel = settings.value(MAP_LAST_ZOOMLEVEL);
        QVariant location = settings.value(MAP_LAST_POSITION);

        // also the init can be only done if we are able to convert variants into target data types
        if (zoomLevel.canConvert<int>() && location.canConvert<GeoCoordinate>()) {
            m_zoomLevel = zoomLevel.toInt();
            m_sceneCoordinate = SceneCoordinate(location.value<GeoCoordinate>());
        }
    }

    // emit zoom level and center coordinate so that all parts of the map system gets initialized
    // NOTE: emit is also done even if we weren't able to read initial valuef from the settings
    //       so that the default values set in the constructor are used
    emit zoomLevelChanged(m_zoomLevel);
    scrollToPosition(m_sceneCoordinate);
}

bool MapEngine::isAutoCenteringEnabled()
{
    return m_autoCenteringEnabled;
}

bool MapEngine::isCenterTileChanged(SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPoint centerTile = convertSceneCoordinateToTileNumber(m_zoomLevel, coordinate);
    QPoint temp = m_centerTile;
    m_centerTile = centerTile;

    return (centerTile != temp);
}

void MapEngine::mapImageReceived(int zoomLevel, int x, int y, const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;

    // add normal tile inside the world
    QPoint tileNumber(x, y);
    m_mapScene->addTile(zoomLevel, tileNumber, image, m_zoomLevel);

    // note: add 1 so odd width is rounded up and even is rounded down
    int tilesGridWidthHalf = (m_viewTilesGrid.width() + 1) / 2;

    // duplicate to east side? (don't need to duplicate over padding)
    if (tileNumber.x() < (tilesGridWidthHalf - MAP_GRID_PADDING)) {
        QPoint adjustedTileNumber(tileNumber.x() + MapTile::lastTileIndex(zoomLevel) + 1,
                                  tileNumber.y());
        m_mapScene->addTile(zoomLevel, adjustedTileNumber, image, m_zoomLevel);
    }

    // duplicate to west side? (don't need to duplicate over padding)
    if (tileNumber.x() > (MapTile::lastTileIndex(zoomLevel)
                          - tilesGridWidthHalf
                          + MAP_GRID_PADDING)) {
        QPoint adjustedTileNumber(tileNumber.x() - MapTile::lastTileIndex(zoomLevel) - 1,
                                  tileNumber.y());
        m_mapScene->addTile(zoomLevel, adjustedTileNumber, image, m_zoomLevel);
    }
}

int MapEngine::normalize(int value, int min, int max)
{
    qDebug() << __PRETTY_FUNCTION__;
    Q_ASSERT_X(max >= min, "parameters", "max can't be smaller than min");

    while (value < min)
        value += max - min + 1;

    while (value > max)
        value -= max - min + 1;

    return value;
}

void MapEngine::receiveOwnLocation(User *user)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(user) {
        m_ownLocation->setPos(SceneCoordinate(user->coordinates()).toPointF());
        if (!m_ownLocation->isVisible())
            m_ownLocation->show();
    } else {
        m_ownLocation->hide();
    }

    m_mapScene->spanItems(currentViewSceneRect());
}

QGraphicsScene* MapEngine::scene()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_mapScene;
}

void MapEngine::scrollerStateChanged(QAbstractAnimation::State newState)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_smoothScrollRunning
        && newState != QAbstractAnimation::Running) {
            m_smoothScrollRunning = false;

            // don't disable auto centering if current animation was stopped by new update from GPS
            if (!m_scrollStartedByGps)
                disableAutoCenteringIfRequired(m_sceneCoordinate);
    }

    m_scrollStartedByGps = false;
}

void MapEngine::scrollToPosition(SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_scroller->stop();
    m_scroller->setEasingCurve(QEasingCurve::InOutQuart);
    m_scroller->setDuration(SMOOTH_CENTERING_TIME_MS);
    m_scroller->setStartValue(m_sceneCoordinate);
    m_scroller->setEndValue(coordinate);
    m_smoothScrollRunning = true;
    m_scroller->start();
}

void MapEngine::setAutoCentering(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_autoCenteringEnabled = enabled;

    if (!m_autoCenteringEnabled && m_gpsLocationItem->isVisible())
        updateDirectionIndicator();
}

void MapEngine::setCenterPosition(SceneCoordinate coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    // jump to opposite side of the world if world horizontal limit is exceeded
    coordinate.setX(normalize(coordinate.x(), OSM_MAP_MIN_PIXEL_X, OSM_MAP_MAX_PIXEL_X));

    // don't allow vertical scene coordinates go out of the map
    coordinate.setY(qBound(double(OSM_MAP_MIN_PIXEL_Y),
                              coordinate.y(),
                              double(OSM_MAP_MAX_PIXEL_Y)));

    if (!m_smoothScrollRunning)
        disableAutoCenteringIfRequired(coordinate);

    m_sceneCoordinate = coordinate;
    emit locationChanged(m_sceneCoordinate);

    if (isCenterTileChanged(coordinate)) {
        getTiles(coordinate);
        m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    }

    m_mapScene->spanItems(currentViewSceneRect());
    emit newMapResolution(viewResolution());

    updateDirectionIndicator();
}

void MapEngine::setGPSEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gpsLocationItem->setEnabled(enabled);
}

void MapEngine::setRoute(Route &route)
{
    qDebug() << __PRETTY_FUNCTION__;

    clearRoute();

    m_mapRouteItem = new MapRouteItem(&route);
    m_mapScene->addItem(m_mapRouteItem);

    centerAndZoomTo(m_mapRouteItem->boundingRect().toRect());
}

void MapEngine::setZoomLevel(int newZoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomLevel = newZoomLevel;
    zoomed();
}

void MapEngine::setTilesGridSize(const QSize &viewSize)
{
    qDebug() << __PRETTY_FUNCTION__;

    // there must be scrolling reserve of at least half tile added to tile amount
    // calculated from view size
    const qreal SCROLLING_RESERVE = 0.5;

    // converting scene tile to tile number does cause grid centering inaccuracy of one tile
    const int CENTER_TILE_INACCURACY = 1;

    int gridWidth = ceil(qreal(viewSize.width()) / OSM_TILE_SIZE_X + SCROLLING_RESERVE)
                    + CENTER_TILE_INACCURACY + (MAP_GRID_PADDING * 2);
    int gridHeight = ceil(qreal(viewSize.height()) / OSM_TILE_SIZE_Y + SCROLLING_RESERVE)
                     + CENTER_TILE_INACCURACY + (MAP_GRID_PADDING * 2);

    m_mapFetcher->setDownloadQueueSize(gridWidth * gridHeight);

    m_tilesGridSize.setHeight(gridHeight);
    m_tilesGridSize.setWidth(gridWidth);
}

void MapEngine::showMapArea(const GeoCoordinate &swBound, const GeoCoordinate &neBound)
{
    qDebug() << __PRETTY_FUNCTION__;

    QRect area;
    area.setTopRight(SceneCoordinate(neBound).toPointF().toPoint());
    area.setBottomLeft(SceneCoordinate(swBound).toPointF().toPoint());

    centerAndZoomTo(area, false);
}

void MapEngine::updateDirectionIndicator()
{
    qDebug() << __PRETTY_FUNCTION__;

    qreal distance = m_gpsPosition.distanceTo(m_sceneCoordinate);

    qreal direction = m_sceneCoordinate.azimuthTo(SceneCoordinate(m_gpsPosition));

    // direction indicator triangle should be drawn only if the gps location item is not currently
    // visible on the view
    bool drawDirectionIndicatorTriangle = true;
    if (currentViewSceneRect().contains(m_gpsLocationItem->pos()))
        drawDirectionIndicatorTriangle = false;

    emit directionIndicatorValuesUpdate(direction, distance, drawDirectionIndicatorTriangle);
}

void MapEngine::updateViewTilesSceneRect()
{
    qDebug() << __PRETTY_FUNCTION__;

    const QPoint ONE_TILE = QPoint(1, 1);
    const double ONE_PIXEL = 1;

    SceneCoordinate topLeft = MapTile::convertTileNumberToSceneCoordinate(m_zoomLevel,
                                                                        m_viewTilesGrid.topLeft());

    // one tile - one pixel is added because returned coordinates are pointing to upper left corner
    // of the last tile.
    SceneCoordinate bottomRight
            = MapTile::convertTileNumberToSceneCoordinate(m_zoomLevel,
                                                          m_viewTilesGrid.bottomRight() + ONE_TILE);
    bottomRight.setX(bottomRight.x() - ONE_PIXEL);
    bottomRight.setY(bottomRight.y() - ONE_PIXEL);

    m_mapScene->tilesSceneRectUpdated(QRect(topLeft.toPointF().toPoint(),
                                            bottomRight.toPointF().toPoint()));
}

void MapEngine::viewResized(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_viewSize = size;
    setTilesGridSize(m_viewSize);

    emit locationChanged(m_sceneCoordinate);
    getTiles(m_sceneCoordinate);
    m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    m_mapScene->setSceneVerticalOverlap(m_viewSize.height(), m_zoomLevel);
}

qreal MapEngine::viewResolution()
{
    qDebug() << __PRETTY_FUNCTION__;

    qreal scale = (1 << (OSM_MAX_ZOOM_LEVEL - m_zoomLevel));

    return MapScene::horizontalResolutionAtLatitude(centerGeoCoordinate().latitude()) * scale;
}

void MapEngine::viewZoomFinished()
{
    qDebug() << __PRETTY_FUNCTION__;

    updateDirectionIndicator();

    if (m_zoomedIn) {
        m_zoomedIn = false;
        m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    }

    if (m_zoomLevel == OSM_MAX_ZOOM_LEVEL)
        emit maxZoomLevelReached();
    else if (m_zoomLevel == MAP_VIEW_MIN_ZOOM_LEVEL)
        emit minZoomLevelReached();
}

void MapEngine::zoomed()
{
    emit zoomLevelChanged(m_zoomLevel);
    m_mapScene->setTilesDrawingLevels(m_zoomLevel);
    m_mapScene->setZoomLevel(m_zoomLevel);
    getTiles(m_sceneCoordinate);
    m_mapScene->setSceneVerticalOverlap(m_viewSize.height(), m_zoomLevel);
    m_mapScene->spanItems(currentViewSceneRect());
    emit newMapResolution(viewResolution());
}

void MapEngine::zoomIn()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomLevel < OSM_MAX_ZOOM_LEVEL) {
        m_zoomLevel++;
        m_zoomedIn = true;
        zoomed();
    }
}

void MapEngine::zoomOut()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomLevel > MAP_VIEW_MIN_ZOOM_LEVEL) {
        m_zoomLevel--;
        zoomed();
    }
}
