/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Ville Tiensuu - ville.tiensuu@ixonos.com
       Kaj Wallin - kaj.wallin@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef FACEBOOKAUTHENTICATION_H
#define FACEBOOKAUTHENTICATION_H

#include <QUrl>
#include "facebookcredentials.h"

/**
* @brief FacebookAuthentication class takes care of parsing and handling of  credentials for
*        Facebook. Other components of Situare application needs credentials to communicate with
*        facebook.
*
* @author Ville Tiensuu
*/
class FacebookAuthentication : public QObject
{
    Q_OBJECT

public:
    /**
    * @brief FacebookAuthentication constructor
    *
    * -Checks if there is valid credentials stored on the file. If there is emits signal.
    *
    * @param parent instance of parent
    */
    FacebookAuthentication(QObject *parent = 0);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/

    /**
    * @brief Getter for m_loginCredentials
    *
    * @return FacebookCredentials
    */
    FacebookCredentials loginCredentials() const;

public slots:

    /**
    * @brief Clears account iformation from settings
    *
    * @param keepUsername keep = true, false otherwise
    */
    void clearAccountInformation(bool keepUsername = false);

    /**
    * @brief Loads username from settings
    *
    * @return QString Loaded username
    */
    const QString loadUsername();

    /**
    * @brief Saves username to settings
    *
    * @param username Username to be saved
    */
    void saveUsername(const QString &username);

    /**
    * @brief Shows the m_webView and loads page that is specified in the m_facebookLoginPage
    *        variable. Specifies font size for the page.
    *    
    */
    void start();

private slots:

    /**
    * @brief  Search credentials from URL that is given as parameter.
    *         If credentials are found thay are stored to loginCredentials variable.
    *
    * @param url URL where this method tries to find credentials.
    * @return bool if credentials are found returns true,
    *               if credentials are not found returns false.
    */
    bool updateCredentials(const QUrl & url);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/

signals:

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
    * @brief This signal is emitted if updateCredentials method finds credentials from URL.
    *        Signal is also emitted at the beginning of the program if there is valid credentials
    *        in the file.
    *
    * @param credentials New credentials
    */
    void credentialsReady(const FacebookCredentials &credentials);

    /**
    * @brief This signal is emitted always when login is called. At first  the application tries
    *        to login using saved cookies
    *
    */
    void loginUsingCookies();

    /**
    * @brief Signals when credentials are invalid new login is needed
    *
    */
    void newLoginRequest();

    /**
    * @brief This signal is emitted when new cookies need to be saved.
    *
    */
    void saveCookiesRequest();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/

private:

    bool m_freshLogin;  ///< Flag for fresh login

    /**
    * @brief Dataclass that contains authorization to use facebook. Dataclass is composed of five
    *        QStrings and setters and getters.
    *
    * @var m_loginCredentials
    */
    FacebookCredentials m_loginCredentials;
};

#endif // FACEBOOKAUTHENTICATION_H
