/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <cmath>

#include <QDebug>

#include "map/osm.h"
#include "scenecoordinate.h"

#include "geocoordinate.h"

bool GeoCoordinate::m_metaTypeIsRegistered = false;

GeoCoordinate::GeoCoordinate() :
        m_latitude(0),
        m_longitude(0),
        m_validityFlags(NoCoordinatesSet)
{
    qDebug() << __PRETTY_FUNCTION__;

    registerMetaType();
}

GeoCoordinate::GeoCoordinate(double latitude, double longitude)
{
    qDebug() << __PRETTY_FUNCTION__;

    registerMetaType();

    setLatitude(latitude);
    setLongitude(longitude);
}

GeoCoordinate::GeoCoordinate(const SceneCoordinate &coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    registerMetaType();

    convertFrom(coordinate);
}

void GeoCoordinate::convertFrom(const SceneCoordinate &coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int MAP_PIXELS_Y = OSM_MAP_MAX_PIXEL_Y + 1;

    double longitude = coordinate.x() / OSM_MAP_PIXELS_X * 360.0 - 180;

    double n = M_PI - 2.0 * M_PI * coordinate.y() / MAP_PIXELS_Y;
    double latitude = 180.0 / M_PI * atan(0.5 * (exp(n) - exp(-n)));

    setLatitude(latitude);
    setLongitude(longitude);
}

qreal GeoCoordinate::distanceTo(const GeoCoordinate &other) const
{
    qDebug() << __PRETTY_FUNCTION__;

    const qreal EARTH_RADIUS = 6371.01;     // Earth radius in km
    const qreal TO_RAD = (M_PI / 180);      // Conversion factor to radians
    const int KM_TO_M = 1000;               // Conversion from kilometers to meters

    GeoCoordinate delta = other - *this;

    qreal a = pow(sin(delta.latitude() * TO_RAD / 2), 2)
              + cos(m_latitude * TO_RAD)
              * cos(other.latitude() * TO_RAD)
              * pow(sin(delta.longitude() * TO_RAD / 2), 2);
    qreal c = 2 * atan2(sqrt(a), sqrt(1 - a));

    return (EARTH_RADIUS * c  * KM_TO_M);
}

bool GeoCoordinate::isNull() const
{
    qDebug() << __PRETTY_FUNCTION__;

    if ((m_latitude == 0) && (m_longitude == 0))
        return true;

    return false;
}

bool GeoCoordinate::isValid()
{
    qDebug() << __PRETTY_FUNCTION__;

    const double MAX_LATITUDE = 90;
    const double MIN_LATITUDE = -90;
    const double MIN_LONGITUDE = -180.0;
    const double MAX_LONGITUDE = 180.0;

    if ((m_validityFlags & LatitudeSet) && (m_validityFlags & LongitudeSet) &&
        (m_latitude >= MIN_LATITUDE) && (m_latitude <= MAX_LATITUDE) &&
        (m_longitude >= MIN_LONGITUDE) && (m_longitude <= MAX_LONGITUDE))

        return true;
    else
        return false;
}

double GeoCoordinate::latitude() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_latitude;
}

double GeoCoordinate::longitude() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_longitude;
}

void GeoCoordinate::registerMetaType()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_metaTypeIsRegistered) {
        qRegisterMetaType<GeoCoordinate>("GeoCoordinate");
        qRegisterMetaTypeStreamOperators<GeoCoordinate>("GeoCoordinate");
        m_metaTypeIsRegistered = true;
    }
}

void GeoCoordinate::setLatitude(double latitude)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_latitude = latitude;
    m_validityFlags |= LatitudeSet;
}

void GeoCoordinate::setLongitude(double longitude)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_longitude = longitude;
    m_validityFlags |= LongitudeSet;
}

const GeoCoordinate operator-(const GeoCoordinate &coordinate1, const GeoCoordinate &coordinate2)
{
    return GeoCoordinate(coordinate1.m_latitude - coordinate2.m_latitude,
                         coordinate1.m_longitude - coordinate2.m_longitude);
}

QDataStream &operator<<(QDataStream &out, const GeoCoordinate &coordinate)
{
    out << coordinate.m_latitude << coordinate.m_longitude;
    return out;
}

QDataStream &operator>>(QDataStream &in, GeoCoordinate &coordinate)
{
    in >> coordinate.m_latitude >> coordinate.m_longitude;
    return in;
}

QDebug operator<<(QDebug dbg, const GeoCoordinate &coordinate)
{
    dbg.nospace() << "(" << coordinate.latitude() << ", " << coordinate.longitude() << ")";

    return dbg.space();
}
