#!/bin/bash

clear

# set default values
UNIT_TESTS_ROOT_DIR="../tests"
REPORT_PATH="../doc"
REPORT_FILE="unit_test_results.txt"
TEMP_FILE_EXEC_NOT_FOUND="temp_exec_not_found.txt"
TEMP_REPORT_FILE="temp_report.txt"

MAKE_OPTIONS="-j3 -k -s -w"

SCRIPT_DIR=(`pwd`)

# check that directories are found
if [[ -d $REPORT_PATH && -d $UNIT_TESTS_ROOT_DIR ]]; then
    # convert relational paths to absolute ones
    UNIT_TESTS_ROOT_DIR_ABSOLUTE=`cd $UNIT_TESTS_ROOT_DIR; pwd`
    REPORT_PATH_ABSOLUTE=`cd $REPORT_PATH; pwd`
    REPORT="$REPORT_PATH_ABSOLUTE/$REPORT_FILE"
    echo ""
    echo "Running recursively all tests under $UNIT_TESTS_ROOT_DIR_ABSOLUTE"
    echo "Saving results to $REPORT"
    echo ""

    # overwrite report summary file and write header
    echo "##########################################" > $REPORT
    echo "# Summary of unit tests executed" >> $REPORT
    echo "# Date: `date`" >> $REPORT
    echo "# User: `whoami`" >> $REPORT
    echo "##########################################" >> $REPORT
    echo "" >> $REPORT

    # find all test .pro files paths, cut .pro extension
    UNIT_TEST_PROJECTS=(`find $UNIT_TESTS_ROOT_DIR_ABSOLUTE | egrep \.pro$ | sed -e s,.pro$,,g`)

    echo "###################################################"
    echo "All unit test executables will be deleted"
    echo "and qmake + make will be run for root tests.pro"
    echo "###################################################"
    echo ""

    # remove all unit test binaries
    for BINARY in "${UNIT_TEST_PROJECTS[@]}"
    do
        # remove only if file really exists
        if [ -a $BINARY ]; then
            rm $BINARY
        fi
    done

    # run qmake
    cd $UNIT_TESTS_ROOT_DIR_ABSOLUTE
    qmake -r
    make $MAKE_OPTIONS
    cd -

    echo ""
    echo "###################################################"
    echo "Executing tests..."
    echo "###################################################"

    # run tests
    for PROJECT in "${UNIT_TEST_PROJECTS[@]}"
    do
        # do not try to test the root tests.pro project
        if [ $PROJECT != "$UNIT_TESTS_ROOT_DIR_ABSOLUTE/tests" ]; then
            # check if unit test file is found and is executable
            if [ -x $PROJECT ]; then
                # go to unit test directory
                cd "`echo $PROJECT | grep --perl-regexp -o .*\(?=/\)`"
                # run unit test, save to temp file, ignore strerr stream
                $PROJECT -silent -o $REPORT_PATH_ABSOLUTE/$TEMP_REPORT_FILE 2>&1 | grep -v -f $SCRIPT_DIR/filtering_patterns_for_unit_tests_output.txt
                cd - > /dev/null
                # print to screen and append to summary
                cat $REPORT_PATH_ABSOLUTE/$TEMP_REPORT_FILE
                cat $REPORT_PATH_ABSOLUTE/$TEMP_REPORT_FILE >> $REPORT_PATH_ABSOLUTE/$REPORT_FILE
                echo ""
                echo "" >> $REPORT_PATH_ABSOLUTE/$REPORT_FILE
            else
                # save path of missing test to temporary file
                MISSING=(`echo $PROJECT | sed -e s,$UNIT_TESTS_ROOT_DIR_ABSOLUTE,,g`)
                echo "$MISSING" >> "$REPORT_PATH_ABSOLUTE/$TEMP_FILE_EXEC_NOT_FOUND"
            fi
        fi
    done

    if [ -f "$REPORT_PATH_ABSOLUTE/$TEMP_FILE_EXEC_NOT_FOUND" ]; then
        # print list of missing test executables
        echo ""
        echo "###################################################"
        echo "Executables for following unit tests were not found"
        echo -e "###################################################\E[31m\E[1m" ## set red color & bold
        cat $REPORT_PATH_ABSOLUTE/$TEMP_FILE_EXEC_NOT_FOUND
        tput sgr0 # restore terminal defaults
        echo ""
        echo "Some possible causes:"
        echo "  - project has set target name explicitly. Target name must be same as the directory name"
        echo "  - don't use shadow build system"
        echo "  - test project is not included in the master test project (tests/tests.pro)"

        # and save same list also to test summary file
        echo "" >> $REPORT
        echo "###################################################" >> $REPORT
        echo "Executables for following unit tests were not found" >> $REPORT
        echo "###################################################"  >> $REPORT
        cat $REPORT_PATH_ABSOLUTE/$TEMP_FILE_EXEC_NOT_FOUND  >> $REPORT
    fi

    # check if unit test output contained any qDebug or qWarning prints
    DEBUG_COUNT=$(egrep -c "(QDEBUG|QWARN)" $REPORT_PATH_ABSOLUTE/$REPORT_FILE)
    if [ $DEBUG_COUNT -ge 1 ]; then
        echo ""
        # print message in red and bold
        echo -e "\E[31m\E[1mDisable debug output from unit tests so test output stays readable!!!"
        tput sgr0 # restore terminal defaults
    fi

    # remove temporary files
    rm $REPORT_PATH_ABSOLUTE/temp_*.txt

    # remove all unit test binaries so they doesn't accidentally get into repository
    for BINARY in "${UNIT_TEST_PROJECTS[@]}"
    do
        # remove only if file really exists
        if [ -a $BINARY ]; then
            rm $BINARY
        fi
    done

    # tell that script reached the end
    echo "" >> $REPORT
    echo "All done!" >> $REPORT
    echo ""
    echo "All done!"
else
    echo "Paths for report and/or tests not found!"
    echo "Script must be run from scripts directory"
fi
echo ""
