/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QSettings>

#include "avatarimage.h"
#include "../common.h"
#include "extendedlistitem.h"
#include "extendedlistitemdelegate.h"
#include "locationlistitem.h"
#include "locationlistview.h"
#include "imagebutton.h"
#include "panelcommon.h"
#include "routing/location.h"
#include "searchhistorylistitem.h"
#include "searchhistorylistview.h"

#include "locationsearchpanel.h"

const QString SETTINGS_SEARCH_HISTORY = "SEARCH_HISTORY";

LocationSearchPanel::LocationSearchPanel(QWidget *parent)
    : PanelBase(parent)
{
    qDebug() << __PRETTY_FUNCTION__;

    // --- HEADER WIDGET ---
    QWidget *resultsHeaderWidget = new QWidget();
    resultsHeaderWidget->setAutoFillBackground(true);
    QPalette labelPalette = resultsHeaderWidget->palette();
    labelPalette.setColor(QPalette::Background, Qt::black);
    resultsHeaderWidget->setPalette(labelPalette);

    QHBoxLayout *headerLayout = new QHBoxLayout();
    resultsHeaderWidget->setLayout(headerLayout);
    headerLayout->setContentsMargins(PANEL_MARGIN_LEFT, PANEL_MARGIN_TOP,
                                     PANEL_MARGIN_RIGHT, PANEL_MARGIN_BOTTOM);

    m_resultsLabel = new QLabel(this);
    headerLayout->addWidget(m_resultsLabel, 0, Qt::AlignCenter);
    setHeaderText(0);

    // --- SEARCH HISTORY LIST VIEW ---
    m_searchHistoryListView = new SearchHistoryListView(this);
    m_searchHistoryListView->setItemDelegate(new ExtendedListItemDelegate(this));

    connect(m_searchHistoryListView, SIGNAL(searchHistoryItemClicked(QString)),
            this, SIGNAL(searchHistoryItemClicked(QString)));

    // --- SEARCH RESULTS LIST VIEW ---
    m_locationListView = new LocationListView(this);
    m_locationListView->setItemDelegate(new ExtendedListItemDelegate(this));

    connect(m_locationListView,
            SIGNAL(locationItemClicked(const GeoCoordinate&, const GeoCoordinate&)),
            this,
            SIGNAL(locationItemClicked(const GeoCoordinate&, const GeoCoordinate&)));

    connect(m_locationListView, SIGNAL(listItemSelectionChanged()),
            this, SLOT(onListItemSelectionChanged()));

    QVBoxLayout *resultsListViewLayout = new QVBoxLayout;
    resultsListViewLayout->setContentsMargins(PANEL_MARGIN_LEFT, PANEL_MARGIN_TOP,
                                       PANEL_MARGIN_RIGHT, PANEL_MARGIN_BOTTOM);
    resultsListViewLayout->addWidget(m_searchHistoryListView);
    resultsListViewLayout->addWidget(m_locationListView);

    // --- MAIN LAYOUT ---
    QVBoxLayout *panelLayout = new QVBoxLayout;
    panelLayout->setSpacing(0);
    setLayout(panelLayout);

    const int MARGIN_LEFT = 0;
    panelLayout->setContentsMargins(MARGIN_LEFT, PANEL_MARGIN_TOP,
                                    PANEL_MARGIN_RIGHT, PANEL_MARGIN_BOTTOM);
    m_noSearchLabel = new QLabel();
    m_noSearchLabel->setText("No search results");
    m_noSearchLabel->setAlignment(Qt::AlignCenter);

    QPalette m_noSearchPalette = palette();
    m_noSearchPalette.setColor(QPalette::Foreground, Qt::white);
    m_noSearchLabel->setPalette(m_noSearchPalette);

    panelLayout->addWidget(resultsHeaderWidget);
    panelLayout->addWidget(m_noSearchLabel, Qt::AlignCenter);
    panelLayout->addLayout(resultsListViewLayout);

    // --- CONTEXT BUTTONS ---
    m_routeButton = new ImageButton(":res/images/route_to_location.png",
                                    ":res/images/route_to_location_s.png", "", this);
    connect(m_routeButton, SIGNAL(clicked()),
            this, SLOT(routeToSelectedLocation()));

    ImageButton *searchLocationButton = new ImageButton(":/res/images/search.png",
                                                        ":/res/images/search_s.png", "", this);

    connect(searchLocationButton, SIGNAL(clicked()),
            this, SIGNAL(requestSearchLocation()));

    m_clearLocationListButton = new ImageButton(":/res/images/back.png",
                                                ":/res/images/back_s.png",
                                                ":/res/images/back_d.png", this);
    m_clearLocationListButton->setDisabled(true);

    connect(m_clearLocationListButton, SIGNAL(clicked()),
            this, SLOT(showSearchHistoryListView()));

    m_itemButtonsLayout->addWidget(m_routeButton);
    m_genericButtonsLayout->addWidget(searchLocationButton);
    m_genericButtonsLayout->addWidget(m_clearLocationListButton);

    showSearchHistoryListView();
    showEmptyPanel(true);
    readSettings();
}

LocationSearchPanel::~LocationSearchPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    QList<QVariant> searchHistories;

    for (int i = 0; i < m_searchHistoryListView->count(); ++i) {
        SearchHistoryListItem *item = dynamic_cast<SearchHistoryListItem*>(
                m_searchHistoryListView->listItemAt(i));

        if (item) {
            QList<QString> searchHistory;
            searchHistory.append(item->title());
            searchHistory.append(item->dateTime().toString());
            searchHistories.append(QVariant(searchHistory));
        }
    }

    settings.setValue(SETTINGS_SEARCH_HISTORY, searchHistories);
}

void LocationSearchPanel::prependSearchHistory(QString searchString, QDateTime dateTime)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int SEARCH_HISTORY_LIMIT = 10;
    static int counter = 0;

    showSearchHistoryListView();
    showEmptyPanel(false);

    if (m_searchHistoryListView->count() >= SEARCH_HISTORY_LIMIT)
        m_searchHistoryListView->removeLastItem();

    SearchHistoryListItem *item = new SearchHistoryListItem();
    item->setSearchHistoryData(searchString, dateTime);
    m_searchHistoryListView->prependListItem(QString::number(counter++), item);
}

void LocationSearchPanel::clearListsSelections()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationListView->clearItemSelection();
    m_searchHistoryListView->clearItemSelection();
}

void LocationSearchPanel::hideEvent(QHideEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QWidget::hideEvent(event);

    clearListsSelections();
}

void LocationSearchPanel::populateLocationListView(const QList<Location> &locations)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationListView->clearList();
    showLocationListView(locations.count());

    for (int i = 0; i < locations.size(); ++i) {
        LocationListItem *item = new LocationListItem();
        item->setLocationData(locations.at(i));
        m_locationListView->addListItem(QString::number(i), item);
    }

    m_locationListView->scrollToTop();
}

void LocationSearchPanel::readSettings()
{
    qDebug() << __PRETTY_FUNCTION__;

    const int SEARCH_HISTORY_LIST_ITEM_COUNT = 2;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    QList<QVariant> searchHistories = settings.value(SETTINGS_SEARCH_HISTORY).toList();

    //Read from end to begin so items are prepended in correct order
    for (int i = searchHistories.count() - 1; i >= 0; --i) {
        QList<QVariant> searchHistory = searchHistories.at(i).toList();
        if (searchHistory.count() == SEARCH_HISTORY_LIST_ITEM_COUNT) {
            prependSearchHistory(searchHistory.at(0).toString(),
                                 QDateTime::fromString(searchHistory.at(1).toString()));
        }
    }
}

void LocationSearchPanel::routeToSelectedLocation()
{
    qDebug() << __PRETTY_FUNCTION__;

    LocationListItem *item = dynamic_cast<LocationListItem *>
                             (m_locationListView->selectedItem());

    if (item)
        emit routeToLocation(item->coordinates());
}

void LocationSearchPanel::showEmptyPanel(bool show)
{
    if (show) {
        m_noSearchLabel->show();
        m_searchHistoryListView->hide();
    }
    else {
        m_noSearchLabel->hide();
    }
}

void LocationSearchPanel::setHeaderText(int count)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_resultsLabel->setText(tr("Search results: %1").arg(count));
}

void LocationSearchPanel::showLocationListView(int locationItemsCount)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_searchHistoryListView->clearItemSelection();
    m_searchHistoryListView->hide();
    setHeaderText(locationItemsCount);
    m_clearLocationListButton->setEnabled(true);
    m_locationListView->show();
    showEmptyPanel(false);
}

void LocationSearchPanel::showSearchHistoryListView()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationListView->clearList();
    m_locationListView->hide();
    m_resultsLabel->setText(tr("Search history:"));
    m_clearLocationListButton->setDisabled(true);
    m_searchHistoryListView->show();
}
