/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Katri Kaikkonen - katri.kaikkonen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "math.h"

#include <QDebug>
#include <QPainter>
#include <QSettings>

#include "common.h"
#include "panelcommon.h"

#include "indicatorbutton.h"

enum State {OFF, ON};                       ///< Enumerator for led state

IndicatorButton::IndicatorButton(QWidget *parent)
    : QToolButton(parent),
      m_drawTriangle(false),
      m_direction(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    const qreal OPACITY = 0.50;
    const int BUTTON_WIDTH = 66;
    const int BUTTON_HEIGHT = 66;

    m_indicatorLeds[OFF].load(":res/images/gps_position.png");
    m_indicatorLeds[ON].load(":res/images/gps_position_s.png");

    setFixedSize(BUTTON_WIDTH, BUTTON_HEIGHT);

    // Normal background
    m_normalColor = new QColor(Qt::black);
    m_normalColor->setAlpha(floor(OPACITY * 255));

    // Selected background
    m_selectedGradient = new QLinearGradient(0, 0, 0, this->height());
    m_selectedGradient->setColorAt(0.02, QColor(0, 113, 181));
    m_selectedGradient->setColorAt(0.25, QColor(24, 142, 214));
    m_selectedGradient->setColorAt(0.5, QColor(41, 162, 239));
    m_selectedGradient->setColorAt(0.75, QColor(82, 195, 255));
    m_selectedGradient->setColorAt(0.98, QColor(115, 215, 255));

    connect(this, SIGNAL(clicked(bool)),
            this, SIGNAL(autoCenteringTriggered(bool)));

    setCheckable(true);
}

IndicatorButton::~IndicatorButton()
{
    qDebug() << __PRETTY_FUNCTION__;

    delete m_normalColor;
    delete m_selectedGradient;
}

const QPoint& IndicatorButton::eventPosition()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_eventPosition;
}

void IndicatorButton::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QToolButton::mousePressEvent(event);

    event->ignore();
}

void IndicatorButton::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QToolButton::mouseReleaseEvent(event);

    event->ignore();
}

void IndicatorButton::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int ROUNDING_RADIUS = 9;

    Q_UNUSED(event);

    QPainterPath backgroundPath;
    backgroundPath.addRoundedRect(this->rect(), ROUNDING_RADIUS, ROUNDING_RADIUS);

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);

    if (isDown())
        painter.fillPath(backgroundPath, QBrush(*m_selectedGradient));
    else
        painter.fillPath(backgroundPath, QBrush(*m_normalColor));

    const QPointF CENTER = QPointF(this->width(), this->height()) / 2;

    if (isChecked()) {
        const QPointF offset = QPointF(m_indicatorLeds[ON].width(),
                                       m_indicatorLeds[ON].height()) / 2;

        painter.drawPixmap(CENTER - offset, m_indicatorLeds[ON]);
    } else {
        const QPointF offset = QPointF(m_indicatorLeds[OFF].width(),
                                       m_indicatorLeds[OFF].height()) / 2;

        painter.drawPixmap(CENTER - offset, m_indicatorLeds[OFF]);
    }

    // draw the direction indicator triangle only when autocentering is disabled and MapEngine
    // doesn't deny drawing (because GPS location item is visible)
    if (!isChecked() && m_drawTriangle) {
        const int TRIANGLE_WIDTH = 14;
        const int TRIANGLE_HEIGHT = 15;
        const int TRIANGLE_DISTANCE_FROM_CENTER = 13;

        const int POINTS = 3;
        const QPointF points[POINTS] = {
            QPointF(-TRIANGLE_WIDTH / 2, -TRIANGLE_DISTANCE_FROM_CENTER),
            QPointF(0, -(TRIANGLE_DISTANCE_FROM_CENTER + TRIANGLE_HEIGHT)),
            QPointF(TRIANGLE_WIDTH / 2, -TRIANGLE_DISTANCE_FROM_CENTER)
        };

        // base triangle is facing up, and needs to be rotated to the required direction
        QTransform rotationTransform;
        rotationTransform.rotate(m_direction);

        // origin is in the top left corner of the button, and needs to be translated to the
        // center of the button
        QTransform translateTransform;
        translateTransform.translate(CENTER.x(), CENTER.y());

        painter.setTransform(rotationTransform * translateTransform);

        // setting the look of the triangle
        const QColor TRIANGLE_BACKGROUND_COLOR = QColor("#68d0f8");
        painter.setBrush(QBrush(TRIANGLE_BACKGROUND_COLOR));
        painter.setPen(Qt::black);

        painter.drawPolygon(points, POINTS);
    }
}

void IndicatorButton::setDirection(qreal direction, bool draw)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_direction = direction;
    m_drawTriangle = draw;

    update();
}
