/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com
        Jussi Laitinen - jussi.laitinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QApplication>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>

#include "coordinates/geocoordinate.h"
#include "friendlistitem.h"
#include "friendlistitemdelegate.h"
#include "friendlistview.h"
#include "imagebutton.h"
#include "panelcommon.h"
#include "user/user.h"

#include "friendlistpanel.h"

FriendListPanel::FriendListPanel(QWidget *parent)
    : PanelBase(parent),
      m_mainWindowIsTopmost(false),
      m_somePanelIsOpen(false)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int FRIENDPANEL_FILTER_MARGIN_LEFT = PANEL_MARGIN_LEFT + 4;
    const int FRIENDPANEL_FILTER_MARGIN_TOP = 0;
    const int FRIENDPANEL_FILTER_MARGIN_RIGHT = PANEL_MARGIN_RIGHT + MAEMO5_SCROLLBAR_WIDTH + 4;
    const int FRIENDPANEL_FILTER_MARGIN_BOTTOM = 0;

    // --- HEADER, HOW MANY FRIENDS ARE SELECTED ---
    m_headerWidget = new QWidget();

    m_headerWidget->hide();
    m_headerWidget->setAutoFillBackground(true);

    QPalette headerPalette = m_headerWidget->palette();
    headerPalette.setColor(QPalette::Background, Qt::black);
    m_headerWidget->setPalette(headerPalette);

    QHBoxLayout *headerLayout = new QHBoxLayout();
    m_headerWidget->setLayout(headerLayout);
    headerLayout->setContentsMargins(FRIENDPANEL_FILTER_MARGIN_LEFT,
                                     FRIENDPANEL_FILTER_MARGIN_TOP,
                                     FRIENDPANEL_FILTER_MARGIN_RIGHT,
                                     FRIENDPANEL_FILTER_MARGIN_BOTTOM);

    m_headerLabel = new QLabel(this);
    headerLayout->addWidget(m_headerLabel, 0, Qt::AlignCenter);

    // --- FRIEND LIST ---
    m_friendListView = new FriendListView(this);
    m_friendListView->setItemDelegate(new FriendListItemDelegate(this));

    QVBoxLayout *listViewLayout = new QVBoxLayout;
    listViewLayout->setContentsMargins(PANEL_MARGIN_LEFT, PANEL_MARGIN_TOP,
                                       PANEL_MARGIN_RIGHT, PANEL_MARGIN_BOTTOM);
    listViewLayout->addWidget(m_friendListView);

    connect(m_friendListView, SIGNAL(friendItemClicked(GeoCoordinate)),
            this, SIGNAL(findFriend(GeoCoordinate)));

    connect(m_friendListView, SIGNAL(listItemSelectionChanged()),
            this, SLOT(onListItemSelectionChanged()));

    // --- FOOTER, TEXT BASED FILTERING ---
    QHBoxLayout *footerLayout = new QHBoxLayout();

    m_filterField = new QLineEdit;
    footerLayout->addWidget(m_filterField);

    connect(m_filterField, SIGNAL(returnPressed()),
            this, SLOT(clearTextFiltering()));

    connect(m_filterField, SIGNAL(textChanged(QString)),
            this, SLOT(filterTextChanged(QString)));

    m_clearTextFilteringButton = new QPushButton();
    footerLayout->addWidget(m_clearTextFilteringButton);
    m_clearTextFilteringButton->setIcon(QIcon::fromTheme(QLatin1String("general_close")));

    connect(m_clearTextFilteringButton, SIGNAL(clicked()),
            this, SLOT(clearTextFiltering()));

    connect(qApp, SIGNAL(topmostWindowChanged(bool)),
            this, SLOT(topmostWindowChanged(bool)));

    // --- MAIN LAYOUT ---
    QVBoxLayout *friendListPanelLayout = new QVBoxLayout();
    friendListPanelLayout->setMargin(0);
    friendListPanelLayout->setSpacing(0);
    setLayout(friendListPanelLayout);

    m_noFriendsLabel = new QLabel();
    m_noFriendsLabel->setText("No friends");
    m_noFriendsLabel->setAlignment(Qt::AlignCenter);

    QPalette noFriendsPalette = palette();
    noFriendsPalette.setColor(QPalette::Foreground, Qt::white);
    m_noFriendsLabel->setPalette(noFriendsPalette);

    friendListPanelLayout->addWidget(m_noFriendsLabel, Qt::AlignCenter);
    friendListPanelLayout->addWidget(m_headerWidget);
    friendListPanelLayout->addLayout(listViewLayout);
    friendListPanelLayout->addLayout(footerLayout);

    // --- CONTEXT BUTTONS ---
    m_routeButton = new ImageButton(":res/images/route_to_friend.png",
                                    ":res/images/route_to_friend_s.png",
                                    ":res/images/route_to_friend_d.png", this);
    connect(m_routeButton, SIGNAL(clicked()),
            this, SLOT(routeToSelectedFriend()));

    m_showContactButton = new ImageButton(":res/images/contact.png",
                                          ":res/images/contact_s.png",
                                          ":res/images/contact_d.png", this);
    connect(m_showContactButton, SIGNAL(clicked()),
            this, SLOT(requestSelectedFriendContactDialog()));

    m_clearGroupFilteringButton = new ImageButton(":res/images/filtered.png",
                                                  ":res/images/filtered_s.png",
                                                  ":res/images/filtered_d.png", this);
    m_clearGroupFilteringButton->setCheckable(true);
    m_clearGroupFilteringButton->setDisabled(true);
    connect(m_clearGroupFilteringButton, SIGNAL(clicked()),
            this, SLOT(clearFiltering()));

    m_itemButtonsLayout->addWidget(m_routeButton);
    m_itemButtonsLayout->addWidget(m_showContactButton);
    m_genericButtonsLayout->addWidget(m_clearGroupFilteringButton);

    showEmptyPanel(true);
}

void FriendListPanel::anyPanelClosed()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_somePanelIsOpen = false;
    updateKeyboardGrabbing();

    clearFiltering();

    m_friendListView->clearItemSelection();
}

void FriendListPanel::anyPanelOpened()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_somePanelIsOpen = true;
    updateKeyboardGrabbing();
}

void FriendListPanel::clearFiltering()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_headerWidget->hide();
    m_clearGroupFilteringButton->setChecked(false);
    m_clearGroupFilteringButton->setDisabled(true);
    m_friendListView->clearFilter();
    clearTextFiltering();
}

void FriendListPanel::clearTextFiltering()
{
    qDebug() << __PRETTY_FUNCTION__;

    // clearing the filtering text field does cause also hiding the filtering layout
    m_filterField->clear();
}

void FriendListPanel::filterTextChanged(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_filterField->isHidden() && !text.isEmpty())
        setFilteringLayoutVisibility(true);
    else if (m_filterField->isVisible() && text.isEmpty())
        setFilteringLayoutVisibility(false);

    m_friendListView->filter(text);
}

void FriendListPanel::friendImageReady(User *user)
{
    qDebug() << __PRETTY_FUNCTION__;

    FriendListItem *item = static_cast<FriendListItem*>(m_friendListView->listItem(user->userId()));

    if (item)
        item->setAvatarImage(user->profileImage());
}

void FriendListPanel::friendInfoReceived(QList<User *> &friendList)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!friendList.isEmpty())
           showEmptyPanel(false);

    QStringList newUserIDs;

    foreach (User *user, friendList) {
        FriendListItem *item = 0;
        if (!m_friendListView->contains(user->userId())) {
            item = new FriendListItem();
            item->setUserData(user);
            m_friendListView->addListItem(user->userId(), item);
        } else {
            item = static_cast<FriendListItem *>(m_friendListView->takeListItemFromView(
                    user->userId()));

            if (item) {
                item->setUserData(user);
                m_friendListView->addListItemToView(item);
            }
        }

        newUserIDs.append(user->userId());
    }

    m_friendListView->clearUnused(newUserIDs);
    m_friendListView->show();
}

void FriendListPanel::hideEvent(QHideEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QWidget::hideEvent(event);
    updateKeyboardGrabbing();
    clearFiltering();

    m_friendListView->clearItemSelection();
}

void FriendListPanel::requestSelectedFriendContactDialog()
{
     qDebug() << __PRETTY_FUNCTION__;

     FriendListItem *item = dynamic_cast<FriendListItem *>(m_friendListView->selectedItem());

     if (item) {
         QString facebookId = item->facebookId();
         if (!facebookId.isEmpty())
             emit requestContactDialog(facebookId);
     }
}

void FriendListPanel::routeToSelectedFriend()
{
    qDebug() << __PRETTY_FUNCTION__;

    FriendListItem *item = dynamic_cast<FriendListItem *>(m_friendListView->selectedItem());

    if (item)
        emit routeToFriend(item->coordinates());
}

void FriendListPanel::setFilteringLayoutVisibility(bool visible)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_filterField->setVisible(visible);
    m_clearTextFilteringButton->setVisible(visible);
}

void FriendListPanel::updateKeyboardGrabbing()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_mainWindowIsTopmost || !m_somePanelIsOpen || !isVisible()) {
        if (QWidget::keyboardGrabber() == m_filterField)
            m_filterField->releaseKeyboard();
    } else if (m_mainWindowIsTopmost && m_somePanelIsOpen && isVisible()) {
        if (QWidget::keyboardGrabber() != m_filterField)
            m_filterField->grabKeyboard();
    }
}

void FriendListPanel::showEvent(QShowEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    QWidget::showEvent(event);
    updateKeyboardGrabbing();
    setFilteringLayoutVisibility(false);
}

void FriendListPanel::showEmptyPanel(bool show)
{
    if (show) {
        m_noFriendsLabel->show();
        m_friendListView->hide();
    }
    else {
        m_noFriendsLabel->hide();
    }
}

void FriendListPanel::showFriendsInList(const QList<QString> &userIDs)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_headerLabel->setText(tr("Selected: %1").arg(userIDs.count()));

    m_headerWidget->show();
    m_clearGroupFilteringButton->setDisabled(false);
    m_clearGroupFilteringButton->setChecked(true);
    m_friendListView->filter(userIDs);

    clearTextFiltering();

    emit openPanelRequested(this);
}

void FriendListPanel::topmostWindowChanged(bool mainWindowIsTopmost)
{
    qDebug() << __PRETTY_FUNCTION__ << mainWindowIsTopmost;

    m_mainWindowIsTopmost = mainWindowIsTopmost;
    updateKeyboardGrabbing();
}
