/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef IMAGEFETCHER_H
#define IMAGEFETCHER_H

#include <QtCore>

class NetworkAccessManager;
class QNetworkReply;
class QUrl;

/**
* @brief ImageFetcher handles requests to get images.
*
* @author Henri Lampela
*/
class ImageFetcher : public QObject
{
    Q_OBJECT

public:

    /**
    * @brief Default constructor
    *
    * @param manager QNetworkAccessManager
    * @param parent Parent object
    */
    ImageFetcher(NetworkAccessManager *manager, QObject *parent = 0);

/*******************************************************************************
 * CLASS SPECIFIC MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:

    /**
    * @brief Fecth image from given url
    *
    * @param url Image url
    */
    void fetchImage(const QUrl &url);

private slots:

    /**
    * @brief This slot is called when network manager has finished
    * the download. Loads image and emits imageReceived signal with
    * url and image. If there was a error in reply emits error-signal.
    *
    * @param reply
    */
    void downloadFinished(QNetworkReply *reply);

    /**
    * @brief This slot is called when next download is started. Takes url
    * from queue, sends request and puts request to download queue.
    */
    void startNextDownload();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signal which is emitted when image
    * is received from the server and loaded to pixmap.
    *
    * @param url URL to image
    * @param image image QPixmap
    */
    void imageReceived(const QUrl &url, const QPixmap &image);

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QList<QNetworkReply*> m_currentDownloads; ///< List of current downloads
    QQueue<QUrl> m_downloadQueue;             ///< Queue of pending requests
    NetworkAccessManager *m_manager;       ///< Network access manager
};

#endif // IMAGEFETCHER_H
