/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Henri Lampela - henri.lampela@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "parser.h"

#include "coordinates/geocoordinate.h"
#include "../error.h"
#include "network/networkaccessmanager.h"

#include "geocodingservice.h"

GeocodingService::GeocodingService(QObject *parent)
    : QObject(parent)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_networkManager = new NetworkAccessManager(this);
    connect(m_networkManager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)), Qt::QueuedConnection);
}

void GeocodingService::parseSearchResults(const QByteArray &jsonReply)
{
    qDebug() << __PRETTY_FUNCTION__;

    const QString NO_DATA = "ZERO_RESULTS";

    m_searchResults.clear();

    QJson::Parser parser;
    bool ok;
    QVariantMap result = parser.parse(jsonReply, &ok).toMap();
    if (!ok) {
        emit error(ErrorContext::SITUARE, SituareError::ERROR_LOCATION_SEARCH_FAILED);
        return;
    } else if(result.value("status") != NO_DATA) {
        foreach(QVariant hitVariant, result["results"].toList()) {
            Location location;
            QMap<QString, QVariant> hitMap = hitVariant.toMap();
            location.setFormattedAddress(hitMap["formatted_address"].toString());

            // parse coordinates
            QVariant geometryVariant = hitMap["geometry"];
            QMap<QString, QVariant> geometryMap = geometryVariant.toMap();
            QVariant locationVariant = geometryMap["location"];
            QMap<QString, QVariant> locationMap = locationVariant.toMap();
            location.setCoordinates(GeoCoordinate(locationMap["lat"].toDouble(),
                                                  locationMap["lng"].toDouble()));

            // parse recommended viewport
            QVariant viewportVariant = geometryMap["viewport"];
            QMap<QString, QVariant> viewportMap = viewportVariant.toMap();
            QVariant swVariant = viewportMap["southwest"];
            QMap<QString, QVariant> swMap = swVariant.toMap();
            QVariant neVariant = viewportMap["northeast"];
            QMap<QString, QVariant> neMap = neVariant.toMap();

            GeoCoordinate southwest = GeoCoordinate(swMap["lat"].toDouble(),
                                                    swMap["lng"].toDouble());

            GeoCoordinate northeast = GeoCoordinate(neMap["lat"].toDouble(),
                                                    neMap["lng"].toDouble());

            location.setViewPort(southwest, northeast);

            m_searchResults.append(location);
        }

        emit locationDataParsed(m_searchResults);
    } else {
        emit error(ErrorContext::SITUARE, SituareError::ERROR_LOCATION_SEARCH_FAILED);
    }
}

void GeocodingService::requestFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    QByteArray replyArray = reply->readAll();

    if (reply->error())
        emit error(ErrorContext::NETWORK, reply->error());
    else
        parseSearchResults(replyArray);

    reply->deleteLater();
}

void GeocodingService::requestLocation(const QString &location)
{
    qDebug() << __PRETTY_FUNCTION__;

    const QString GEOCODE_SERVICE_ADDRESS = "http://maps.google.com/maps/api/geocode/json?address=";
    const QString DEVICE_CONTAINS_LOCATION_SENSOR = "&sensor=true";

    QString url = GEOCODE_SERVICE_ADDRESS;
    url.append(location);
    url.append(DEVICE_CONTAINS_LOCATION_SENSOR);

    sendRequest(url);
}

void GeocodingService::sendRequest(const QUrl &url)
{
    qDebug() << __PRETTY_FUNCTION__;

    QNetworkRequest request;

    request.setUrl(url);
    request.setAttribute(QNetworkRequest::CacheSaveControlAttribute, false);

    QByteArray ba;
    m_networkManager->post(request, ba, false);
}
