/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAPTILE_H
#define MAPTILE_H

#include <QGraphicsPixmapItem>

class SceneCoordinate;

/**
* @brief Map tile item
*
* @author Sami Rämö - sami.ramo (at) ixonos.com
*/
class MapTile : public QGraphicsPixmapItem
{
public:
    /**
    * @brief Constructor
    */
    MapTile();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
     * @brief Convert tile x & y numbers to MapScene coordinates
     *
     * @param zoomLevel Zoom level
     * @param tileNumber x & y numbers of the tile
     * @return Scene coordinate
     */
    static SceneCoordinate convertTileNumberToSceneCoordinate(int zoomLevel, QPoint tileNumber);

    /**
     * @brief Calculate maximum value for tile in this zoom level.
     *
     * @param zoomLevel zoom level
     * @return int tile's maximum value
     */
    static int lastTileIndex(int zoomLevel);

    /**
    * @brief Set drawing level of the tile based on current zoom level
    *
    * Drawing order of MapTiles, which has the zoom level higher than the current
    * zoom level, is reversed and those MapTiles are mapped between lower level MapTiles.
    * Example: If maximum zoom level is 18 and current view zoomlevel is 15, then
    * the drawing order from top to bottom is 15, 16, 14, 17, 13, 18, 12, 11, 10, ...
    * @param currentZoomLevel current zoom level
    */
    void setSceneLevel(int currentZoomLevel);

    /**
    * @brief Setter for tile number
    *
    * Does also set the position for the item in the MapScene coordinate system
    * @param tileNumber Tile number
    */
    void setTileNumber(QPoint tileNumber);

    /**
    * @brief Setter for zoom level
    *
    * Does also set the zValue.
    *
    * @param tileZoomLevel Zoom level of the tile
    * @param currentViewZoomLevel Current view zoom level
    */
    void setZoomLevel(int tileZoomLevel, int currentViewZoomLevel);

    /**
    * @brief Getter for tile number
    *
    * @return Tile number
    */
    QPoint tileNumber();

    /**
     * @brief Return tile path created from tile values.
     *
     * @param zoomLevel tile's zoom level
     * @param x tile's x value
     * @param y tile's y value
     * @return QString tile path
     */
    static QString tilePath(int zoomLevel, int x, int y);

    /**
     * @brief Maximum number of individual tiles per side at given zoom level
     *
     * @param zoomLevel Zoom level
     * @return amount of tiles per side at given zoom level
     */
    static int tilesPerSide(int zoomLevel);

    /**
    * @brief Getter for zoom level
    *
    * @return Zoom level
    */
    int zoomLevel();

private:
    /**
    * @brief Set position of the tile in the MapScene coordinate system
    *
    * Does set the position based on the m_zoomLevel and the m_TileNumber. Position is set to
    * (UNDEFINED, UNDEFINED) if there is something wrong with zoom level or tile numbers
    */
    void setPosition();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    int m_zoomLevel;        ///< Zoom level

    QPoint m_tileNumber;    ///< Tile number
};

#endif // MAPTILE_H
