/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAPENGINE_H
#define MAPENGINE_H

#include <QtCore>

#include "routing/route.h"
#include "coordinates/geocoordinate.h"
#include "coordinates/scenecoordinate.h"

class QGraphicsScene;

class FriendItemsHandler;
class GPSLocationItem;
class MapFetcher;
class MapRouteItem;
class MapScene;
class MapScroller;
class MapTile;
class OwnLocationItem;
class User;

/**
 * @brief Map engine
 *
 * Logic for controlling map functionality. Does also include static methods for
 * converting coordinates.
 *
 * NOTE: MapEngine must have a view attached before using it!
 *
 * @author Sami Rämö - sami.ramo (at) ixonos.com
 * @author Jussi Laitinen - jussi.laitinen (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
 * @author Ville Tiensuu - ville.tiensuu (at) ixonos.com
 */
class MapEngine : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     *
     * @param parent Parent
     */
    MapEngine(QObject *parent = 0);

    /**
     * @brief Destructor
     * Saves view of the map to settings file
     */
    ~MapEngine();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
     * @brief Coordinates of the current center point
     *
     * @return Current coordinates
     */
    GeoCoordinate centerGeoCoordinate();

    /**
     * @brief Convert MapScene coordinate to tile x & y numbers.
     *
     * @param zoomLevel ZoomLevel
     * @param coordinate MapScene coordinate
     * @return tile x & y numbers
     */
    static QPoint convertSceneCoordinateToTileNumber(int zoomLevel, SceneCoordinate coordinate);

    /**
     * @brief Set initial values for the map
     *
     * Set initial location and zoom level from the settings, if available, or use the default
     * values set in the constructor. Signals locationChanged() and zoomLevelChanged() are emitted,
     * so init should be called after those signals are connected.
     */
    void init();

    /**
     * @brief Return given value translated between min and max
     *
     * If given value is not inside the given range (min <= value <= max), then the allowed range
     * is adder or subtracted until the value does fit in the range.
     *
     * @param value Value to be normalized
     * @param min Minimum allowed value
     * @param max Maximum allowed value
     * @return value which is moved to be inside the given limits
     */
    static int normalize(int value, int min, int max);

    /**
     * @brief Getter for scene
     *
     * @return QGraphicsScene
     */
    QGraphicsScene* scene();

    /**
     * @brief Sets new zoom level
     *
     * @return newZoomLevel value that is set to new zoom level
     */
    void setZoomLevel(const int newZoomLevel);

public slots:
    /**
     * @brief Center smoothly to given latitude and longitude coordinates.
     *
     * @param coordinate Latitude & longitude coordinates for location
     */
    void centerToCoordinates(GeoCoordinate coordinate);

    /**
     * @brief Slot to catch user own location data
     *
     * @param user User info
     */
    void receiveOwnLocation(User *user);

    /**
     * @brief Set auto centering.
     *
     * @param enabled true if enabled, false otherwise
     */
    void setAutoCentering(bool enabled);

    /**
     * @brief Slot for enabling / disabling GPS
     *
     * GPS location item is disabled or enabled based on GPS state
     *
     * @param enabled True is GPS is enabled, otherwise false
     */
    void setGPSEnabled(bool enabled);

    /**
     * @brief Slot for view resizing.
     *
     * @param size view size
     */
    void viewResized(const QSize &size);

private:
    /**
     * @brief Calculate grid of tile coordinates from current scene coordinate.
     *
     * Grid size is calculated from view size and scene's current center coordinate.
     *
     * @param coordinate scene's current center coordinate
     * @return QRect grid of tile coordinates
     */
    QRect calculateTileGrid(SceneCoordinate coordinate);

    /**
    * @brief Center and zoom to given rect
    *
    * Map is centered to center point of the rect and zoomed so that whole rect is visible
    * as big as possible.
    *
    * @param rect Target rect
    * @param useMargins true if margins should be added to rect, false otherwise
    */
    void centerAndZoomTo(QRect rect, bool useMargins = true);

    /**
    * @brief Returns the rect of the current scene area drawn on the view
    *
    * @returns Rect of the current scene area drawn on the view
    */
    QRectF currentViewSceneRect() const;

    /**
     * @brief Request disabling of auto centering if centered too far from the real location.
     *
     * @param coordinate scene's center coordinate
     */
    void disableAutoCenteringIfRequired(SceneCoordinate coordinate);

    /**
     * @brief Get new tiles.
     *
     * Calculates which tiles has to be fetched. Does emit fetchImage for tiles which
     * aren't already in the scene.
     * @param coordinate scene's center coordinate
     */
    void getTiles(SceneCoordinate coordinate);

    /**
     * @brief Check if auto centering is enabled
     *
     * @return true if enabled, false otherwise
     */
    bool isAutoCenteringEnabled();

    /**
     * @brief Check if center tile has changed.
     *
     * @param coordinate scene's center coordinate
     * @return bool true if center tile changed, false otherwise
     */
    bool isCenterTileChanged(SceneCoordinate coordinate);

    /**
     * @brief Set size of tiles grid based on view size
     *
     * @param viewSize Current view size
     */
    void setTilesGridSize(const QSize &viewSize);

    /**
     * @brief Updates direction and distance values to indicator button
     *
     * Calculate and update direction and distance values and send those to indicator button
     */
    void updateDirectionIndicator();

    /**
     * @brief Updates the current view rect including margins
     *
     * Calculates tiles rect in scene based on m_viewTilesGrid and
     * calls MapScene::viewRectUpdated()
     */
    void updateViewTilesSceneRect();

    /**
     * @brief Calculate scale at the map center of the map in meters/pixel
     *
     * @return qreal Scale of the map in meters/pixel
     */
    qreal viewResolution();

    /**
     * @brief This method is ran always when the map is zoomed
     *
     * This method is the right place for all actions which must be done when ever map is zoomed.
     */
    void zoomed();

private slots:
    /**
    * @brief Slot for clearing the current route
    */
    void clearRoute();

    /**
     * @brief This slot is called after friend items position have been updated
     *
     * Does run MapScene::spanItems()
     */
    void friendsPositionsUpdated();

    /**
     * @brief Slot for GPS position updates
     *
     * GPS location item is updated and map centered to new location (if automatic
     * centering is enabled).
     *
     * @param position New coordinates from GPS
     * @param accuracy Accuracy of the GPS fix
     */
    void gpsPositionUpdate(GeoCoordinate position, qreal accuracy);

    /**
     * @brief Slot for received map tile images
     *
     * Does receive map tile images from MapFetcher. Calls MapScene::addTile() for creating and adding
     * the actual MapTile object to the MapScene.
     *
     * Tile is added also to outside the world horizontal limits, if required, for spanning the map.
     *
     * @param zoomLevel Zoom level
     * @param x Tile x index
     * @param y Tile y index
     * @param image Received pixmap
     */
    void mapImageReceived(int zoomLevel, int x, int y, const QPixmap &image);

    /**
     * @brief Called when MapScroller state is changed
     *
     * Does check if the smooth scroll effect was interrupted and should the auto centering
     * feature to be disabled.
     *
     * @param newState New state
     */
    void scrollerStateChanged(QAbstractAnimation::State newState);

    /**
     * @brief Scroll smoothly to given scene position
     *
     * @param coordinate Target position in the scene
     */
    void scrollToPosition(SceneCoordinate coordinate);

    /**
     * @brief Set center point in the scene
     *
     * Does emit locationChanged signal.
     * @param coordinate Scene coordinates for new position
     */
    void setCenterPosition(SceneCoordinate coordinate);

    /**
     * @brief Builds and sets route, also centers it
     *
     * @param route Route route information
     */
    void setRoute(Route &route);

    /**
    * @brief Shows map area defined by bounds.
    *
    * Calls centerAndZoomTo() with area defined by bounds.
    * @param swBound south-west bound of location item
    * @param neBound north-east bound of location item
    */
    void showMapArea(const GeoCoordinate &swBound, const GeoCoordinate &neBound);

    /**
     * @brief Slot for actions after view zoom is finished
     *
     * Does run removeOutOfViewTiles
     */
    void viewZoomFinished();

    /**
     * @brief Slot for zooming in
     */
    void zoomIn();

    /**
     * @brief Slot for zooming out
     */
    void zoomOut();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signal when direction and distance from current map center point to current GPS
    *        location is changed
    *
    * @param direction Direction in degrees
    * @param distance Distance in meters
    * @param draw Should the indicator triangle be drawn or not
    */
    void directionIndicatorValuesUpdate(qreal direction, qreal distance, bool draw);

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
     * @brief Signal for image fetching.
     *
     * @param zoomLevel Zoom level
     * @param x Tile x index
     * @param y Tile y index
     */
    void fetchImage(int zoomLevel, int x, int y);

    /**
    * @brief Signal when friend image is ready
    *
    * @param user Friend
    */
    void friendImageReady(User *user);

    /**
     * @brief Signal when friend list locations are fetched
     *
     * @param friendsList Friends list data
     */
    void friendsLocationsReady(QList<User *> &friendsList);

    /**
     * @brief Request view centering to new locaiton
     *
     * @param coordinate New scene coordinates
     */
    void locationChanged(SceneCoordinate coordinate);

    /**
     * @brief Signal is emitted when location item is clicked.
     *
     * @param userIDs list of friends user IDs in the group
     */
    void locationItemClicked(const QList<QString> &userIDs);

    /**
     * @brief Signal to notify map scrolling.
     */
    void mapScrolledManually();

    /**
     * @brief Signal to notify when map is zoomed in to the maxmimum.
     */
    void maxZoomLevelReached();

    /**
     * @brief Signal to notify when map is zoomed out to the minimum.
     */
    void minZoomLevelReached();

    /**
     * @brief Signal to pass the scale of the map to map scale
     */
    void newMapResolution(qreal scale);

    /**
     * @brief Request view changing zoom level
     *
     * @param newZoomLevel New zoom level
     */
    void zoomLevelChanged(int newZoomLevel);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_autoCenteringEnabled;    ///< Auto centering enabled
    bool m_scrollStartedByGps;      ///< Smooth scroll is started by GPS?
    bool m_smoothScrollRunning;     ///< Smooth scroll is running?
    bool m_zoomedIn;                ///< Flag for checking if zoomed in when zoom is finished

    int m_zoomLevel;                ///< Current zoom level

    QPoint m_centerTile;            ///< Current center tile
    SceneCoordinate m_lastAutomaticPosition; ///< Last automatically set position in scene coordinate
    SceneCoordinate m_sceneCoordinate;       ///< Current center coordinate

    QRect m_viewTilesGrid;          ///< Current grid of tiles in view (includes margin)

    QSize m_tilesGridSize;          ///< Current size of the tiles grid
    QSize m_viewSize;               ///< Current view size

    FriendItemsHandler *m_friendItemsHandler;   ///< Handler for friend and group items
    GeoCoordinate m_gpsPosition;                ///< Latest GPS position
    GPSLocationItem *m_gpsLocationItem;         ///< Item pointing current location from GPS
    MapFetcher *m_mapFetcher;                   ///< Fetcher for map tiles
    MapRouteItem *m_mapRouteItem;               ///< Map route item
    MapScene *m_mapScene;                       ///< Scene for map tiles
    MapScroller *m_scroller;                    ///< Kinetic scroller
    OwnLocationItem *m_ownLocation;             ///< Item to show own location
};

#endif // MAPENGINE_H
