/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef GPSPOSITION_H
#define GPSPOSITION_H

#include <QObject>

#include "coordinates/geocoordinate.h"

class GPSPositionPrivate;

/**
* @brief GPSPosition class is an interface for GPS.
*/
class GPSPosition : public QObject
{
    Q_OBJECT

public:
    /**
    * @brief Friend class for GPSPosition.
    */
    friend class GPSPositionPrivate;

    /**
    * @brief Constructor.
    *
    * @param parent QObject
    */
    GPSPosition(QObject *parent = 0);

    /**
    * @brief GPS position mode.
    */
    enum Mode {Default, Simulation};

/******************************************************************************
* MEMBER FUNCTIONS AND SLOTS
******************************************************************************/
public:
    /**
    * @brief Enables power save mode.
    *
    * Starts GPS for position update and then stops it.
    */
    void enablePowerSave(bool enabled);

    /**
    * @brief Returns is GPS initialized.
    *
    * @return true if initialized, false otherwise
    */
    bool isInitialized() const;

    /**
    * @brief Checks if GPS is running.
    *
    * @return true if GPS running, false otherwise
    */
    bool isRunning() const;

    /**
    * @brief Return last known position.
    */
    GeoCoordinate lastPosition() const;

    /**
    * @brief Informs gps to emit last known position.
    */
    void requestLastPosition();

    /**
    * @brief Requests update from GPS.
    *
    * Enables GPS if it is disabled
    */
    void requestUpdate();

    /**
    * @brief Set GPS mode.
    *
    * Modes: default and simulation.
    * @param mode GPS mode
    * @param filePath file path to NMEA file if simulation mode is used
    */
    void setMode(Mode mode, const QString &filePath = 0);

    /**
    * @brief Set GPS update interval
    *
    * @return interval interval in milliseconds
    */
    void setUpdateInterval(int interval);

    /**
    * @brief Start GPS.
    */
    void start();

    /**
    * @brief Stop GPS.
    */
    void stop();

/******************************************************************************
* SIGNALS
******************************************************************************/
signals:
    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
    * @brief Signal for position information.
    *
    * @param coordinate latitude and longitude values
    * @param accuracy accuracy in metres
    */
    void position(GeoCoordinate coordinate, qreal accuracy);

    /**
    * @brief Signal for timeout.
    */
    void timeout();

/*******************************************************************************
* DATA MEMBERS
*******************************************************************************/
private:
    GPSPositionPrivate *m_gpsPositionPrivate;   ///< GPSPositionPrivate object
};

#endif // GPSPOSITIONINTERFACE_H
