 /*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com
        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
 */


#ifndef ENGINE_H
#define ENGINE_H

#include <QObject>
#include <QTime>

#include "coordinates/geocoordinate.h"

class QTimer;

class Application;
class ContactManager;
class FacebookAuthentication;
class FacebookCredentials;
class GeocodingService;
class GPSPosition;
class Location;
class MainWindow;
class MapEngine;
class MCE;
class NetworkAccessManager;
class Route;
class RoutingService;
class SituareService;
class User;

/**
* @brief Engine class for Situare Application
*
* This class handles all the underlaying login of the Situare
* application.
*/
class SituareEngine : public QObject
{
    Q_OBJECT
public:
    /**
    * @brief Constructor
    *
    */
    SituareEngine();

    /**
    * @brief Destructor
    */
    ~SituareEngine();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:
    /**
    * @brief Slot to intercept error signal from ImageFetcher and SituareService
    *
    * @param context Error context
    * @param error Error message
    */
    void error(const int context, const int error);

    /**
    * @brief Slot to intercept signal when username is fetched from settings
    *
    */
    void fetchUsernameFromSettings();

    /**
    * @brief Slot to intercept signal when location search is issued
    *
    * @param location QString location
    */
    void locationSearch(QString location);

    /**
    * @brief Slot to intercept signal when Login/Logout action is pressed
    *
    */
    void loginActionPressed();

    /**
    * @brief Slot to intercept signal from successful login
    *
    */
    void loginOk();

    /**
    * @brief Slot to intercept signal when user has cancelled login process
    */
    void loginProcessCancelled();

    /**
    * @brief Changes application state when logged out
    *
    */
    void logout();

    /**
    * @brief Calls reverseGeo from SituareService to translate coordinates to street address
    *
    */
    void requestAddress();

    /**
    * @brief Calls updateLocation from SituareService to send the location update to
    *        Situare server.
    *
    * @param status Status message
    * @param publish Publish on Facebook
    */
    void requestUpdateLocation(const QString &status = QString(), bool publish = false);

    /**
    * @brief Slot to refresh user data
    */
    void refreshUserData();

    /**
    * @brief Slot to intercept signal from successful location update
    *
    */
    void updateWasSuccessful();

    /**
    * @brief Slot to intercept signal when new user data is available.
    *        Splits User and friendsList data and emits them as two different signals.
    *
    * @param user instance of User
    * @param friendsList list of User instances (friends)
    */
    void userDataChanged(User *user, QList<User *> &friendsList);

private:
    /**
    * @brief Read settings and determine whether to use GPS and autocentering.
    * When values does not found on the settings, GPS and autocentering are enabled as a default.
    */
    void initializeGpsAndAutocentering();

    /**
      * @brief Connect signals coming from Facebook authenticator
      */
    void signalsFromFacebookAuthenticator();

    /**
      * @brief Connect signals coming from GeocodingService
      */
    void signalsFromGeocodingService();

    /**
      * @brief Connect signals coming from GPS
      */
    void signalsFromGPS();

    /**
      * @brief Connect signals coming from MainWindow
      */
    void signalsFromMainWindow();

    /**
      * @brief Connect signals coming from MapEngine
      */
    void signalsFromMapEngine();

    /**
      * @brief Connect signals coming from MapView
      */
    void signalsFromMapView();

    /**
      * @brief Connect signals coming from RoutingService
      */
    void signalsFromRoutingService();

    /**
      * @brief Connect signals coming from Situare
      */
    void signalsFromSituareService();

private slots:
    /**
    * @brief Set auto centering feature enabled / disabled
    *
    * @param enabled true if enabled, false otherwise
    */
    void changeAutoCenteringSetting(bool enabled);

    /**
    * @brief Slot for disabling automatic centering when map is scrolled manually
    */
    void disableAutoCentering();

    /**
    * @brief Calls vibration feedback.
    */
    void draggingModeTriggered();

    /**
    * @brief Enables automatic location update.
    *
    * @param enabled true if enabled, false otherwise
    * @param updateIntervalMsecs update interval in milliseconds
    */
    void enableAutomaticLocationUpdate(bool enabled, int updateIntervalMsecs = 0);

    /**
    * @brief Slot to intercept signal when user's/friend's image is downloaded
    *
    * @param user Instance of user/friend
    */
    void imageReady(User *user);

    /**
    * @brief Requests automatic update.
    *
    * Makes automatic location update request if user has moved enough.
    *
    * @param position geo coordinates
    */
    void requestAutomaticUpdateIfMoved(GeoCoordinate position);

    /**
    * @brief Route is parsed and is ready for further processing.
    *
    * @param route Route item containing parsed route details
    */
    void routeParsed(Route &route);

    /**
    * @brief Routes to geo coordinates.
    *
    * Uses map center coordinates as start point.
    * @param endPointCoordinates end point geo coordinates
    */
    void routeTo(const GeoCoordinate &endPointCoordinates);

    /**
    * @brief Route to current cursor position
    */
    void routeToCursor();

    /**
    * @brief Slot for setting auto centering state.
    *
    * Calls gps to send last known position
    *
    * @param enabled true if auto centering was enabled, false otherwise
    */
    void setAutoCentering(bool enabled);

    /**
     * @brief Sets zoom level to default when first GPS location is received if autocentering
     * is enabled.
     */
    void setFirstStartZoomLevel();

    /**
    * @brief Slot for setting GPS state.
    *
    * @param enabled true if gps should be enabled, false otherwise
    */
    void setGPS(bool enabled);

    /**
    * @brief Slot for setting power saving state.
    *
    * @param enabled true if enabled, false otherwise
    */
    void setPowerSaving(bool enabled);

    /**
    * @brief Shows contact dialog.
    *
    * Calls MainWindow showContactDialog with contact guid defined by contact's Facebook ID.
    * @param facebookId contact's facebookId
    */
    void showContactDialog(const QString &facebookId);

    /**
    * @brief Automatic update interval timer timeout.
    *
    * Requests update location if user has moved.
    */
    void startAutomaticUpdate();

    /**
    * @brief Called when topmost window is changed
    *
    * Does set power saving state.
    *
    * @param isMainWindow True if MainWindow is the topmost one
    */
    void topmostWindowChanged(bool isMainWindow);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signals when automatic location update was enabled.
    *
    * @param enabled true if enabled, false otherwise
    */
    void automaticLocationUpdateEnabled(bool enabled);

    /**
    * @brief Signal to clear locationUpdateDialog's data
    *
    */
    void clearUpdateLocationDialogData();

    /**
    * @brief Signal when direction and distance from current map center point to current GPS
    *        location is changed
    *
    * @param direction Direction in degrees
    * @param distance Distance in meters
    * @param draw Should the indicator triangle be drawn or not
    */
    void directionIndicatorValuesUpdate(qreal direction, qreal distance, bool draw);

    /**
    * @brief Signals when new friends data is ready
    *
    * @param friendList List of User instances (friends)
    */
    void friendsLocationsReady(QList<User *> &friendList);

    /**
    * @brief Signals when friend's image is ready
    *
    * @param user Instance of friend
    */
    void friendImageReady(User *user);

    /**
    * @brief Emited when location request is parsed and is ready for further processing
    *
    * @param result List of Location items
    */
    void locationDataParsed(QList<Location> &result);

    /**
    * @brief Signals when new user data is ready
    *
    * @param user Instance of User
    */
    void userLocationReady(User *user);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_autoCenteringEnabled;        ///< Auto centering flag
    bool m_automaticUpdateFirstStart;   ///< Automatic location update first start flag
    bool m_automaticUpdateRequest;      ///< Flag for automatic update request
    bool m_userMoved;                   ///< Flag for user move

    QTimer *m_automaticUpdateIntervalTimer; ///< Automatic update interval timer

    ContactManager *m_contactManager;                ///< Instance of contact manager
    FacebookAuthentication *m_facebookAuthenticator; ///< Instance for facebook authenticator
    GeocodingService *m_geocodingService;            ///< Instance of the geocoding service
    GeoCoordinate m_lastUpdatedGPSPosition;          ///< Last updated GPS position
    GPSPosition *m_gps;                              ///< Instance of the gps position
    MainWindow *m_ui;                                ///< Instance of the MainWindow UI
    MapEngine *m_mapEngine;                          ///< MapEngine
    NetworkAccessManager *m_networkAccessManager;    ///< NetworkAccessManager
    RoutingService *m_routingService;  ///< Instance of the routing service
    SituareService *m_situareService;  ///< Instance of the situare server communication service
    MCE *m_mce;                        ///< Instance of the MCE
};

#endif // ENGINE_H
