 /*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com
        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
 */

#include <cmath>

#include <QMessageBox>
#include <QNetworkReply>

#include "application.h"
#include "common.h"
#include "contactmanager.h"
#include "../error.h"
#include "facebookservice/facebookauthentication.h"
#include "gps/gpsposition.h"
#include "map/mapengine.h"
#include "routing/geocodingservice.h"
#include "routing/routingservice.h"
#include "mce.h"
#include "network/networkaccessmanager.h"
#include "situareservice/situareservice.h"
#include "ui/mainwindow.h"

#include "engine.h"

const QString SETTINGS_GPS_ENABLED = "GPS_ENABLED"; ///< GPS setting
const QString SETTINGS_AUTO_CENTERING_ENABLED = "AUTO_CENTERING_ENABLED";///< Auto centering setting
const int DEFAULT_ZOOM_LEVEL_WHEN_GPS_IS_AVAILABLE = 12;  ///< Default zoom level when GPS available
const qreal USER_MOVEMENT_MINIMUM_LONGITUDE_DIFFERENCE = 0.003;///< Min value for user move latitude
const qreal USER_MOVEMENT_MINIMUM_LATITUDE_DIFFERENCE = 0.001;///< Min value for user move longitude
const int MIN_UPDATE_INTERVAL_MSECS = 5*60*1000;

SituareEngine::SituareEngine()
    : m_autoCenteringEnabled(false),
      m_automaticUpdateFirstStart(true),
      m_automaticUpdateRequest(false),
      m_userMoved(false),
      m_automaticUpdateIntervalTimer(0),
      m_lastUpdatedGPSPosition(GeoCoordinate())
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui = new MainWindow;
    m_ui->updateItemVisibility();

    Application *application = static_cast<Application *>(qApp);
    application->registerWindow(m_ui->winId());

    connect(application, SIGNAL(topmostWindowChanged(bool)),
            this, SLOT(topmostWindowChanged(bool)));

    m_networkAccessManager = new NetworkAccessManager(this);

    // build MapEngine
    m_mapEngine = new MapEngine(this);
    m_ui->setMapViewScene(m_mapEngine->scene());

    // build GPS
    m_gps = new GPSPosition(this);

    // build SituareService
    m_situareService = new SituareService(this);

    // build FacebookAuthenticator
    m_facebookAuthenticator = new FacebookAuthentication(this);

    // build routing service
    m_routingService = new RoutingService(this); // create this when needed, not in constructor!

    // build geocoding service
    m_geocodingService = new GeocodingService(this);

    // connect signals
    signalsFromMapEngine();
    signalsFromGeocodingService();
    signalsFromGPS();
    signalsFromRoutingService();
    signalsFromSituareService();
    signalsFromMainWindow();
    signalsFromFacebookAuthenticator();

    connect(this, SIGNAL(userLocationReady(User*)),
            m_ui, SIGNAL(userLocationReady(User*)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_ui, SIGNAL(friendsLocationsReady(QList<User*>&)));

    connect(this, SIGNAL(userLocationReady(User*)),
            m_mapEngine, SLOT(receiveOwnLocation(User*)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_mapEngine, SIGNAL(friendsLocationsReady(QList<User*>&)));

    connect(this, SIGNAL(friendImageReady(User*)),
            m_ui, SIGNAL(friendImageReady(User*)));

    connect(this, SIGNAL(friendImageReady(User*)),
            m_mapEngine, SIGNAL(friendImageReady(User*)));

    m_automaticUpdateIntervalTimer = new QTimer(this);
    connect(m_automaticUpdateIntervalTimer, SIGNAL(timeout()),
            this, SLOT(startAutomaticUpdate()));

    // signals connected, now it's time to show the main window
    // but init the MapEngine before so starting location is set
    m_mapEngine->init();
    m_ui->show();

    m_facebookAuthenticator->start();

    m_gps->setMode(GPSPosition::Default);
    initializeGpsAndAutocentering();

    m_mce = new MCE(this);
    connect(m_mce, SIGNAL(displayOff(bool)), this, SLOT(setPowerSaving(bool)));
    
    m_contactManager = new ContactManager(this);
    m_contactManager->requestContactGuids();
}

SituareEngine::~SituareEngine()
{
    qDebug() << __PRETTY_FUNCTION__;

    delete m_ui;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(SETTINGS_GPS_ENABLED, m_gps->isRunning());
    settings.setValue(SETTINGS_AUTO_CENTERING_ENABLED, m_autoCenteringEnabled);
}

void SituareEngine::changeAutoCenteringSetting(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__ << enabled;

    m_autoCenteringEnabled = enabled;
    setAutoCentering(enabled);
}

void SituareEngine::disableAutoCentering()
{
    qDebug() << __PRETTY_FUNCTION__;

    changeAutoCenteringSetting(false);
}

void SituareEngine::draggingModeTriggered()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_mce)
        m_mce->vibrationFeedback();
}

void SituareEngine::enableAutomaticLocationUpdate(bool enabled, int updateIntervalMsecs)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Show automatic update confirmation dialog
    if (m_automaticUpdateFirstStart && m_gps->isRunning() && enabled) {
        m_ui->showEnableAutomaticUpdateLocationDialog(
                tr("Do you want to enable automatic location update with %1 min update interval?")
                .arg(updateIntervalMsecs/1000/60));
        m_automaticUpdateFirstStart = false;
    } else {
        if (enabled && m_gps->isRunning()) {
            m_ui->buildInformationBox(tr("Automatic location update enabled"));
            if (updateIntervalMsecs < MIN_UPDATE_INTERVAL_MSECS)
                m_automaticUpdateIntervalTimer->setInterval(MIN_UPDATE_INTERVAL_MSECS);
            else
                m_automaticUpdateIntervalTimer->setInterval(updateIntervalMsecs);

            connect(m_gps, SIGNAL(position(GeoCoordinate, qreal)),
                    this, SLOT(requestAutomaticUpdateIfMoved(GeoCoordinate)));

            m_automaticUpdateIntervalTimer->start();

        } else {
            disconnect(m_gps, SIGNAL(position(GeoCoordinate, qreal)),
                    this, SLOT(requestAutomaticUpdateIfMoved(GeoCoordinate)));

            m_automaticUpdateIntervalTimer->stop();
        }
    }
}

void SituareEngine::error(const int context, const int error)
{
    qDebug() << __PRETTY_FUNCTION__;

    switch(error)
    {
    case SituareError::ERROR_GENERAL:
        if(context == ErrorContext::SITUARE) {
            m_ui->toggleProgressIndicator(false);
            m_ui->buildInformationBox(tr("Unknown server error"), true);
        }
        break;
    case 1: //errors: SituareError::ERROR_MISSING_ARGUMENT and QNetworkReply::ConnectionRefusedError
        m_ui->toggleProgressIndicator(false);
        if(context == ErrorContext::SITUARE) {
            m_ui->buildInformationBox(tr("Missing parameter from request"), true);
        } else if(context == ErrorContext::NETWORK) {
            m_ui->buildInformationBox(tr("Connection refused by the server"), true);
        }
        break;
    case QNetworkReply::RemoteHostClosedError:
        if(context == ErrorContext::NETWORK) {
            m_ui->toggleProgressIndicator(false);
            m_ui->buildInformationBox(tr("Connection closed by the server"), true);
        }
        break;
    case QNetworkReply::HostNotFoundError:
        if(context == ErrorContext::NETWORK) {
            m_ui->toggleProgressIndicator(false);
            m_ui->buildInformationBox(tr("Remote server not found"), true);
        }
        break;
    case QNetworkReply::TimeoutError:
        if(context == ErrorContext::NETWORK) {
            m_ui->toggleProgressIndicator(false);
            m_ui->buildInformationBox(tr("Connection timed out"), true);
        }
        break;
    case QNetworkReply::UnknownNetworkError:
        if(context == ErrorContext::NETWORK) {
            m_ui->toggleProgressIndicator(false);
            m_ui->buildInformationBox(tr("No network connection"), true);
        }
        break;
    case SituareError::SESSION_EXPIRED:
        m_ui->buildInformationBox(tr("Session expired. Please login again"), true);
        m_facebookAuthenticator->clearAccountInformation(true); // keep username = true
        m_situareService->clearUserData();
        m_ui->loggedIn(false);
        m_ui->loginFailed();
        break;
    case SituareError::LOGIN_FAILED:
        m_ui->toggleProgressIndicator(false);
        m_ui->buildInformationBox(tr("Invalid E-mail address or password"), true);
        m_ui->loginFailed();
        break;
    case SituareError::UPDATE_FAILED:
        m_ui->toggleProgressIndicator(false);
        m_ui->buildInformationBox(tr("Update failed, please try again"), true);
        break;
    case SituareError::DATA_RETRIEVAL_FAILED:
        m_ui->toggleProgressIndicator(false);
        m_ui->buildInformationBox(tr("Data retrieval failed, please try again"), true);
        break;
    case SituareError::ADDRESS_RETRIEVAL_FAILED:
        m_ui->toggleProgressIndicator(false);
        m_ui->buildInformationBox(tr("Address retrieval failed"), true);
        break;
    case SituareError::IMAGE_DOWNLOAD_FAILED:
        m_ui->buildInformationBox(tr("Image download failed"), true);
        break;
    case SituareError::MAP_IMAGE_DOWNLOAD_FAILED:
        m_ui->buildInformationBox(tr("Map image download failed"), true);
        break;
    case SituareError::GPS_INITIALIZATION_FAILED:
        setGPS(false);
        m_ui->buildInformationBox(tr("GPS initialization failed"), true);
        break;
    case SituareError::INVALID_JSON:
        m_ui->buildInformationBox(tr("Malformatted reply from server"), true);
        m_ui->loggedIn(false);
        m_facebookAuthenticator->clearAccountInformation(false); // clean all
        break;
    case SituareError::ERROR_ROUTING_FAILED:
        m_ui->toggleProgressIndicator(false);
        m_ui->buildInformationBox(tr("Routing failed"), true);
        break;
    case SituareError::ERROR_LOCATION_SEARCH_FAILED:
        m_ui->buildInformationBox(tr("No results found"), true);
        break;
    default:
        m_ui->toggleProgressIndicator(false);
        if(context == ErrorContext::NETWORK)
            qCritical() << __PRETTY_FUNCTION__ << "QNetworkReply::NetworkError: " << error;
        else
            qCritical() << __PRETTY_FUNCTION__ << "Unknown error: " << error;
        break;
    }
}

void SituareEngine::fetchUsernameFromSettings()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->setUsername(m_facebookAuthenticator->loadUsername());
}

void SituareEngine::imageReady(User *user)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(user->type())
        emit userLocationReady(user);
    else
        emit friendImageReady(user);
}

void SituareEngine::initializeGpsAndAutocentering()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    QVariant gpsEnabled = settings.value(SETTINGS_GPS_ENABLED);
    QVariant autoCenteringEnabled = settings.value(SETTINGS_AUTO_CENTERING_ENABLED);

    if (m_gps->isInitialized()) {

        if (gpsEnabled.toString().isEmpty()) { // First start. Situare.conf file does not exists

            connect(m_gps, SIGNAL(position(GeoCoordinate, qreal)),
                    this, SLOT(setFirstStartZoomLevel()));

            changeAutoCenteringSetting(true);
            setGPS(true);

            m_ui->buildInformationBox(tr("GPS enabled"));

        } else { // Normal start
            changeAutoCenteringSetting(autoCenteringEnabled.toBool());
            setGPS(gpsEnabled.toBool());

            if (gpsEnabled.toBool())
                m_ui->buildInformationBox(tr("GPS enabled"));
        }
    } else {
        setGPS(false);
    }
}

void SituareEngine::locationSearch(QString location)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!location.isEmpty())
        m_geocodingService->requestLocation(location);
}

void SituareEngine::loginActionPressed()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_networkAccessManager->isConnected()) {
        if(m_ui->loginState()) {
            logout();
            m_situareService->clearUserData();
        } else {
            m_facebookAuthenticator->start();
        }
    }
    else {
        error(ErrorContext::NETWORK, QNetworkReply::UnknownNetworkError);
    }
}

void SituareEngine::loginOk()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->loggedIn(true);

    m_ui->show();
    m_situareService->fetchLocations(); // request user locations

    if (m_gps->isRunning())
        m_ui->readAutomaticLocationUpdateSettings();
}

void SituareEngine::loginProcessCancelled()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->toggleProgressIndicator(false);
    m_ui->updateItemVisibility();
}

void SituareEngine::logout()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->loggedIn(false);

    // signal to clear locationUpdateDialog's data
    connect(this, SIGNAL(clearUpdateLocationDialogData()),
            m_ui, SIGNAL(clearUpdateLocationDialogData()));
    emit clearUpdateLocationDialogData();

    m_facebookAuthenticator->clearAccountInformation(); // clear all
    m_automaticUpdateFirstStart = true;
}

void SituareEngine::refreshUserData()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_networkAccessManager->isConnected()) {
        m_ui->toggleProgressIndicator(true);
        m_situareService->fetchLocations();
    }
    else {
        error(ErrorContext::NETWORK, QNetworkReply::UnknownNetworkError);
    }
}

void SituareEngine::requestAddress()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_networkAccessManager->isConnected()) {
        if (m_gps->isRunning())
            m_situareService->reverseGeo(m_gps->lastPosition());
        else
            m_situareService->reverseGeo(m_mapEngine->centerGeoCoordinate());
    }
    else {
        error(ErrorContext::NETWORK, QNetworkReply::UnknownNetworkError);
    }
}

void SituareEngine::requestUpdateLocation(const QString &status, bool publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_networkAccessManager->isConnected()) {
        m_ui->toggleProgressIndicator(true);

        if (m_gps->isRunning())
            m_situareService->updateLocation(m_gps->lastPosition(), status, publish);
        else
            m_situareService->updateLocation(m_mapEngine->centerGeoCoordinate(), status, publish);
    }
    else {
        error(ErrorContext::NETWORK, QNetworkReply::UnknownNetworkError);
    }
}

void SituareEngine::requestAutomaticUpdateIfMoved(GeoCoordinate position)
{
    qDebug() << __PRETTY_FUNCTION__;

    if ((fabs(m_lastUpdatedGPSPosition.longitude() - position.longitude()) >
         USER_MOVEMENT_MINIMUM_LONGITUDE_DIFFERENCE) ||
        (fabs(m_lastUpdatedGPSPosition.latitude() - position.latitude()) >
         USER_MOVEMENT_MINIMUM_LATITUDE_DIFFERENCE)) {

        m_lastUpdatedGPSPosition = position;
        m_userMoved = true;
    }

    if (m_automaticUpdateRequest && m_userMoved) {
        requestUpdateLocation(tr("Automatic location update"));
        m_automaticUpdateRequest = false;
        m_userMoved = false;
    }
}

void SituareEngine::routeParsed(Route &route)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(route);

    m_ui->toggleProgressIndicator(false);
}

void SituareEngine::routeTo(const GeoCoordinate &endPointCoordinates)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->toggleProgressIndicator(true);

    if (m_gps->isRunning())
        m_routingService->requestRoute(m_gps->lastPosition(), endPointCoordinates);
    else
        m_routingService->requestRoute(m_mapEngine->centerGeoCoordinate(), endPointCoordinates);
}

void SituareEngine::routeToCursor()
{
    qDebug() << __PRETTY_FUNCTION__;

    routeTo(m_mapEngine->centerGeoCoordinate());
}

void SituareEngine::setAutoCentering(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__ << enabled;

    m_ui->setIndicatorButtonEnabled(enabled);
    m_mapEngine->setAutoCentering(enabled);
    m_ui->setCrosshairVisibility(!enabled);

    if (enabled) {
        setGPS(true);
        m_gps->requestLastPosition();
    }
}

void SituareEngine::setFirstStartZoomLevel()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_autoCenteringEnabled) // autocentering is disabled when map is scrolled
        m_mapEngine->setZoomLevel(DEFAULT_ZOOM_LEVEL_WHEN_GPS_IS_AVAILABLE);

    disconnect(m_gps, SIGNAL(position(GeoCoordinate, qreal)),
               this, SLOT(setFirstStartZoomLevel()));
}

void SituareEngine::setGPS(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__ << enabled;

    if (m_gps->isInitialized()) {
        m_ui->setGPSButtonEnabled(enabled);
        m_mapEngine->setGPSEnabled(enabled);

        if (enabled && !m_gps->isRunning()) {
            m_gps->start();
            m_gps->requestLastPosition();

            if(m_ui->loginState())
                m_ui->readAutomaticLocationUpdateSettings();
        }
        else if (!enabled && m_gps->isRunning()) {
            m_gps->stop();
            changeAutoCenteringSetting(false);
            enableAutomaticLocationUpdate(false);
        }
    }
    else {
        if (enabled)
            m_ui->buildInformationBox(tr("Unable to start GPS"));
        m_ui->setGPSButtonEnabled(false);
        m_mapEngine->setGPSEnabled(false);
    }
}

void SituareEngine::setPowerSaving(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__ << enabled;

    m_gps->enablePowerSave(enabled);

    if(m_autoCenteringEnabled)
        m_mapEngine->setAutoCentering(!enabled);
}

void SituareEngine::showContactDialog(const QString &facebookId)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString guid = m_contactManager->contactGuid(facebookId);

    if (!guid.isEmpty())
        m_ui->showContactDialog(guid);
    else
        m_ui->buildInformationBox(tr("Unable to find contact.\nAdd Facebook IM "
                                     "account from Conversations to use this feature."), true);
}

void SituareEngine::signalsFromFacebookAuthenticator()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_facebookAuthenticator, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));

    connect(m_facebookAuthenticator, SIGNAL(credentialsReady(FacebookCredentials)),
            m_situareService, SLOT(credentialsReady(FacebookCredentials)));

    connect(m_facebookAuthenticator, SIGNAL(credentialsReady(FacebookCredentials)),
            this, SLOT(loginOk()));

    connect(m_facebookAuthenticator, SIGNAL(newLoginRequest()),
            m_ui, SLOT(startLoginProcess()));

    connect(m_facebookAuthenticator, SIGNAL(saveCookiesRequest()),
            m_ui, SLOT(saveCookies()));

    connect(m_facebookAuthenticator, SIGNAL(loginUsingCookies()),
            m_ui, SLOT(loginUsingCookies()));
}

void SituareEngine::signalsFromGeocodingService()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_geocodingService, SIGNAL(locationDataParsed(const QList<Location>&)),
            m_ui, SIGNAL(locationDataParsed(const QList<Location>&)));

    connect(m_geocodingService, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));
}

void SituareEngine::signalsFromGPS()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_gps, SIGNAL(position(GeoCoordinate, qreal)),
            m_mapEngine, SLOT(gpsPositionUpdate(GeoCoordinate, qreal)));

    connect(m_gps, SIGNAL(timeout()),
            m_ui, SLOT(gpsTimeout()));

    connect(m_gps, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));
}

void SituareEngine::signalsFromMainWindow()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_ui, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));

    connect(m_ui, SIGNAL(fetchUsernameFromSettings()),
            this, SLOT(fetchUsernameFromSettings()));

    connect(m_ui, SIGNAL(loginActionPressed()),
            this, SLOT(loginActionPressed()));

    connect(m_ui, SIGNAL(saveUsername(QString)),
            m_facebookAuthenticator, SLOT(saveUsername(QString)));

    connect(m_ui, SIGNAL(updateCredentials(QUrl)),
            m_facebookAuthenticator, SLOT(updateCredentials(QUrl)));

    // signals from map view
    connect(m_ui, SIGNAL(mapViewScrolled(SceneCoordinate)),
            m_mapEngine, SLOT(setCenterPosition(SceneCoordinate)));

    connect(m_ui, SIGNAL(mapViewResized(QSize)),
            m_mapEngine, SLOT(viewResized(QSize)));

    connect(m_ui, SIGNAL(viewZoomFinished()),
            m_mapEngine, SLOT(viewZoomFinished()));

    // signals from zoom buttons (zoom panel and volume buttons)
    connect(m_ui, SIGNAL(zoomIn()),
            m_mapEngine, SLOT(zoomIn()));

    connect(m_ui, SIGNAL(zoomOut()),
            m_mapEngine, SLOT(zoomOut()));

    // signals from menu buttons
    connect(m_ui, SIGNAL(gpsTriggered(bool)),
            this, SLOT(setGPS(bool)));

    //signals from dialogs
    connect(m_ui, SIGNAL(cancelLoginProcess()),
            this, SLOT(loginProcessCancelled()));

    connect(m_ui, SIGNAL(requestReverseGeo()),
            this, SLOT(requestAddress()));

    connect(m_ui, SIGNAL(statusUpdate(QString,bool)),
            this, SLOT(requestUpdateLocation(QString,bool)));

    connect(m_ui, SIGNAL(enableAutomaticLocationUpdate(bool, int)),
            this, SLOT(enableAutomaticLocationUpdate(bool, int)));

    // signals from user info tab
    connect(m_ui, SIGNAL(refreshUserData()),
            this, SLOT(refreshUserData()));

    connect(m_ui, SIGNAL(centerToCoordinates(GeoCoordinate)),
            m_mapEngine, SLOT(centerToCoordinates(GeoCoordinate)));

    // routing signal from friend list tab & search location tab
    connect(m_ui, SIGNAL(routeTo(const GeoCoordinate&)),
            this, SLOT(routeTo(const GeoCoordinate&)));

    // signals from location search panel
    connect(m_ui,
            SIGNAL(locationItemClicked(const GeoCoordinate&, const GeoCoordinate&)),
            m_mapEngine,
            SLOT(showMapArea(const GeoCoordinate&, const GeoCoordinate&)));

    connect(m_ui, SIGNAL(searchHistoryItemClicked(QString)),
            this, SLOT(locationSearch(QString)));

    // signals from routing tab
    connect(m_ui, SIGNAL(clearRoute()),
            m_mapEngine, SLOT(clearRoute()));

    connect(m_ui, SIGNAL(routeToCursor()),
            this, SLOT(routeToCursor()));

    // signals from distance indicator button
    connect(m_ui, SIGNAL(autoCenteringTriggered(bool)),
            this, SLOT(changeAutoCenteringSetting(bool)));

    connect(m_ui, SIGNAL(draggingModeTriggered()),
            this, SLOT(draggingModeTriggered()));

    // signal from search location dialog
    connect(m_ui, SIGNAL(searchForLocation(QString)),
            this, SLOT(locationSearch(QString)));

    // signal from friend list panel
    connect(m_ui, SIGNAL(requestContactDialog(const QString &)),
            this, SLOT(showContactDialog(const QString &)));
}

void SituareEngine::signalsFromMapEngine()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_mapEngine, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));

    connect(m_mapEngine, SIGNAL(locationChanged(SceneCoordinate)),
            m_ui, SIGNAL(centerToSceneCoordinates(SceneCoordinate)));

    connect(m_mapEngine, SIGNAL(zoomLevelChanged(int)),
            m_ui, SIGNAL(zoomLevelChanged(int)));

    connect(m_mapEngine, SIGNAL(mapScrolledManually()),
            this, SLOT(disableAutoCentering()));

    connect(m_mapEngine, SIGNAL(maxZoomLevelReached()),
            m_ui, SIGNAL(maxZoomLevelReached()));

    connect(m_mapEngine, SIGNAL(minZoomLevelReached()),
            m_ui, SIGNAL(minZoomLevelReached()));

    connect(m_mapEngine, SIGNAL(locationItemClicked(QList<QString>)),
            m_ui, SIGNAL(locationItemClicked(QList<QString>)));

    connect(m_mapEngine, SIGNAL(newMapResolution(qreal)),
            m_ui, SIGNAL(newMapResolution(qreal)));

    connect(m_mapEngine, SIGNAL(directionIndicatorValuesUpdate(qreal, qreal, bool)),
            m_ui, SIGNAL(directionIndicatorValuesUpdate(qreal, qreal, bool)));
}

void SituareEngine::signalsFromRoutingService()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_routingService, SIGNAL(routeParsed(Route&)),
            this, SLOT(routeParsed(Route&)));

    connect(m_routingService, SIGNAL(routeParsed(Route&)),
            m_mapEngine, SLOT(setRoute(Route&)));

    connect(m_routingService, SIGNAL(routeParsed(Route&)),
            m_ui, SIGNAL(routeParsed(Route&)));

    connect(m_routingService, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));
}

void SituareEngine::signalsFromSituareService()
{
    qDebug() << __PRETTY_FUNCTION__;

    connect(m_situareService, SIGNAL(error(int, int)),
            this, SLOT(error(int, int)));

    connect(m_situareService, SIGNAL(imageReady(User*)),
            this, SLOT(imageReady(User*)));

    connect(m_situareService, SIGNAL(reverseGeoReady(QString)),
            m_ui, SIGNAL(reverseGeoReady(QString)));

    connect(m_situareService, SIGNAL(userDataChanged(User*, QList<User*>&)),
            this, SLOT(userDataChanged(User*, QList<User*>&)));

    connect(m_situareService, SIGNAL(updateWasSuccessful()),
            this, SLOT(updateWasSuccessful()));

    connect(m_situareService, SIGNAL(updateWasSuccessful()),
            m_ui, SIGNAL(clearUpdateLocationDialogData()));
}

void SituareEngine::startAutomaticUpdate()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gps->requestUpdate();
    m_automaticUpdateRequest = true;
}

void SituareEngine::topmostWindowChanged(bool isMainWindow)
{
    qDebug() << __PRETTY_FUNCTION__;

    setPowerSaving(!isMainWindow);
}

void SituareEngine::updateWasSuccessful()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_networkAccessManager->isConnected())
        m_situareService->fetchLocations();
    else
        error(ErrorContext::NETWORK, QNetworkReply::UnknownNetworkError);
}

void SituareEngine::userDataChanged(User *user, QList<User *> &friendsList)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_ui->toggleProgressIndicator(false);

    emit userLocationReady(user);
    emit friendsLocationsReady(friendsList);
}
