/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/


#ifndef SITUARESERVICE_H
#define SITUARESERVICE_H

#include <QObject>
#include "../facebookservice/facebookcredentials.h"
#include "../user/user.h"
#include "imagefetcher.h"

class NetworkAccessManager;
class QNetworkReply;
class QNetworkRequest;
class QPointF;
class QUrl;

/**
* @brief SituareService class for communicating with Situare server
*
* @author Henri Lampela
* @class SituareService situareservice.h "situareservice/situareservice.h"
*/
class SituareService : public QObject
{
    Q_OBJECT

public:

    /**
    * @brief Default constructor
    *
    * @param parent instance of parent
    */
    SituareService(QObject *parent = 0);

    /**
    * @brief Destructor
    *
    */
    ~SituareService();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/

    /**
    * @brief Retrieves location user and friends information from Situare server
    *
    */
    void fetchLocations();

    /**
    * @brief Translates coordinates to street address via Situare server
    *
    * @param coordinates coordinates to be translated
    */
    void reverseGeo(const QPointF &coordinates);

    /**
    * @brief Updates location to the Situare server
    *
    * @param coordinates current cordinates
    * @param status message
    * @param publish publish location on Facebook wall (true/false)
    */
    void updateLocation(const QPointF &coordinates, const QString &status, const bool &publish);

public slots:

    /**
    * @brief Public slot, to clear user data
    *
    */
    void clearUserData();

    /**
    * @brief Public slot, which indicates when facebook credentials are ready
    *
    * @param credentials New credentials
    */
    void credentialsReady(const FacebookCredentials &credentials);

    /**
    * @brief Public slot, which indicates when http request has been completed
    *
    * @param reply storage for http reply
    */
    void requestFinished(QNetworkReply *reply);

private:

    /**
    * @brief Requests ImageFetcher if user/friend has a profile image
    *        uses members: m_user and m_friendsList
    *
    */
    void addProfileImages();

    /**
    * @brief Forms a http cookie
    *
    * @param apiKeyValue application key
    * @param expiresValue session expire date&time from Facebook
    * @param userValue user id from Facebook
    * @param sessionKeyValue session key from Facebook
    * @param sessionSecretValue session secret from Facebook
    * @param signatureValue md5 generated signature
    * @param localeValue used locale
    * @return QString formed cookie
    */
    QString formCookie(const QString &apiKeyValue, QString expiresValue, QString userValue,
                       QString sessionKeyValue, QString sessionSecretValue,
                       const QString &signatureValue, const QString &localeValue);

    /**
    * @brief Forms a http url
    *
    * @param baseUrl Server url
    * @param phpScript Server script
    * @param urlParameters optional parameters for url
    * @return QUrl formed url
    */
    QUrl formUrl(const QString &baseUrl, const QString &phpScript,
                 QString urlParameters = QString());

    /**
    * @brief Forms url parameters
    *
    * @param coordinates current coordinates
    * @param status optional status message
    * @param publish optional publish location on Facebook wall (true/false)
    * @return QString
    */
    QString formUrlParameters(const QPointF &coordinates, QString status = QString(),
                              QString publish = QString());

    /**
    * @brief Parses user and friend data from JSON string
    *
    * @param jsonReply JSON string
    */
    void parseUserData(const QByteArray &jsonReply);

    /**
    * @brief Sends http request
    *
    * @param url destination
    * @param cookieType type of the cookie
    * @param cookie http cookie
    */
    void sendRequest(const QUrl &url, const QString &cookieType, const QString &cookie);

private slots:

    /**
    * @brief Slot for received images
    *
    * @param url Image url
    * @param image Received image
    */
    void imageReceived(const QUrl &url, const QPixmap &image);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/

signals:

    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
    * @brief Signal for image fetching
    *
    * @param url Image url
    */
    void fetchImage(const QUrl &url);

    /**
    * @brief Signals when address data is retrieved
    *
    * @param address Street address
    */
    void reverseGeoReady(const QString &address);

    /**
    * @brief Signals when updateLocation request finished successfully
    *
    */
    void updateWasSuccessful();

    /**
    * @brief Signals when user data is retrieved
    *
    * @param user instance of user
    * @param friendList list of friends
    */
    void userDataChanged(User *user, QList<User *> &friendList);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/

private:

    int m_nbrOfImages;      ///< Indicates number of friends whose images has been downloaded
    int m_visited;          ///< Indicates number of friends with images

    bool m_defaultImage;    ///< Indicates if some of the friends/user doesn't have a image

    QList<QNetworkReply *> m_currentRequests;   ///< List of current http requests
    QList<User *> m_friendsList;                ///< List of friends(User)

    NetworkAccessManager *m_networkManager;    ///< Pointer to QNetworkAccessManager

    FacebookCredentials m_credentials;          ///< handle for FacebookCredentials
    ImageFetcher *m_imageFetcher;               ///< Instance of the image fetcher
    User *m_user;                               ///< Pointer to User



};

#endif // SITUARESERVICE_H
