/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef GPSPOSITIONPRIVATE_H
#define GPSPOSITIONPRIVATE_H

#include <QPointF>

#include <QGeoPositionInfo>
#include <QGeoPositionInfoSource>
#include <QNmeaPositionInfoSource>

#include "gpsposition.h"

QTM_USE_NAMESPACE

/**
* @brief GPSPositionPrivate class use GPS to receive location information.
*/
class GPSPositionPrivate : public QObject
{
    Q_OBJECT

public:
    /**
    * @brief Constructor creates GPS source.
    *
    * @param parent QObject
    */
    GPSPositionPrivate(QObject *parent);

/******************************************************************************
* MEMBER FUNCTIONS AND SLOTS
******************************************************************************/
public:
    /**
    * @brief Returns is GPS initialized.
    *
    * @return true if initialized, false otherwise
    */
    bool isInitialized();

    /**
    * @brief Checks if GPS is running.
    *
    * @return true if GPS running, false otherwise
    */
    bool isRunning();

    /**
    * @brief Return last known position.
    */
    QPointF lastPosition();

    /**
    * @brief Informs gps to emit last known position.
    */
    void requestLastPosition();

    /**
    * @brief Requests update from GPS.
    *
    * Enables GPS if it is disabled
    */
    void requestUpdate();

    /**
    * @brief Set GPS mode.
    *
    * Modes: default and simulation.
    * @param mode GPS mode
    * @param filePath file path to NMEA file if simulation mode is used
    */
    void setMode(GPSPosition::Mode mode, const QString &filePath = 0);

    /**
    * @brief Sets power save mode.
    *
    * Starts GPS for position update and then stops it.
    */
    void setPowerSave(bool enabled);

    /**
    * @brief Set GPS update interval
    *
    * @return interval interval in milliseconds
    */
    void setUpdateInterval(int interval);

    /**
    * @brief Start GPS.
    */
    void start();

    /**
    * @brief Stop GPS.
    */
    void stop();

private:
    /**
    * @brief Return horizontal accuracy
    *
    * @param positionInfo geo position info
    * @return accuracy value, -1 if undefined. Returns -1 also is timestamp is not valid
    *         (when using network positioning)
    */
    qreal accuracy(const QGeoPositionInfo &positionInfo);

private slots:

    /**
    * @brief Slot for received position update.
    *
    * @param positionInfo Geo position info.
    */
    void positionUpdated(const QGeoPositionInfo &positionInfo);

    /**
    * @brief Slot for update timeout.
    *
    * Called when request timeout occurs.
    */
    void updateTimeout();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QGeoPositionInfoSource *m_gpsSource;        ///< GPS position info source
    GPSPosition *m_parent;                      ///< Parent object
    bool m_initialized;                         ///< GPS is initialized
    bool m_running;                             ///< GPS is running
    int m_updateInterval;                       ///< GPS update interval
};

const int DEFAULT_UPDATE_INTERVAL = 5000;       ///< Default update interval

#endif // GPSPOSITIONPRIVATE_H
