/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QFile>
#include <QDir>
#include <QApplication>
#include <QDebug>
#include <QTimer>

#include "common.h"
#include "gpscommon.h"
#include "gpsposition.h"
#include "gpspositionprivateliblocation.h"
#include "liblocationwrapper.h"

GPSPositionPrivate::GPSPositionPrivate(QObject *parent)
    : QObject(parent),
      m_liblocationWrapper(0),
      m_initialized(false),
      m_running(false),
      m_updateInterval(DEFAULT_UPDATE_INTERVAL)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_parent = static_cast<GPSPosition*>(parent);
}

bool GPSPositionPrivate::isInitialized()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_initialized;
}

void GPSPositionPrivate::setMode(GPSPosition::Mode mode, const QString &filePath)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(filePath);

    if (m_liblocationWrapper) {
        disconnect(m_liblocationWrapper, 0, 0, 0);
        delete m_liblocationWrapper;
    }

    if (mode == GPSPosition::Default) {
        m_liblocationWrapper = new LiblocationWrapper(this);

        if (!m_liblocationWrapper) {     
            m_initialized = false;
            emit m_parent->error(SituareError::GPS_INITIALIZATION_FAILED);
            return;
        }
    }

    if (m_liblocationWrapper) {
        m_initialized = true;
        m_liblocationWrapper->init(m_updateInterval);

        connect(m_liblocationWrapper, SIGNAL(locationChanged(const GeoPositionInfo &)),
                this, SLOT(positionUpdated(const GeoPositionInfo &)));
        connect(m_liblocationWrapper, SIGNAL(errorMessage(const QString &)),
                this, SLOT(locationError(const QString &)));
    }
}

void GPSPositionPrivate::start()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_initialized && !isRunning()) {
        m_liblocationWrapper->startUpdates();
        m_running = true;
    }
}

void GPSPositionPrivate::stop()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_initialized && isRunning()) {
        m_liblocationWrapper->stopUpdates();
        m_running = false;
    }
}

bool GPSPositionPrivate::isRunning()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_running;
}

QPointF GPSPositionPrivate::lastPosition()
{
    qDebug() << __PRETTY_FUNCTION__;

    GeoPositionInfo positionInfo =  m_liblocationWrapper->lastKnownPosition();

    return QPointF(positionInfo.coordinate().longitude(), positionInfo.coordinate().latitude());
}

void GPSPositionPrivate::requestLastPosition()
{
    qDebug() << __PRETTY_FUNCTION__;

    GeoPositionInfo positionInfo = m_liblocationWrapper->lastKnownPosition();

    if (positionInfo.isValid()) {
        GeoCoordinate coordinate = positionInfo.coordinate();
        emit m_parent->position(QPointF(coordinate.longitude(), coordinate.latitude()),
                      accuracy(positionInfo));
    }
}

void GPSPositionPrivate::positionUpdated(const GeoPositionInfo &positionInfo)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (positionInfo.coordinate().isValid()) {
        GeoCoordinate coordinate = positionInfo.coordinate();
        emit m_parent->position(QPointF(coordinate.longitude(), coordinate.latitude()),
                      accuracy(positionInfo));
    }
}

void GPSPositionPrivate::locationError(const QString &errorMessage)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(errorMessage);

    emit m_parent->error(SituareError::GPS_INITIALIZATION_FAILED);
}

void GPSPositionPrivate::setUpdateInterval(int interval)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_updateInterval != interval) {
        m_updateInterval = interval;
        m_liblocationWrapper->init(m_updateInterval);
    }
}

qreal GPSPositionPrivate::accuracy(const GeoPositionInfo &positionInfo)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (positionInfo.isAccurate())
        return positionInfo.accuracy();
    else
        return GPS_ACCURACY_UNDEFINED;
}
