/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jukka Saastamoinen - jukka.saastamoinen@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Katri Kaikkonen - katri.kaikkonen@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef USERINFO_H
#define USERINFO_H

#include <QWidget>
#include "updatelocation/updatelocationdialog.h"

class ImageButton;

/**
* @brief UserInfo shows user's data in expandable item.
*
* @class UserInfo userinfo.h "ui/userinfo.h"
*/
class UserInfo : public QWidget
{
    Q_OBJECT
public:
    /**
    * @brief Constructor
    *
    * @param parent Parent
    */
    UserInfo(QWidget *parent=0);

    /**
    * @brief Desctructor
    * saves unsend user message to settings
    *
    */
    ~UserInfo();

/*******************************************************************************
* BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
*******************************************************************************/
protected:
    /**
    * @brief Set current mouse position to mouse event position.
    *
    * @param event QMouseEvent mouse press event
    */
    void mousePressEvent(QMouseEvent *event);

    /**
    * @brief Call toggleHeight if mouse position is unchanged.
    *
    * Check if current mouse position is same as mouse release position.
    * @param event QMouseEvent mouse release event
    */
    void mouseReleaseEvent(QMouseEvent *event);

    /**
    * @brief This function is called when the widget is drawn
    *
    * @param aPaintEvent Pointer to paint event
    */
    void paintEvent(QPaintEvent *aPaintEvent);

/******************************************************************************
* MEMBER FUNCTIONS AND SLOTS
******************************************************************************/
public:
    /**
    * @brief Sets the users current address
    *
    * @param address Reference to users current address
    */
    void setAddress(const QString &address);

    /**
    * @brief Sets the users current coordinates
    *
    * @param coordinates Reference to users current coordinates
    */
    void setCoordinates(const QPointF &coordinates);    

    /**
    * @brief Sets the user picture
    *
    * @param image Reference to user picture
    */
    void setProfileImage(const QPixmap &image);

    /**
    * @brief Sets the time of updated message
    *
    * @param time Reference to time when message was sent.
    */
    void setTime(const QString &time);

   /**
    * @brief Sets the user name
    *
    * @param name Reference to user name
    */
    void setUserName(const QString &name);

public slots:

    /**
    * @brief Clears backups of message and publish on Facebook setting
    */
    void clearUpdateLocationDialogData();

    /**
    * @brief Saves status message and Facebook publish setting
    *
    * @param status message that user sends. Message is stored to m_backupMessage data member
    * @param publish setting that determines whether the user status message is published on
    *        Facebook. This value is stored to m_backupFacebookPublishPolicity data member.
    */
    void backupUpdateLocationDialogData(const QString &status, bool publish);

    /**
    * @brief Sets the message text
    *
    * @param text Reference to user message
    */
    void setMessageText(const QString &text);


    /**
    * @brief Emits signal to inform user about failed message/location update.
    *        Contains logic to send signal only after failed message/location update.
    *
    * @param error from Situare server. Error is not used in this function
    */
    void verifyMessageUpdateFailure(const int error);

private:

    /**
    * @brief reads Unsend message from settings at startup
    *
    */
    void restoreUnsendMessage();

    /**
    * @brief Set shortened or full-length text to labels.
    *
    * @param expanded true if full-length text is set, false otherwise
    */
    void setText(bool expanded);

    /**
    * @brief Set shortened texts from User data.
    *
    * Text length is defined by MAXIMUM_CHARS.
    */
    QString shortenText(const QLabel *label, const QString &text, int textMaxWidth);

private slots:
    /**
    * @brief Slot for find button click
    */
    void findButtonClicked();

    /**
    * @brief Slot function to forward messageUpdate launch signal
    *
    */
    void messageUpdate();    

    /**
    * @brief Slot function to get indication when dialog is finished
    *
    */
    void updateLocationDialogFinished(int reason);

/******************************************************************************
* SIGNALS
******************************************************************************/
signals:
    /**
    * @brief Signal for finding user.
    *
    * @param coordinates user geo coordinates
    */
    void findUser(const QPointF &coordinates);

    /**
    * @brief Signal that used to inform user that his message/location update tp Situare server
    *        was failed.
    *        This signal is originally sended from UserInfo
    * @param message message for notification
    * @param modal true if modal notification otherwice false
    */
    void notificateUpdateFailing(const QString &message, bool modal);

    /**
    * @brief Signal for refreshing user data.
    *
    */
    void refreshUserData();

    /**
    * @brief Signal for requesting reverseGeo from SituareEngine
    *
    */
    void requestReverseGeo();

    /**
    * @brief Signals, when address data is ready
    *
    * @param address Street address
    */
    void reverseGeoReady(const QString &address);

    /**
    * @brief Signal for requestLocationUpdate from SituareEngine via MainWindow class
    *
    * @param status Status message
    * @param publish Publish on Facebook
    */
    void statusUpdate(const QString &status, const bool &publish);
/******************************************************************************
* DATA MEMBERS
******************************************************************************/
private:
    bool m_backupFacebookPublishPolicity; ///< Backup of publish on Facebook checkbox value
    bool m_expanded;                    ///< Item expanded state   
    bool m_messageUpdateVerified;       ///< Place for message/location update check
    QLabel *m_locationLabel;            ///< Location label
    QLabel *m_nameLabel;                ///< Name label
    QLabel *m_statusTextLabel;          ///< Status text label
    QLabel *m_updatedLabel;             ///< Updated label
    QPixmap m_backgroundBottomImage;    ///< Bottom background image
    QPixmap m_backgroundMiddleImage;    ///< Middle background image
    QPixmap m_backgroundTopImage;       ///< Top background image
    QPoint m_mousePosition;             ///< Current mouse press position
    QPointF m_coordinates;              ///< User current coordinates
    QString m_address;                  ///< Address from where the new message was sent
    QString m_backupMessage;            ///< Backup of users message
    QString m_messageText;              ///< User's message
    QString m_time;                     ///< Time when the new message was sent
    QString m_userName;                 ///< User's name
    ImageButton *m_findButton;          ///< User find button   
    UpdateLocationDialog *m_updateLocation; ///< Update location dialog
};

#endif // USERINFO_H
