/*
 * This file is part of simple-fmtx-widget
 *
 * Copyright (C) 2009 Pekka Rönkkö. All rights reserved.
 *
 * This code is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the root
 * directory.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libosso.h>
#include "simple-fmtx-widget.h"
#include <gconf/gconf-client.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <stdlib.h>

HD_DEFINE_PLUGIN_MODULE (SimpleFmtxWidget, simple_fmtx_widget_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

#define SIMPLE_FMTX_WIDGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                        SIMPLE_TYPE_FMTX_WIDGET, SimpleFmtxWidgetPrivate))

typedef enum
{
    GET,
    SET,
} PropertyCommand;

typedef enum
{
    ON,
    ON_PRESSED,
    OFF,
    OFF_PRESSED,
} ButtonState;

typedef struct _SimpleFmtxWidgetPrivate SimpleFmtxWidgetPrivate;
struct _SimpleFmtxWidgetPrivate
{
    gboolean fmtx_state;
    osso_context_t *osso_context;
    GtkWidget *button;
    GtkWidget *label;
    GtkWidget *state_label;
	GtkWidget* btn_off_image;
    GtkWidget* btn_off_pressed;
	GtkWidget* btn_on_image;
	GtkWidget* btn_on_pressed;
	GtkWidget* fixed;
	ButtonState current_button_state;
    gboolean is_boost_on;
    gboolean is_boost_installed;
    DBusConnection *bus;
    GConfClient* gconf;
    gboolean signals_connected;
};
static gboolean block_leave_notify = FALSE;

#define GCONF_KEY_BOOST "/apps/simplefmtxwidget/boost"
#define IMAGE_DIR "/usr/share/icons/hicolor/scalable/hildon/"
#define FM_BOOST "/sbin/fm-boost"
/*****************************************************************************************/
static gpointer get_set(SimpleFmtxWidget *self, PropertyCommand commad, const gchar* val, gboolean state_val);
static void set_freq_label(SimpleFmtxWidget *self);
static gboolean get_current_state(SimpleFmtxWidget *self);
static gboolean button_clicked (GtkWidget *widget,
        GdkEventButton *event,
        gpointer user_data);
static gboolean button_released (GtkWidget *widget,
        GdkEventButton *event,
        gpointer user_data);

static GtkWidget *build_ui (SimpleFmtxWidget *self);

static gboolean event_leave_notify_button(GtkWidget *widget,
                                    GdkEventCrossing *event,
                                    gpointer user_data);
static void create_images(SimpleFmtxWidget *self);

static void set_button_image(SimpleFmtxWidget *self, ButtonState state);
static void open_settings_dialog(HDHomePluginItem *hdhomepluginitem, gpointer user_data);
static void launch_script(void);
static void set_gconf(SimpleFmtxWidget *self, gboolean boost);
static gboolean get_gconf(SimpleFmtxWidget *self);
static void visible_notify (GObject *object,
                             GParamSpec *spec,
                             gpointer data);
static void update_ui(SimpleFmtxWidget* self);
/*****************************************************************************************/

static void launch_script(void)
{


    if(g_file_test(FM_BOOST, G_FILE_TEST_EXISTS))
    {
        system("/usr/bin/sudo /sbin/fm-boost");
        hildon_banner_show_information(NULL, NULL, "FM-Transmitter Power Level set to: 118");
    }
    else
    {
        hildon_banner_show_information(NULL, NULL, "Error while running script /sbin/fm-boost");
    }
}
static void set_gconf(SimpleFmtxWidget *self, gboolean boost)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

    gconf_client_set_bool(priv->gconf, GCONF_KEY_BOOST, boost, NULL);
    priv->is_boost_on = boost;
}

static gboolean get_gconf(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

    priv->is_boost_on = gconf_client_get_bool(priv->gconf, GCONF_KEY_BOOST,NULL);

    return priv->is_boost_on;
}   

static void set_button_image(SimpleFmtxWidget *self, ButtonState state)
{
	SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

	switch (state) 
	{
		case ON:
	        gtk_widget_hide(priv->btn_off_image);
	        gtk_widget_hide(priv->btn_off_pressed);
	        gtk_widget_hide(priv->btn_on_pressed);
	        
	        gtk_widget_show(priv->btn_on_image);
	        priv->current_button_state = ON;
			break;
		case ON_PRESSED:
	        gtk_widget_hide(priv->btn_off_image);
	        gtk_widget_hide(priv->btn_off_pressed);
	        gtk_widget_hide(priv->btn_on_image);
	        
	        gtk_widget_show(priv->btn_on_pressed);
	        priv->current_button_state = ON_PRESSED;
			break;
		case OFF:
	        gtk_widget_hide(priv->btn_off_pressed);
	        gtk_widget_hide(priv->btn_on_image);
	        gtk_widget_hide(priv->btn_on_pressed);
	        
	        gtk_widget_show(priv->btn_off_image);
	        priv->current_button_state = OFF;
			break;
		case OFF_PRESSED:
	        gtk_widget_hide(priv->btn_off_image);
	        gtk_widget_hide(priv->btn_on_image);
	        gtk_widget_hide(priv->btn_on_pressed);
	        
	        gtk_widget_show(priv->btn_off_pressed);
	        priv->current_button_state = OFF_PRESSED;
			break;
		default:
			break;
	}
}
static void create_images(SimpleFmtxWidget *self)
{
	SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);
	
    priv->btn_off_image = gtk_image_new_from_file(IMAGE_DIR"fmtx-widget-button-off.png");
	priv->btn_off_pressed = gtk_image_new_from_file(IMAGE_DIR"fmtx-widget-button-off-pressed.png");
	
	priv->btn_on_image = gtk_image_new_from_file(IMAGE_DIR"fmtx-widget-button-on.png");
	priv->btn_on_pressed = gtk_image_new_from_file(IMAGE_DIR"fmtx-widget-button-on-pressed.png");
	
}

static gboolean get_current_state(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);
    
    gchar* current_state = (gchar*)get_set(self, GET, "state", FALSE);

    if (g_strcmp0(current_state, "enabled") == 0)
    {
        priv->fmtx_state = TRUE;
        set_button_image(self, OFF);
    
        gtk_label_set_text(GTK_LABEL(priv->state_label), "Transmit is ON");       

        g_free(current_state);
        return TRUE;
    }
    else if (g_strcmp0(current_state, "disabled") == 0)
    {
        priv->fmtx_state = FALSE;
        set_button_image(self, ON);

        gtk_label_set_text(GTK_LABEL(priv->state_label), "Transmit is OFF");       

        g_free(current_state);
        return FALSE;
    }
    g_free(current_state);
    return FALSE;
}
static void set_freq_label(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

    gchar* freq = NULL;
    guint32 f = (guint32)get_set(self, GET, "frequency", FALSE);
    gdouble f_d = (gdouble)f/1000;
    freq = g_strdup_printf("%.2f MHz",f_d);

    gtk_label_set_text(GTK_LABEL(priv->label), freq);
    g_free(freq);
}

static gboolean event_leave_notify_button(GtkWidget *widget,
                                    GdkEventCrossing *event,
                                    gpointer user_data)
{
    if (block_leave_notify)
    {
        return TRUE;
    }
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (user_data);

    if (priv->current_button_state == ON_PRESSED)
    {
        set_button_image(user_data, ON);

    }
    else
    {
        set_button_image(user_data, OFF);
    }
    return TRUE;

}

static gboolean button_clicked (GtkWidget *widget,
        GdkEventButton *event,
        gpointer user_data)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (user_data);
    
    if (priv->fmtx_state == TRUE)
    {
        set_button_image(user_data, OFF_PRESSED);
    }
    else
    {
        set_button_image(user_data, ON_PRESSED);
    }
    return TRUE;
}
static gboolean button_released (GtkWidget *widget,
        GdkEventButton *event,
        gpointer user_data)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (user_data);
   
    if (priv->fmtx_state == TRUE)
    {
        set_button_image(user_data, OFF);
	    get_set((SimpleFmtxWidget*)user_data, SET, "state", FALSE);
    }
    else if (priv->fmtx_state == FALSE)
    {
        set_button_image(user_data, ON);
	    const gchar* error = get_set((SimpleFmtxWidget*)user_data, SET, "state", TRUE);
        if (priv->is_boost_on && priv->is_boost_installed)
        {   
            if (g_strcmp0(error, "error") != 0)                 
            {
                launch_script();
            }
        }
    }
        get_current_state(user_data);
    return TRUE;

}

static gboolean fmtx_widget_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	cairo_t *cr;
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));

	GdkColor color;
	if (1) {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.75);
	} else {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "SelectionColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.6);
	}
	
	gint width, height, x, y;
	gint radius = 5;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;
	
	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width - radius, y);
	cairo_curve_to(cr, x + width - radius, y, x + width, y, x + width,
				y + radius);
	cairo_line_to(cr, x + width, y + height - radius);
	cairo_curve_to(cr, x + width, y + height - radius, x + width,
				y + height, x + width - radius, y + height);
	cairo_line_to(cr, x + radius, y + height);
	cairo_curve_to(cr, x + radius, y + height, x, y + height, x,
				y + height - radius);
	cairo_line_to(cr, x, y + radius);
	cairo_curve_to(cr, x, y + radius, x, y, x + radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill_preserve(cr);
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ActiveTextColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);
	
	cairo_destroy(cr);

	return GTK_WIDGET_CLASS (simple_fmtx_widget_plugin_parent_class)->expose_event (widget, event);
}

static void fmtx_widget_realize (GtkWidget *widget)
{
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (simple_fmtx_widget_plugin_parent_class)->realize (widget);
}

static GtkWidget *build_ui (SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);
    
    GtkWidget* fixed = gtk_fixed_new();
    GtkWidget* label = gtk_label_new("");
    priv->state_label = label;
    gtk_widget_set_size_request(label, 110, 20);
	gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0.6);

    GtkWidget* label_freq = gtk_label_new("");
    priv->label = label_freq;
    gtk_widget_set_size_request(label_freq, 110, 20);
	gtk_misc_set_alignment (GTK_MISC (label_freq), 0.5, 0.9);

    hildon_helper_set_logical_font (label, "SmallSystemFont");
    hildon_helper_set_logical_font (label_freq, "SmallSystemFont");



    /* Event box for button */
    GtkWidget *event_box2 = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(event_box2), FALSE);
    g_signal_connect(event_box2, "button-release-event", G_CALLBACK (button_released), self);
    g_signal_connect(event_box2, "button-press-event", G_CALLBACK (button_clicked), self);
    g_signal_connect(event_box2, "leave-notify-event", G_CALLBACK(event_leave_notify_button), self);


    gtk_widget_set_size_request(event_box2, 80, 80);

    /* Button images */
    gtk_fixed_put(GTK_FIXED(fixed), priv->btn_off_image, 15, 25);
    gtk_fixed_put(GTK_FIXED(fixed), priv->btn_off_pressed, 15, 25);
    gtk_fixed_put(GTK_FIXED(fixed), priv->btn_on_image, 15, 25);
    gtk_fixed_put(GTK_FIXED(fixed), priv->btn_on_pressed, 15, 25);
    gtk_widget_set_no_show_all(priv->btn_off_image, TRUE);
    gtk_widget_set_no_show_all(priv->btn_off_pressed, TRUE);
    gtk_widget_set_no_show_all(priv->btn_on_image, TRUE);
    gtk_widget_set_no_show_all(priv->btn_on_pressed, TRUE);
    
    gtk_widget_show(GTK_WIDGET(priv->btn_on_image));

    gtk_fixed_put(GTK_FIXED(fixed), event_box2, 15, 25);
    gtk_fixed_put(GTK_FIXED(fixed), label, 0, 0);
    gtk_fixed_put(GTK_FIXED(fixed), label_freq, 0, 102);

    priv->fixed = fixed;

    gtk_widget_show_all(GTK_WIDGET (fixed));

    return fixed;
}

static gboolean is_fmtx_boost_installed(void)
{
    if(g_file_test(FM_BOOST, G_FILE_TEST_EXISTS))
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
} 

static gboolean is_boost_set(SimpleFmtxWidget *self)
{
    return  get_gconf(self);
}
void select_boost_clicked(HildonCheckButton *button, gpointer *data)
{
    gboolean is_boost = hildon_check_button_get_active (HILDON_CHECK_BUTTON (button));

    set_gconf((SimpleFmtxWidget*)data, is_boost);
}
void change_freq_clicked(HildonButton *button,  gpointer *data)
{
   SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (data);

   if (priv->osso_context)
   {
	    osso_cp_plugin_execute(priv->osso_context, "libcpfmtx.so", NULL, TRUE);
	    get_current_state((SimpleFmtxWidget*)data);
        set_freq_label((SimpleFmtxWidget*)data);
   }
   return;
}

static void open_settings_dialog(HDHomePluginItem *hdhomepluginitem, gpointer user_data)
{
    block_leave_notify = TRUE;
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (hdhomepluginitem);

    HildonDialog* settings_dialog = HILDON_DIALOG (hildon_dialog_new ());

    GtkWidget *vbox = gtk_vbox_new(FALSE, 5);

    GtkWidget *label = NULL;
    GtkWidget* check_button = hildon_check_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label (GTK_BUTTON (check_button), "Boost");
    g_signal_connect(check_button, "toggled", G_CALLBACK(select_boost_clicked), hdhomepluginitem);

	GtkWidget* freq_button = hildon_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	gtk_button_set_label (GTK_BUTTON (freq_button), "Change frequency");
    g_signal_connect(freq_button, "clicked", G_CALLBACK(change_freq_clicked), hdhomepluginitem);

    if (is_fmtx_boost_installed())
    {
        priv->is_boost_installed = TRUE;
        label = gtk_label_new ("Boost will increase FM transmitter power level.\nFm-boost must be installed.");
        gtk_widget_set_sensitive(check_button, TRUE);
    }
    else
    {
        priv->is_boost_installed = FALSE;
        label = gtk_label_new ("Boost will increase FM transmitter power level.\nFm-boost must be installed.\n\nFm-boost is not installed");
        gtk_widget_set_sensitive(check_button, FALSE);
    }
    
    if (is_boost_set((SimpleFmtxWidget*)hdhomepluginitem))
    {
        hildon_check_button_set_active((HildonCheckButton*)check_button, TRUE);
    }
    else
    {
        hildon_check_button_set_active((HildonCheckButton*)check_button, FALSE);
    } 

    gtk_window_set_title (GTK_WINDOW (settings_dialog), "Settings");

    hildon_dialog_add_button (HILDON_DIALOG (settings_dialog), GTK_STOCK_OK, GTK_RESPONSE_NONE);

    gtk_container_add (GTK_CONTAINER (vbox), label);
    gtk_container_add (GTK_CONTAINER (vbox), check_button);
    gtk_container_add (GTK_CONTAINER (vbox), freq_button);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(settings_dialog)->vbox), vbox);
    gtk_widget_show_all (GTK_WIDGET (settings_dialog));

    gtk_dialog_run (GTK_DIALOG (settings_dialog));
    gtk_widget_destroy (GTK_WIDGET(settings_dialog));
    block_leave_notify = FALSE;
}
static void update_ui(SimpleFmtxWidget* self)
{
    set_freq_label(self);
    get_current_state(self);
}
static DBusHandlerResult signal_filter(DBusConnection *connection, DBusMessage *message, void *user_data)
{
    update_ui((SimpleFmtxWidget*)user_data);
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void set_dbus_listener(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE(self);
    dbus_connection_setup_with_g_main (priv->bus, NULL);
    dbus_bus_add_match (priv->bus, "type='signal',interface='com.nokia.FMTx.Device',path='/com/nokia/fmtx/default',member='Changed'", NULL);
    dbus_connection_add_filter (priv->bus, signal_filter, self, NULL);
    priv->signals_connected = TRUE;
}

static void simple_fmtx_widget_plugin_init (SimpleFmtxWidget *desktop_plugin)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (desktop_plugin);
    priv->osso_context = osso_initialize(PACKAGE, PACKAGE_VERSION, TRUE, NULL);
    
    GdkScreen *screen = NULL;
    GdkColormap *colormap = NULL;

    priv->bus = (DBusConnection*) osso_get_sys_dbus_connection (priv->osso_context);
    priv->gconf = gconf_client_get_default();
    /* Needed for transparency */
    screen = gtk_widget_get_screen(GTK_WIDGET(desktop_plugin));
    colormap = gdk_screen_get_rgba_colormap(screen);
    gtk_widget_set_app_paintable (GTK_WIDGET(desktop_plugin), TRUE);

    gtk_widget_set_colormap(GTK_WIDGET(desktop_plugin), colormap);
   
    create_images(desktop_plugin);

    set_dbus_listener(desktop_plugin);

	gtk_window_set_default_size (GTK_WINDOW (desktop_plugin), 110, 125);

    GtkWidget *contents = build_ui (desktop_plugin);

    /* Setting dialog */
    hd_home_plugin_item_set_settings((HDHomePluginItem*)desktop_plugin, TRUE);
    g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(open_settings_dialog), NULL);

    priv->is_boost_on = is_boost_set(desktop_plugin);
    priv->is_boost_installed = is_fmtx_boost_installed();
    set_freq_label(desktop_plugin);
    get_current_state(desktop_plugin);

    g_signal_connect (desktop_plugin, "notify::is-on-current-desktop",
                      G_CALLBACK (visible_notify),
                      desktop_plugin);


    gtk_container_add (GTK_CONTAINER (desktop_plugin), contents);
}
static void simple_fmtx_widget_plugin_finalize(GObject *object)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (object);
    if (priv->signals_connected  == TRUE)
    {
        dbus_connection_remove_filter(priv->bus, signal_filter,object);
        priv->signals_connected = FALSE;
    }

    osso_deinitialize(priv->osso_context);
    priv->osso_context = NULL;
    g_object_unref(priv->gconf);
    G_OBJECT_CLASS (simple_fmtx_widget_plugin_parent_class)->finalize (object); 
}

static void simple_fmtx_widget_plugin_class_init (SimpleFmtxWidgetClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);
    object_class->finalize = simple_fmtx_widget_plugin_finalize;
    g_type_class_add_private (class, sizeof (SimpleFmtxWidgetPrivate));

	widget_class->realize = fmtx_widget_realize;
	widget_class->expose_event = fmtx_widget_expose_event;
}
static void simple_fmtx_widget_plugin_class_finalize (SimpleFmtxWidgetClass *class)
{

}
static gpointer get_set(SimpleFmtxWidget *self, PropertyCommand commad, const gchar* val, gboolean state_val)
{
    DBusGConnection *connection;
    GError *error = NULL;
    DBusGProxy *proxy = NULL;
    GValue value = { 0 };
    gchar* command = NULL;

    connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (connection == NULL || error)
    {
        if (error)
        {
            g_clear_error (&error);
        }
        return 0;
    }

    proxy = dbus_g_proxy_new_for_name (connection, "com.nokia.FMTx",
                                       "/com/nokia/fmtx/default",
                                       "org.freedesktop.DBus.Properties");

    if (proxy == NULL)
    {
        return NULL;
    }

    switch (commad)
    {
        case SET:
            command = g_strdup("Set");
            break;
        case GET:
            command = g_strdup("Get");
            break;
        default:
            break;
    }

    if (g_strcmp0(val, "frequency") == 0)
    {
        dbus_g_proxy_call (proxy, command,
                           &error,
                           G_TYPE_STRING,
                           "org.freedesktop.DBus.Properties",
                           G_TYPE_STRING,
                           val,
                           G_TYPE_INVALID,
                           G_TYPE_VALUE,
                           &value,
                           G_TYPE_INVALID);

        if (error)
        {
            g_clear_error (&error);
            g_free(command);
            return NULL;
        }

        guint32 retval = g_value_get_uint(&value);
        g_value_unset (&value);
        g_free(command);
        return (gpointer)retval;
    }
    else if (g_strcmp0(val, "state") == 0)
    {
	    if (commad == GET)
	    {
	    	dbus_g_proxy_call (proxy, command,
	    	                   &error,
	    	                   G_TYPE_STRING,
	    	                   "org.freedesktop.DBus.Properties",
	    	                   G_TYPE_STRING,
	    	                   val,
	    	                   G_TYPE_INVALID,
	    	                   G_TYPE_VALUE,
	    	                   &value,
	    	                   G_TYPE_INVALID);

	    	if (error)
	    	{
	    	    g_clear_error (&error);
	    	    g_free(command);
	    	    return NULL;
	    	}

	    	gchar* retval = g_strdup(g_value_get_string(&value));
	    	g_value_unset (&value);
	    	g_free(command);
	    	return (gchar*)retval;
	    }
	    else if (commad == SET)
	    {
	    	g_value_init(&value, G_TYPE_STRING);
		    switch (state_val)
		    {
		    case FALSE:
		        	g_value_set_string(&value, "disabled");
		        break;
		    case TRUE:
		       	    g_value_set_string(&value, "enabled");
		        break;
		    default:
		        break;
		    }
		    dbus_g_proxy_call (proxy, command,
		                       &error,
		                       G_TYPE_STRING,
		                       "org.freedesktop.DBus.Properties",
		                       G_TYPE_STRING,
		                       val,
		                       G_TYPE_VALUE,
		                       &value,
		                       G_TYPE_INVALID,
		                       G_TYPE_INVALID);

		    if (error)
		    {
		        hildon_banner_show_information(NULL, NULL, error->message);
		        g_clear_error (&error);
		        g_free(command);
		        return "error";
		    }
	    }
    }
    g_free(command);
    return NULL;
}

/* Listen if desktop where widget is, is visible */
static void visible_notify (GObject *object,
                             GParamSpec *spec,
                             gpointer data)
{
    gboolean is_visible;
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (data);

    g_object_get (object, "is-on-current-desktop", &is_visible, NULL);

    if(is_visible)
    {
       //visible
        update_ui((SimpleFmtxWidget*)data);
        if (priv->signals_connected == FALSE)
        {
            set_dbus_listener(data);
        }
    }
    else
    {
       //not visible
        if (priv->signals_connected  == TRUE)
        {
            dbus_connection_remove_filter(priv->bus, signal_filter,object);
            priv->signals_connected = FALSE;
        }
    }

}
