/*
 * simple-brightness-applet - simple statusarea applet that allows the user to select the brightness from a range of 1-5
 *
 * Copyright (c) 2009-2010 Faheem Pervez <trippin1@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *	  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *	  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

//#ifdef HAVE_CONFIG_H
#include <config.h>
//#endif

#include "simple-brightness-applet.h"

#include <glib/gi18n-lib.h>
#include <string.h>
#include <libosso.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>

#ifdef HILDON_DISABLE_DEPRECATED
#undef HILDON_DISABLE_DEPRECATED
#endif

HD_DEFINE_PLUGIN_MODULE (SimpleBrightnessApplet, simple_brightness_applet, HD_TYPE_STATUS_MENU_ITEM)

#define SIMPLE_BRIGHTNESS_GCONF_PATH "/system/osso/dsm/display"
#define BRIGHTNESS_KEY SIMPLE_BRIGHTNESS_GCONF_PATH "/display_brightness"

#define SIMPLE_BRIGHTNESS_APPLET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
						SIMPLE_TYPE_BRIGHTNESS_APPLET, SimpleBrightnessAppletPrivate))

struct _SimpleBrightnessAppletPrivate
{
	GConfClient *gconf_client;
	GtkHBox *applet_contents;
	HildonControlbar *brightness_ctrlbar;
	GtkWidget *settings_dialog;
	GtkWidget *dispchkbtn;
	GtkWidget *applet_button;

	gulong brightness_ctrlbar_valchanged_id;
	gulong dispchkbtn_toggled_id;
	guint gconfnotify_id;
	guint display_keepalive_timeout;

	gboolean keepdispon_active;

	osso_context_t *osso_context;
};

static void simple_brightness_applet_finalize (GObject *object);

/* Callbacks: */
static inline void simple_brightness_applet_show_fail_msg (void)
{
	hildon_banner_show_information (NULL, NULL, _("Failed to keep backlight on"));
}

static gboolean simple_brightness_applet_keep_backlight_alive (SimpleBrightnessApplet *plugin)
{
	osso_display_state_on (plugin->priv->osso_context);
	if (osso_display_blanking_pause (plugin->priv->osso_context) != OSSO_OK)
	{
		simple_brightness_applet_show_fail_msg ();
		plugin->priv->display_keepalive_timeout = 0;
		g_signal_handler_block (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
		plugin->priv->keepdispon_active = FALSE;
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), FALSE);
		g_signal_handler_unblock (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
		return FALSE;
	}
	return TRUE;
}

static void simple_brightness_applet_on_gconf_value_changed (GConfClient *gconf_client G_GNUC_UNUSED, guint cnxn_id G_GNUC_UNUSED, GConfEntry *entry G_GNUC_UNUSED, SimpleBrightnessApplet *plugin)
{
	g_signal_handler_block (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);
	hildon_controlbar_set_value (plugin->priv->brightness_ctrlbar, gconf_client_get_int (plugin->priv->gconf_client, BRIGHTNESS_KEY, NULL));
	g_signal_handler_unblock (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);
}

static void simple_brightness_applet_on_value_changed (HildonControlbar *brightness_ctrlbar G_GNUC_UNUSED, SimpleBrightnessApplet *plugin)
{
	gconf_client_set_int (plugin->priv->gconf_client, BRIGHTNESS_KEY, hildon_controlbar_get_value (plugin->priv->brightness_ctrlbar), NULL);
}

static void simple_brightness_applet_on_settings_button_clicked (GtkWidget *button G_GNUC_UNUSED, SimpleBrightnessApplet *plugin)
{
	gtk_widget_hide (plugin->priv->settings_dialog);

	if (plugin->priv->osso_context)
		osso_cp_plugin_execute (plugin->priv->osso_context, "libcpdisplay.so", NULL, TRUE);

	gtk_widget_destroy (plugin->priv->settings_dialog);
}

static void simple_brightness_applet_on_dispchkbtn_toggled (GtkWidget *button G_GNUC_UNUSED, SimpleBrightnessApplet *plugin)
{
	GdkPixbuf *pixbuf;

	plugin->priv->keepdispon_active = hildon_check_button_get_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn));
	g_signal_handler_block (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
	gtk_widget_hide(plugin->priv->settings_dialog);

	if (plugin->priv->keepdispon_active)
	{
		simple_brightness_applet_keep_backlight_alive (plugin);
		plugin->priv->display_keepalive_timeout = g_timeout_add_seconds (50, (GSourceFunc) simple_brightness_applet_keep_backlight_alive, plugin);
		if (plugin->priv->display_keepalive_timeout == 0)
		{
			plugin->priv->keepdispon_active = FALSE;
			hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), FALSE);
			simple_brightness_applet_show_fail_msg ();
			goto finish;
		}

		gtk_widget_set_name (plugin->priv->applet_button, "hildon-reject-button-finger");
		pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (), "general_brightness", 18, 0, NULL);
		if (pixbuf)
		{
			hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), pixbuf);
			g_object_unref (pixbuf);
		}
	}
	else
	{
		if (g_source_remove (plugin->priv->display_keepalive_timeout))
		{
			gtk_widget_set_name (plugin->priv->applet_button, "GtkButton-finger");
			hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), NULL);
			plugin->priv->display_keepalive_timeout = 0;
		}
		else
		{
			plugin->priv->keepdispon_active = TRUE;
			hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), plugin->priv->keepdispon_active);
		}
	}

finish:
	g_signal_handler_unblock (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
	gtk_widget_destroy(plugin->priv->settings_dialog);
}

/* Setup applet and dialogs: */
static void simple_brightness_applet_on_button_clicked (GtkWidget *button, SimpleBrightnessApplet *plugin)
{
	GtkWidget *settings_button;

	gtk_widget_hide (gtk_widget_get_toplevel (button));

	plugin->priv->settings_dialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (plugin->priv->settings_dialog), "Simple Brightness Applet:");
	gtk_window_set_destroy_with_parent (GTK_WINDOW (plugin->priv->settings_dialog), TRUE);

	settings_button = hildon_gtk_button_new (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);
	gtk_button_set_label (GTK_BUTTON (settings_button), _("Open Display Settings..."));
	gtk_button_set_image (GTK_BUTTON (settings_button), gtk_image_new_from_icon_name ("general_brightness", HILDON_ICON_SIZE_THUMB));
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (plugin->priv->settings_dialog)->vbox), settings_button, TRUE, TRUE, 0);
	g_signal_connect (settings_button, "clicked", G_CALLBACK (simple_brightness_applet_on_settings_button_clicked), plugin);

	plugin->priv->dispchkbtn = hildon_check_button_new (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);
	gtk_button_set_label (GTK_BUTTON (plugin->priv->dispchkbtn), _("Keep display on"));
	gtk_button_set_alignment (GTK_BUTTON (plugin->priv->dispchkbtn), 0.5f, 0.5f);
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), plugin->priv->keepdispon_active);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (plugin->priv->settings_dialog)->vbox), plugin->priv->dispchkbtn, TRUE, TRUE, 0);
	plugin->priv->dispchkbtn_toggled_id = g_signal_connect (plugin->priv->dispchkbtn, "toggled", G_CALLBACK (simple_brightness_applet_on_dispchkbtn_toggled), plugin); 

	gtk_widget_show_all (plugin->priv->settings_dialog);
}

static inline void simple_brightness_applet_setup (SimpleBrightnessApplet *plugin)
{
	plugin->priv->applet_contents = GTK_HBOX (gtk_hbox_new (FALSE, 0));

	plugin->priv->applet_button = hildon_gtk_button_new (HILDON_SIZE_FINGER_HEIGHT);
	gtk_box_pack_start (GTK_BOX (plugin->priv->applet_contents), plugin->priv->applet_button, FALSE, TRUE, 0);
	g_signal_connect (plugin->priv->applet_button, "clicked", G_CALLBACK(simple_brightness_applet_on_button_clicked), plugin);
	gtk_button_set_image (GTK_BUTTON (plugin->priv->applet_button), gtk_image_new_from_icon_name ("general_brightness", HILDON_ICON_SIZE_THUMB));

	plugin->priv->brightness_ctrlbar = HILDON_CONTROLBAR (hildon_controlbar_new ()); /* Yes, I know: I'm very naughty for using a depreciated widget. But let me know, Nokia, when you actually stop using it too. */
	hildon_gtk_widget_set_theme_size(GTK_WIDGET(plugin->priv->brightness_ctrlbar), HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);
	hildon_controlbar_set_range (plugin->priv->brightness_ctrlbar, gconf_client_get_int(plugin->priv->gconf_client, "/system/osso/dsm/display/display_brightness_level_step", NULL), gconf_client_get_int(plugin->priv->gconf_client, "/system/osso/dsm/display/max_display_brightness_levels", NULL));
	hildon_controlbar_set_value (plugin->priv->brightness_ctrlbar, gconf_client_get_int(plugin->priv->gconf_client, BRIGHTNESS_KEY, NULL));
	plugin->priv->brightness_ctrlbar_valchanged_id = g_signal_connect (plugin->priv->brightness_ctrlbar, "value-changed", G_CALLBACK(simple_brightness_applet_on_value_changed), plugin);
	gtk_box_pack_start (GTK_BOX (plugin->priv->applet_contents), GTK_WIDGET (plugin->priv->brightness_ctrlbar), FALSE, TRUE, 0);

	gconf_client_add_dir (plugin->priv->gconf_client, SIMPLE_BRIGHTNESS_GCONF_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
	plugin->priv->gconfnotify_id = gconf_client_notify_add (plugin->priv->gconf_client, BRIGHTNESS_KEY, (GConfClientNotifyFunc)simple_brightness_applet_on_gconf_value_changed, plugin, NULL, NULL);
}

/* GObject stuff: */
static void simple_brightness_applet_class_init (SimpleBrightnessAppletClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = (GObjectFinalizeFunc) simple_brightness_applet_finalize;

	g_type_class_add_private (klass, sizeof (SimpleBrightnessAppletPrivate));
}

static void simple_brightness_applet_class_finalize (SimpleBrightnessAppletClass *klass G_GNUC_UNUSED) {}

static void simple_brightness_applet_init (SimpleBrightnessApplet *plugin)
{
	plugin->priv = SIMPLE_BRIGHTNESS_APPLET_GET_PRIVATE (plugin);
	if (G_UNLIKELY (!plugin->priv))
		return;
	memset (plugin->priv, 0, sizeof (SimpleBrightnessAppletPrivate));

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8"); 

	plugin->priv->keepdispon_active = FALSE;

	plugin->priv->osso_context = osso_initialize (PACKAGE, PACKAGE_VERSION, TRUE, NULL);
	if (G_UNLIKELY (!plugin->priv->osso_context))
		return;

	plugin->priv->gconf_client = gconf_client_get_default ();
	if (G_UNLIKELY (!plugin->priv->gconf_client))
		return;

	simple_brightness_applet_setup (plugin);
	gtk_container_add (GTK_CONTAINER (plugin), GTK_WIDGET (plugin->priv->applet_contents));
	gtk_widget_show_all (GTK_WIDGET (plugin));
}

static void simple_brightness_applet_finalize (GObject *object)
{
	SimpleBrightnessApplet *plugin = SIMPLE_BRIGHTNESS_APPLET (object);

	if (plugin->priv->brightness_ctrlbar_valchanged_id != 0)
		g_signal_handler_disconnect (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);

	if (plugin->priv->display_keepalive_timeout != 0)
		g_source_remove (plugin->priv->display_keepalive_timeout);

	if (plugin->priv->gconfnotify_id != 0)
	{
		gconf_client_notify_remove (plugin->priv->gconf_client, plugin->priv->gconfnotify_id);
		gconf_client_remove_dir (plugin->priv->gconf_client, SIMPLE_BRIGHTNESS_GCONF_PATH, NULL);
	}

	if (plugin->priv->gconf_client)
	{
		gconf_client_clear_cache (plugin->priv->gconf_client);
		g_object_unref (plugin->priv->gconf_client);
	}

	if (plugin->priv->osso_context)
		osso_deinitialize (plugin->priv->osso_context);

	G_OBJECT_CLASS (simple_brightness_applet_parent_class)->finalize (object);
}
