#include "summarywindowdesktop.h"
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QDebug>
#include <QMenuBar>
#include <QActionGroup>
#include <QDate>
#include <QMessageBox>
#include <QStandardItemModel>

SummaryWindowDesktop::SummaryWindowDesktop(int minYear,
                                           int minMonth,
                                           int maxYear,
                                           int maxMonth,
                                           QWidget *parent)
        : SummaryWindow (parent)
{
        setAttribute(Qt::WA_DeleteOnClose);
        setWindowFlags(windowFlags() | Qt::Window);
        setWindowTitle(tr("Month Summary"));

        summaryData = NULL;
        _minYear = minYear;
        _minMonth = minMonth;
        _maxYear = maxYear;
        _maxMonth = maxMonth;
        _startYear = 0;
        _startMonth = 0;
        _endYear = 0;
        _endMonth = 0;

        QStandardItemModel *monthModel = new QStandardItemModel(0, 1, this);
        for (int i = 0; i<12; i++) {
                QStandardItem *item = new QStandardItem (QDate::longMonthName(i + 1));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                monthModel->appendRow(item);
        }

        QStandardItemModel *yearModel = new QStandardItemModel(0, 1, this);
        for (int i = minYear; i<= maxYear; i++) {
                QStandardItem *item = new QStandardItem (QString("%1").arg(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                yearModel->appendRow(item);
        }

        /* main vbox */
        vbox = new QVBoxLayout(this);

        /* period stuff */
        QHBoxLayout *hbox = new QHBoxLayout();
        vbox->addLayout(hbox, 0);

        fromLabel = new QLabel(tr("Months from"));
        hbox->addWidget(fromLabel, 0, Qt::AlignCenter);

        startMonthCombo = new QComboBox();
        startMonthCombo->setModel(monthModel);
        startMonthCombo->setCurrentIndex(-1);
        hbox->addWidget(startMonthCombo, 1);

        startYearCombo = new QComboBox();
        startYearCombo->setModel(yearModel);
        startYearCombo->setCurrentIndex(-1);
        hbox->addWidget(startYearCombo, 1);

        toLabel = new QLabel(tr("to"));
        hbox->addWidget(toLabel, 0, Qt::AlignCenter);

        endMonthCombo = new QComboBox();
        endMonthCombo->setModel(monthModel);
        endMonthCombo->setCurrentIndex(-1);
        hbox->addWidget(endMonthCombo, 1);

        endYearCombo = new QComboBox();
        endYearCombo->setModel(yearModel);
        endYearCombo->setCurrentIndex(-1);
        hbox->addWidget(endYearCombo, 1);

        loadButton = new QPushButton(tr("Update"));
        connect(loadButton, SIGNAL(clicked()), this, SLOT(loadButtonClicked()));
        hbox->addWidget(loadButton);

        tabWidget = new QTabWidget();
        vbox->addWidget(tabWidget, 1);

        /* statistics mode stuff */
        statsBox = new QFrame();
        QGridLayout *statsLayout = new QGridLayout(statsBox);

        statsLayout->addWidget(new QLabel(tr("Period totals")), 0, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Budget")), 1, 0, 1, 1, Qt::AlignCenter);
        totalBudget = new QLabel(QString("0"));
        statsLayout->addWidget(totalBudget, 2, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Expenses")), 3, 0, 1, 1, Qt::AlignCenter);
        totalExpense = new QLabel(QString("0"));
        statsLayout->addWidget(totalExpense, 4, 0, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Saved")), 5, 0, 1, 1, Qt::AlignCenter);
        totalSaved = new QLabel(QString("0"));
        statsLayout->addWidget(totalSaved, 6, 0, 1, 1, Qt::AlignCenter);

        statsLayout->addWidget(new QLabel(tr("Period expenses")), 0, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Minimum")), 1, 1, 1, 1, Qt::AlignCenter);
        minExpense = new QLabel(QString("0"));
        statsLayout->addWidget(minExpense, 2, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Maximum")), 3, 1, 1, 1, Qt::AlignCenter);
        maxExpense = new QLabel(QString("0"));
        statsLayout->addWidget(maxExpense, 4, 1, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Average")), 5, 1, 1, 1, Qt::AlignCenter);
        averageExpense = new QLabel(QString("0"));
        statsLayout->addWidget(averageExpense, 6, 1, 1, 1, Qt::AlignCenter);

        statsLayout->addWidget(new QLabel(tr("Period saves")), 0, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Minimum")), 1, 2, 1, 1, Qt::AlignCenter);
        minSaved = new QLabel(QString("0"));
        statsLayout->addWidget(minSaved, 2, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Maximum")), 3, 2, 1, 1, Qt::AlignCenter);
        maxSaved = new QLabel(QString("0"));
        statsLayout->addWidget(maxSaved, 4, 2, 1, 1, Qt::AlignCenter);
        statsLayout->addWidget(new QLabel(tr("Average")), 5, 2, 1, 1, Qt::AlignCenter);
        averageSaved = new QLabel(QString("0"));
        statsLayout->addWidget(averageSaved, 6, 2, 1, 1, Qt::AlignCenter);

        chart = new ConceptsChart();

        graphBox = new QFrame();
        QVBoxLayout *graphLayout = new QVBoxLayout(graphBox);
        viewLabel = new QLabel(tr("Monthly accumulated view"));
        graphLayout->addWidget(viewLabel, 0, Qt::AlignCenter);
        graph = new ExpensesGraph();
        graphLayout->addWidget(graph, 1);

        tabWidget->addTab(statsBox, tr("Statistics"));
        tabWidget->addTab(chart, tr("Items"));
        tabWidget->addTab(graphBox, tr("Evolution"));
}

SummaryWindowDesktop::~SummaryWindowDesktop()
{
        if (summaryData != NULL) {
                delete summaryData;
        }
}

void SummaryWindowDesktop::loadButtonClicked()
{
        /* start date */
        _startMonth = startMonthCombo->currentIndex() + 1;
        if (_startMonth == 0) {
                message(tr("Select a starting month"));
                return;
        }

        if (startYearCombo->currentIndex() == -1) {
                message(tr("Select a starting year"));
                return;
        } else {
                _startYear = startYearCombo->currentText().toInt();
        }

        if (_startYear == _minYear && _startMonth < _minMonth) {
                message(tr("Minimum date is %1 %2")
                        .arg(QDate::longMonthName(_minMonth))
                        .arg(_minYear));
                return;
        }

        if (_startYear == _maxYear && _startMonth > _maxMonth) {
                message(tr("Maximum date is %1 %2")
                        .arg(QDate::longMonthName(_minMonth))
                        .arg(_minYear));
                return;
        }

        /* end date */
        _endMonth = endMonthCombo->currentIndex() + 1;
        if (_endMonth == 0) {
                message(tr("Select a finish month"));
                return;
        }

        if (endYearCombo->currentIndex() == -1) {
                message(tr("Select a finish year"));
                return;
        } else {
                _endYear = endYearCombo->currentText().toInt();
        }

        if (_endYear == _minYear && _endMonth < _minMonth) {
                message(tr("Minimum date is %1 %2")
                        .arg(QDate::longMonthName(_minMonth))
                        .arg(_minYear));
                return;
        }

        if (_endYear == _maxYear && _endMonth > _maxMonth) {
                message(tr("Maximum date is %1 %2")
                        .arg(QDate::longMonthName(_minMonth))
                        .arg(_minYear));
                return;
        }

        /* starting date must be lower or equal to end date */
        if (QDate(_startYear, _startMonth, 1) > QDate(_endYear, _endMonth, 1)) {
                message(tr("Start date must be prior to end date"));
                return;
        }

        emit periodChanged(_startYear, _startMonth, _endYear, _endMonth);
}

void SummaryWindowDesktop::setSummaryData(SummaryData *data)
{
        if (summaryData != NULL) {
                delete summaryData;
        }

        summaryData = data;

        totalBudget->setText(QString("%1").arg(summaryData->totalBudget()));
        totalExpense->setText(QString("%1").arg(summaryData->totalExpense()));

        if (summaryData->totalSaved() > 0) {
                totalSaved->setText(QString("<font color='green'>%1</font>")
                                    .arg(summaryData->totalSaved()));
        } else {
                totalSaved->setText(QString("<font color='red'>%1</font>")
                                    .arg(summaryData->totalSaved()));
        }

        minExpense->setText(QString("%1").arg(summaryData->minExpense()));
        maxExpense->setText(QString("%1").arg(summaryData->maxExpense()));
        averageExpense->setText(QString("%1").arg(summaryData->averageExpense()));

        minSaved->setText(QString("%1").arg(summaryData->minSaved()));
        maxSaved->setText(QString("%1").arg(summaryData->maxSaved()));

        if (summaryData->averageSaved() > 0) {
                averageSaved->setText(QString("<font color='green'>%1</font>")
                                    .arg(summaryData->averageSaved()));
        } else {
                averageSaved->setText(QString("<font color='red'>%1</font>")
                                    .arg(summaryData->averageSaved()));
        }

        chart->setSummaryData(summaryData);
        if (chart->isVisible()) {
                chart->repaint();
        }

        graph->setModel(summaryData);
        if (graph->isVisible()) {
                graph->repaint();
        }

        if (summaryData->totalExpense() == 0) {
                message(tr("Warning: there are no expenses stored in the selected period"));
        }
}

void SummaryWindowDesktop::mousePressEvent(QMouseEvent *event)
{
        if (graph->isVisible()) {
                graph->changeView();
                if (graph->accumulated()) {
                        viewLabel->setText(tr("Monthly accumulated view"));
                } else {
                        viewLabel->setText(tr("Month expenses view"));
                }
        }
}

void SummaryWindowDesktop::message(QString text)
{
        QMessageBox msgBox(this);
        msgBox.setText(text);
        msgBox.exec();
}

QSize SummaryWindowDesktop::minimumSizeHint() const
{
        return QSize(800, 400);
}

QSize SummaryWindowDesktop::sizeHint() const
{
        return QSize(800, 400);
}
