#include "expensedetailsdialogdesktop.h"
#include <QVBoxLayout>
#include <QFormLayout>
#include <QLabel>
#include <QDebug>
#include <QStandardItem>
#include <QMessageBox>
#include <QPushButton>

ExpenseDetailsDialogDesktop::ExpenseDetailsDialogDesktop(const QDate &date,
                                                         QStandardItemModel *concepts,
                                                         QWidget *parent, Qt::WindowFlags f)
        : ExpenseDetailsDialog (parent, f)
{
        newExpense = true;
        expenseId = 0;

        setWindowTitle(tr("Add new expense"));

        buildUI (date, concepts);

        if (date.year() == QDate::currentDate().year()
            && date.month() == QDate::currentDate().month()) {
                    /* current month. set default day in the picker */
                    day->setCurrentIndex(date.day() - 1);
        } else {
                day->setCurrentIndex(-1);
        }
        concept->setCurrentIndex(-1);
}


ExpenseDetailsDialogDesktop::ExpenseDetailsDialogDesktop(const QDate &date,
                                                         QStandardItemModel *concepts,
                                                         QStandardItemModel *expenses,
                                                         int row,
                                                         QWidget *parent,
                                                         Qt::WindowFlags f)
        : ExpenseDetailsDialog (parent, f)
{
        newExpense = false;
        expenseId = expenses->item(row, 0)->text().toInt();

        setWindowTitle(tr("Expense details"));

        buildUI(date, concepts);

        QPushButton *deleteButton = new QPushButton(tr("Delete"));
        box->addButton(deleteButton, QDialogButtonBox::ActionRole);
        connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));

        /* day */
        day->setCurrentIndex(expenses->item(row, 1)->text().toInt() - 1);

        /* amount */
        amount->setText(expenses->item(row, 2)->text());

        /* description */
        description->setText(expenses->item(row, 4)->text());

        /* concept */
        /* concept may be empty, so we need to check it */
        if (!expenses->item(row, 3)->text().isEmpty()) {
                QList<QStandardItem*> list = concepts->findItems(expenses->item(row, 3)->text(),
                                                                 Qt::MatchFixedString,
                                                                 0);
                concept->setCurrentIndex(list[0]->index().row());
        }
}

void ExpenseDetailsDialogDesktop::buildUI (const QDate &date,
                                           QStandardItemModel *concepts)
{
        QVBoxLayout *layout = new QVBoxLayout (this);

        QFormLayout *form = new QFormLayout ();
        form->setHorizontalSpacing(50);
        layout->addLayout (form, 1);

        QDate currentDate = QDate::currentDate();
        bool isCurrentMonth;
        if (date.month() == currentDate.month()
            && date.year() == currentDate.year()) {
                isCurrentMonth = true;
        } else {
                isCurrentMonth = false;
        }
        QStandardItemModel *dayModel = new QStandardItemModel (0, 2, this);
        for (int i=1; i <= date.daysInMonth(); i++) {
                const QDate _date = QDate(date.year(), date.month(), i);
                QStandardItem *dayName = new QStandardItem (humanizeDay(_date, currentDate, isCurrentMonth));
                dayName->setTextAlignment(Qt::AlignCenter);
                dayName->setEditable(false);
                QStandardItem *dayNumber = new QStandardItem(QString("%1").arg(i));
                dayModel->setItem(i - 1, 0, dayName);
                dayModel->setItem(i - 1, 1, dayNumber);
        }
        day = new QComboBox();
        day->setModel(dayModel);
        day->setModelColumn(0);
        form->addRow (new QLabel(tr("Day")), day);

        concept = new QComboBox();
        concept->setModel(concepts);
        concept->setModelColumn(0);
        form->addRow (new QLabel(tr("Item")), concept);

        amount = new QLineEdit();
        form->addRow (new QLabel(tr("Amount")), amount);

        description = new QLineEdit();
        form->addRow (new QLabel(tr("Description")), description);

        box = new QDialogButtonBox (QDialogButtonBox::Cancel, Qt::Horizontal);
        connect(box, SIGNAL(rejected()), this, SLOT(reject()));
        layout->addWidget (box, 0, Qt::AlignBottom);

        QPushButton *saveButton = new QPushButton(tr("Save"));
        box->addButton(saveButton, QDialogButtonBox::ActionRole);
        connect(saveButton, SIGNAL(clicked()), this, SLOT(checkValues()));
}

QString ExpenseDetailsDialogDesktop::humanizeDay(const QDate &date,
                                               const QDate &currentDate,
                                               bool isCurrentMonth)
{
        int currentDay = currentDate.day();
        int day = date.day();

        QString humanizedDay = QString("%1, %2").arg(date.toString("ddd")).arg(date.day());

        if (isCurrentMonth) {
                if (day == currentDay) {
                        humanizedDay = QString(tr("Today"));
                } else if (currentDay - day == 1) {
                        humanizedDay = QString(tr("Yesterday"));
                } else if (currentDay + 1 == day) {
                                humanizedDay = QString(tr("Tomorrow"));
                }
        }

	return humanizedDay;
}

ExpenseDetailsDialogDesktop::~ExpenseDetailsDialogDesktop()
{
}

void ExpenseDetailsDialogDesktop::checkValues()
{
        Expense *e;

        int dayIndex = day->currentIndex();
        QStandardItemModel *dayModel = dynamic_cast<QStandardItemModel*>(day->model());
        int _day = dayModel->item(dayIndex, 1)->text().toInt();

        if (_day == 0) {
                QMessageBox msgBox(this);
                msgBox.setText(tr("Invalid day"));
                msgBox.exec();
                return;
        }

        int conceptIndex = concept->currentIndex();
        QString conceptName;
        if (conceptIndex >= 0) {
                QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(concept->model());
                conceptName.append(m->item(conceptIndex, 0)->text());
        }

        QString _description(description->text());
        if (conceptIndex < 0 && _description.isEmpty()) {
                QMessageBox msgBox(this);
                msgBox.setText(tr("Choose a concept or enter a description"));
                msgBox.exec();
                return;
        }

        double _amount(amount->text().toDouble());
        if (_amount <= 0) {
                QMessageBox msgBox(this);
                msgBox.setText(tr("Invalid amount"));
                msgBox.exec();
                return;
        }

        if (conceptIndex < 0) {
                e = new Expense (expenseId, _day, _amount, "", _description);
        } else {
                /* amount and concept */
                e = new Expense (expenseId, _day, _amount, conceptName, _description);
        }

        accept();
        if (newExpense) {
                emit newExpenseConfirmed(e);
        } else {
                emit updateExpenseConfirmed(e);
        }
}

void ExpenseDetailsDialogDesktop::deleteClicked()
{
        emit deleteExpenseConfirmed (new Expense (expenseId,0,0,"",""));
        accept();
}

QSize ExpenseDetailsDialogDesktop::minimumSizeHint() const
{
        return QSize(400, 200);
}

QSize ExpenseDetailsDialogDesktop::sizeHint() const
{
        return QSize(400, 200);
}
