#include <QDebug>
#include <QStandardItem>
#include <QDir>
#include "controller.h"

#ifdef Q_WS_MAEMO_5
#include "maemo/accountdetailsdialogmaemo.h"
#include "maemo/accountsdialogmaemo.h"
#include "maemo/budgetdialogmaemo.h"
#include "maemo/conceptdetailsdialogmaemo.h"
#include "maemo/conceptsdialogmaemo.h"
#include "maemo/confirmationdialogmaemo.h"
#include "maemo/expensedetailsdialogmaemo.h"
#include "maemo/scheduleddetailsdialogmaemo.h"
#include "maemo/scheduledlistmaemo.h"
#include "maemo/summarywindowmaemo.h"
#include "maemo/expenseslistmaemo.h"
#include "maemo/mainwindowmaemo.h"
#else
#include "desktop/mainwindowdesktop.h"
#include "desktop/accountsdialogdesktop.h"
#include "desktop/accountdetailsdialogdesktop.h"
#include "desktop/budgetdialogdesktop.h"
#include "desktop/conceptsdialogdesktop.h"
#include "desktop/conceptdetailsdialogdesktop.h"
#include "desktop/confirmationdialogdesktop.h"
#include "desktop/expenseslistdesktop.h"
#include "desktop/scheduledlistdesktop.h"
#include "desktop/expensedetailsdialogdesktop.h"
#include "desktop/scheduleddetailsdialogdesktop.h"
#include "desktop/summarywindowdesktop.h"
#endif

Controller::Controller()
{
        QString configFile(QDir::home().path());
        configFile.append(QDir::separator()).append(DATA_FOLDER);
        configFile.append(QDir::separator()).append(CONFIG_FILENAME);

        settings = new QSettings (configFile, QSettings::NativeFormat, this);
        concepts = new QStandardItemModel(0, 2, this);
        scheduled = new QStandardItemModel(0, 7, this);
        accounts = new QStandardItemModel(0, 4, this);

#ifdef Q_WS_MAEMO_5
        mainWindow = new MainWindowMaemo();
#else
        mainWindow = new MainWindowDesktop();
        createModelHeaders();
#endif


        mainWindow->show();

        loadConcepts();
        loadAccounts();

        activeAccount = NULL;
        month = NULL;

        if (accounts->rowCount() == 0) {
                /* no accounts in the database */
                mainWindow->message(tr("You must create an account to be able to write down expenses"));
        } else {
                int activeAccountId = settings->value(ACTIVE_ACCOUNT_KEY, 0).toInt();
                QList<QStandardItem*> list = accounts->findItems(QString("%1").arg(activeAccountId),
                                                                 Qt::MatchFixedString,
                                                                 0);
                if (list.size() == 0) {
                        /* can only happen if the account is manually deleted from the database */
                        mainWindow->message(tr("There's no valid account selected"));
                } else {
                        int row = list[0]->index().row();
                        activeAccount = new Account(accounts->item(row, 0)->text().toInt(),
                                                    accounts->item(row, 1)->text(),
                                                    accounts->item(row, 2)->text().toDouble(),
                                                    this);
                        QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                        accounts->item(row, 3)->setIcon(icon);
                        /* everything went fine. Load account dependant info */
                        loadScheduled();
                        month = new MonthData(QDate::currentDate(), activeAccount, this);
                        loadMonth();

                        mainWindow->setMonthData(month);
                }
        }

        connect(mainWindow, SIGNAL(previousMonth()), this, SLOT(previousMonth()));
        connect(mainWindow, SIGNAL(nextMonth()), this, SLOT(nextMonth()));
        connect(mainWindow, SIGNAL(addExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(mainWindow, SIGNAL(budgetSelected()), this, SLOT(budgetSelected()));
        connect(mainWindow, SIGNAL(accountsSelected()), this, SLOT(accountsSelected()));
        connect(mainWindow, SIGNAL(conceptsSelected()), this, SLOT(conceptsSelected()));
        connect(mainWindow, SIGNAL(expensesSelected()), this, SLOT(expensesSelected()));
        connect(mainWindow, SIGNAL(scheduledSelected()), this, SLOT(scheduledSelected()));
        connect(mainWindow, SIGNAL(summarySelected()), this, SLOT(summarySelected()));
}

Controller::~Controller()
{
        delete mainWindow;
}

void Controller::previousMonth()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

        MonthData *old = month;
        month = new MonthData(old->previous(), activeAccount, this);
        loadMonth();
        mainWindow->setMonthData(month);
        delete old;
}

void Controller::nextMonth()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

        if (month->next() <= QDate::currentDate()) {
                MonthData *old = month;
                month = new MonthData(old->next(), activeAccount, this);
                loadMonth();
                mainWindow->setMonthData(month);
                delete old;
        } else {
                mainWindow->message(tr("Already at current month"));
        }
}

/* ------------------ account dialog functions ---------------------*/

void Controller::newAccountSelected()
{
#ifdef Q_WS_MAEMO_5
        AccountDetailsDialogMaemo dialog(mainWindow);
#else
        AccountDetailsDialogDesktop dialog(mainWindow);
#endif

        connect(&dialog, SIGNAL(newAccountConfirmed(Account*)), this, SLOT(newAccountConfirmed(Account*)));

        dialog.exec();
}

void Controller::newAccountConfirmed(Account *a)
{
        if (!db.exists(a)) {
                if (db.create(a)) {
                        loadAccounts();
                        if (accounts->rowCount() == 1) {
                                /* this is the first account in the system. Activate it */
                                if (activeAccount != NULL) {
                                        delete activeAccount;
                                }
                                activeAccount = new Account(accounts->item(0, 0)->text().toInt(),
                                                            accounts->item(0, 1)->text(),
                                                            accounts->item(0, 2)->text().toDouble(),
                                                            this);
                                settings->setValue(ACTIVE_ACCOUNT_KEY, activeAccount->id());
                                QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                                accounts->item(0, 3)->setIcon(icon);
                                loadScheduled();
                                month = new MonthData(QDate::currentDate(), activeAccount, this);
                                loadMonth();
                                mainWindow->setMonthData(month);
                        }

                } else {
                        mainWindow->message(tr("Error creating account"));
                }
        } else {
                mainWindow->message(tr("Account already exists"));
        }
        delete a;
}

void Controller::accountsSelected()
{
#ifdef Q_WS_MAEMO_5
        AccountsDialogMaemo dialog(accounts, mainWindow);
#else
        AccountsDialogDesktop dialog(accounts, mainWindow);
#endif
        connect(&dialog, SIGNAL(newAccountSelected()), this, SLOT(newAccountSelected()));
        connect(&dialog, SIGNAL(accountDetailsSelected(QModelIndex)), this, SLOT(accountDetailsSelected(QModelIndex)));

        dialog.exec();
}

void Controller::accountDetailsSelected(QModelIndex index)
{
#ifdef Q_WS_MAEMO_5
        AccountDetailsDialogMaemo dialog(accounts, index.row(), activeAccount, mainWindow);
#else
        AccountDetailsDialogDesktop dialog(accounts, index.row(), activeAccount, mainWindow);
#endif

        connect(&dialog, SIGNAL(deleteAccountConfirmed(Account*)),
                this, SLOT(deleteAccountConfirmed(Account*)));
        connect(&dialog, SIGNAL(updateAccountConfirmed(Account*)),
                this, SLOT(updateAccountConfirmed(Account*)));
        connect(&dialog, SIGNAL(selectAccountConfirmed(Account*)),
                this, SLOT(selectAccountConfirmed(Account*)));

        dialog.exec();
}

void Controller::deleteAccountConfirmed(Account *a)
{
        if (a->id() == activeAccount->id()) {
                mainWindow->message(tr("Cannot delete active account"));
                delete a;
                return;
        }

        if (db.remove(a)) {
                loadAccounts();
        } else {
                mainWindow->message(tr("Error deleting account"));
        }
        delete a;
}

void Controller::updateAccountConfirmed(Account *a)
{
        if (db.update(a)) {
                loadAccounts();
        } else {
                mainWindow->message(tr("Error updating account"));
        }

        delete a;
}

void Controller::selectAccountConfirmed(Account *a)
{
        if (activeAccount != NULL) {
                delete activeAccount;
        }
        activeAccount = a;
        activeAccount->setParent(this);
        settings->setValue(ACTIVE_ACCOUNT_KEY, a->id());
        loadAccounts();
        loadScheduled();

        if (month != NULL) {
                delete month;
        }
        month = new MonthData(QDate::currentDate(), activeAccount, this);
        loadMonth();
        mainWindow->setMonthData(month);
}


/* ------------------ add expense dialog functions ---------------------*/

void Controller::newExpenseSelected()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

#ifdef Q_WS_MAEMO_5
        ExpenseDetailsDialogMaemo dialog(month->date(), concepts, (QWidget*)QObject::sender());
#else
        ExpenseDetailsDialogDesktop dialog(month->date(), concepts, (QWidget*)QObject::sender());
#endif
        connect(&dialog, SIGNAL(newExpenseConfirmed(Expense*)), this, SLOT(newExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::newExpenseConfirmed(Expense *e)
{
        if (db.create(month->year(), month->month(), activeAccount->id(),  e)) {
                loadMonth(true);
        }
        delete e;
}

void Controller::expensesSelected()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

#ifdef Q_WS_MAEMO_5
        ExpensesList *list = new ExpensesListMaemo (month->expenses(), mainWindow);
#else
        ExpensesList *list = new ExpensesListDesktop (month->expenses(), mainWindow);
#endif

        connect(list, SIGNAL(newExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(list, SIGNAL(expenseDetailsSelected(QModelIndex)), this, SLOT(expenseDetailsSelected(QModelIndex)));

        list->show();

}

void Controller::expenseDetailsSelected(QModelIndex index)
{
#ifdef Q_WS_MAEMO_5
        ExpenseDetailsDialogMaemo dialog(month->date(), concepts, month->expenses(), index.row(), (QWidget*)QObject::sender());
#else
        ExpenseDetailsDialogDesktop dialog(month->date(), concepts, month->expenses(), index.row(), (QWidget*)QObject::sender());
#endif

        connect(&dialog, SIGNAL(deleteExpenseConfirmed(Expense*)),
                this, SLOT(deleteExpenseConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateExpenseConfirmed(Expense*)),
                this, SLOT(updateExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::deleteExpenseConfirmed(Expense *e)
{
        if (db.remove(e)) {
                loadMonth(true);
        } else {
                mainWindow->message(tr("Error deleting expense"));
        }

        delete e;
}

void Controller::updateExpenseConfirmed(Expense *e)
{
        if (db.update(e)) {
                loadMonth(true);
        } else {
                mainWindow->message(tr("Error updating expense"));
        }

        delete e;
}

/* ------------------ budget dialog functions ---------------------*/

void Controller::budgetSelected()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

#ifdef Q_WS_MAEMO_5
        BudgetDialogMaemo dialog(month->budget(), mainWindow);
#else
        BudgetDialogDesktop dialog(month->budget(), mainWindow);
#endif

        connect(&dialog, SIGNAL(changeBudget(double)), this, SLOT(changeBudgetConfirmed(double)));

        dialog.exec();
}

void Controller::changeBudgetConfirmed (double budget)
{
        if (db.updateBudget(month->year(), month->month(), activeAccount->id(), budget)) {
                month->setBudget(budget);
        }
}

/* ------------------ concept management dialog functions ---------------------*/

void Controller::conceptsSelected()
{
#ifdef Q_WS_MAEMO_5
        ConceptsDialogMaemo dialog(concepts, mainWindow);
#else
        ConceptsDialogDesktop dialog(concepts, mainWindow);
#endif

        connect(&dialog, SIGNAL(newConceptSelected()), this, SLOT(newConceptSelected()));
        connect(&dialog, SIGNAL(conceptDetailsSelected(QModelIndex)), this, SLOT(conceptDetailsSelected(QModelIndex)));

        dialog.exec();
}

void Controller::newConceptSelected()
{
#ifdef Q_WS_MAEMO_5
        ConceptDetailsDialogMaemo dialog(mainWindow);
#else
        ConceptDetailsDialogDesktop dialog(mainWindow);
#endif
        connect(&dialog, SIGNAL(newConceptConfirmed(Concept*)), this, SLOT(newConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::conceptDetailsSelected(const QModelIndex &index)
{
#ifdef Q_WS_MAEMO_5
        ConceptDetailsDialogMaemo dialog(concepts, index.row(), mainWindow);
#else
        ConceptDetailsDialogDesktop dialog(concepts, index.row(), mainWindow);
#endif
        connect(&dialog, SIGNAL(deleteConceptConfirmed(Concept*)),
                this, SLOT(deleteConceptConfirmed(Concept*)));
        connect(&dialog, SIGNAL(updateConceptConfirmed(Concept*)),
                this, SLOT(updateConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::newConceptConfirmed(Concept *c)
{
        if (!db.exists(c)) {
                if (db.create(c)) {
                        loadConcepts();
                } else {
                        mainWindow->message(tr("Error creating concept"));
                }
        } else {
                mainWindow->message(tr("Concept already exists"));
        }
        delete c;
}

void Controller::deleteConceptConfirmed(Concept *c)
{
        if (db.remove(c)) {
                loadConcepts();
                if (activeAccount) {
                        loadScheduled();
                        loadMonth(true);
                }
        } else {
                mainWindow->message(tr("Error deleting concept"));
        }
        delete c;
}

void Controller::updateConceptConfirmed(Concept *c)
{
        if (db.update(c)) {
                loadConcepts();
        } else {
                mainWindow->message(tr("Error updating concept"));
        }

        delete c;
}

/* ------------------ scheduled expenses functions ---------------------*/

void Controller::scheduledSelected()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

#ifdef Q_WS_MAEMO_5
        ScheduledListMaemo *list = new ScheduledListMaemo (scheduled, mainWindow);
#else
        ScheduledListDesktop *list = new ScheduledListDesktop (scheduled, mainWindow);
#endif

        connect(list, SIGNAL(newScheduledSelected()), this, SLOT(newScheduledSelected()));
        connect(list, SIGNAL(scheduledDetailsSelected(QModelIndex)), this, SLOT(scheduledDetailsSelected(QModelIndex)));

        list->show();
}

void Controller::newScheduledSelected()
{
#ifdef Q_WS_MAEMO_5
        ScheduledDetailsDialogMaemo dialog(concepts, (QWidget*)QObject::sender());
#else
        ScheduledDetailsDialogDesktop dialog(concepts, (QWidget*)QObject::sender());
#endif

        connect(&dialog, SIGNAL(newScheduledConfirmed(Expense*, bool)), this, SLOT(newScheduledConfirmed(Expense*, bool)));

        dialog.exec();
}

void Controller::scheduledDetailsSelected(QModelIndex index)
{
#ifdef Q_WS_MAEMO_5
        ScheduledDetailsDialogMaemo dialog(concepts, scheduled, index.row(), (QWidget*)QObject::sender());
#else
        ScheduledDetailsDialogDesktop dialog(concepts, scheduled, index.row(), (QWidget*)QObject::sender());
#endif

        connect(&dialog, SIGNAL(deleteScheduledConfirmed(Expense*)),
                this, SLOT(deleteScheduledConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateScheduledConfirmed(Expense*, bool)),
                this, SLOT(updateScheduledConfirmed(Expense*, bool)));

        dialog.exec();
}

void Controller::newScheduledConfirmed(Expense *e, bool createInCurrentMonth)
{
        if (db.createScheduled(activeAccount->id(), e)) {
                loadScheduled();
                if (createInCurrentMonth) {
                    for (int day = 1; day <= 31; day++) {
                        if (e->day()  & (1 << (day - 1))) {
                            Expense *newExpense = new Expense (0, day, e->amount(), e->concept(), e->description());
                            db.create(QDate::currentDate().year(), QDate::currentDate().month(),
                                      activeAccount->id(), newExpense);
                            delete newExpense;
                        }
                    }
                    loadMonth(true);
                }
        } else {
                mainWindow->message(tr("Error creating scheduled expense"));
        }

        delete e;
}

void Controller::deleteScheduledConfirmed(Expense *e)
{
        if (db.removeScheduled(e)) {
                loadScheduled();
        } else {
                mainWindow->message(tr("Error deleting scheduled expense"));
        }

        delete e;
}

void Controller::updateScheduledConfirmed(Expense *e, bool createInCurrentMonth)
{
        /* store the previous frecuency value from the updated scheduled */
        QList<QStandardItem*> list = scheduled->findItems(QString("%1").arg(e->id()),
                                                          Qt::MatchFixedString,
                                                          0);
        if (db.updateScheduled(e)) {
                loadScheduled();
                if (createInCurrentMonth) {
                    for (int day = 1; day <= 31; day++) {
                        if (e->day()  & (1 << (day - 1))) {
                            Expense *newExpense = new Expense (0, day, e->amount(), e->concept(), e->description());
                            db.create(QDate::currentDate().year(), QDate::currentDate().month(),
                                      activeAccount->id(), newExpense);
                            delete newExpense;
                        }
                    }
                    loadMonth(true);
                }
        } else {
                mainWindow->message(tr("Error updating scheduled expense"));
        }

        delete e;
}

/*---------------- query  funcions -------------*/

void Controller::summarySelected()
{
        if (!activeAccount) {
                mainWindow->message(tr("An account must be selected to perform this action"));
                return;
        }

        int minYear, minMonth, maxYear, maxMonth;

        if (db.getDataInterval(activeAccount->id(), minYear, minMonth, maxYear, maxMonth)) {
#ifdef Q_WS_MAEMO_5
                summaryWindow = new SummaryWindowMaemo(minYear, minMonth, maxYear, maxMonth, mainWindow);
#else
                summaryWindow = new SummaryWindowDesktop(minYear, minMonth, maxYear, maxMonth, mainWindow);
#endif

                connect (summaryWindow, SIGNAL(periodChanged(int, int, int, int)),
                         this, SLOT(summaryPeriodChanged(int, int, int, int)));

                summaryWindow->show();
        } else {
                mainWindow->message(tr("Error loading data interval"));
        }
}

void Controller::summaryPeriodChanged(int startYear, int startMonth, int endYear, int endMonth)
{
        SummaryData *data = new SummaryData(startYear, startMonth, endYear, endMonth);
        if (db.load(activeAccount->id(), data)) {
                summaryWindow->setSummaryData(data);
        } else {
                mainWindow->message(tr("Error loading interval data"));
                delete data;
        }
}

/*---------------- util funcions -------------*/

void Controller::loadMonth(bool reload)
{
        month->enableExpenseNotifications(false);
        if (reload) {
                month->clear();
                db.load(month, true);
        } else {
                if (db.exists(month)) {
                        db.load (month, false);
                } else {
                        db.create(month);
                        /* create scheduled expenses in new month, but only if it is the current one */
                        if (month->isCurrent()) {
                                int i;
                                for (i = 0; i < scheduled->rowCount(); i++) {
                                        int frec = scheduled->item(i, SCHEDULED_MONTH)->text().toInt();
                                        if (frec & (1 << (month->month() - 1))) {
                                            for (int day = 1; day <= 31; day++) {
                                                if (scheduled->item(i, SCHEDULED_DAY)->text().toInt() & (1 << (day - 1))) {
                                                    /* check if date is valid before creating the expense because of leap years */
                                                    if (QDate::isValid(month->year(),
                                                                       month->month(),
                                                                       day)) {
                                                        Expense *e = new Expense(0,
                                                                                 day,
                                                                                 scheduled->item(i, SCHEDULED_AMOUNT)->text().toDouble(),
                                                                                 scheduled->item(i, SCHEDULED_CONCEPT)->text(),
                                                                                 scheduled->item(i, SCHEDULED_DESCRIPTION)->text());
                                                        db.create(month->year(), month->month(), activeAccount->id(), e);
                                                        delete e;
                                                    }
                                                }
                                            }
                                        }
                                }
                                db.load(month, true);
                        }
                }
        }
        month->enableExpenseNotifications(true);
}

void Controller::loadConcepts()
{
        concepts->removeRows(0, concepts->rowCount());
        db.loadConcepts(concepts);
}

void Controller::loadScheduled()
{
        scheduled->removeRows(0, scheduled->rowCount());
        db.loadScheduled(activeAccount->id(), scheduled);
}

void Controller::loadAccounts()
{
        accounts->removeRows(0, accounts->rowCount());
        db.loadAccounts(accounts);

        if (activeAccount) {
                QList<QStandardItem*> list = accounts->findItems(QString("%1").arg(activeAccount->id()),
                                                                 Qt::MatchFixedString,
                                                                 0);
                if (list.size() != 0) {
                        int row = list[0]->index().row();
                        QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                        accounts->item(row, 3)->setIcon(icon);
                }
        }
}

void Controller::createModelHeaders()
{
        QStandardItem *header;

        /* account model headers */
        header = new QStandardItem (tr("Id"));
        header->setTextAlignment(Qt::AlignCenter);
        accounts->setHorizontalHeaderItem(0, header);
        header = new QStandardItem (tr("Name"));
        header->setTextAlignment(Qt::AlignCenter);
        accounts->setHorizontalHeaderItem(1, header);
        header = new QStandardItem (tr("Budget"));
        header->setTextAlignment(Qt::AlignCenter);
        accounts->setHorizontalHeaderItem(2, header);
        header = new QStandardItem (tr("Active"));
        header->setTextAlignment(Qt::AlignCenter);
        accounts->setHorizontalHeaderItem(3, header);

        /* concepts model headers */
        header = new QStandardItem (tr("Name"));
        header->setTextAlignment(Qt::AlignCenter);
        concepts->setHorizontalHeaderItem(0, header);
        header = new QStandardItem (tr("Price"));
        header->setTextAlignment(Qt::AlignCenter);
        concepts->setHorizontalHeaderItem(1, header);

        /* scheduled model headers */
        header = new QStandardItem (tr("Day"));
        header->setTextAlignment(Qt::AlignCenter);
        scheduled->setHorizontalHeaderItem(SCHEDULED_DAY_AS_STRING, header);
        header = new QStandardItem (tr("Amount"));
        header->setTextAlignment(Qt::AlignCenter);
        scheduled->setHorizontalHeaderItem(SCHEDULED_AMOUNT, header);
        header = new QStandardItem (tr("Item"));
        header->setTextAlignment(Qt::AlignCenter);
        scheduled->setHorizontalHeaderItem(SCHEDULED_CONCEPT, header);
        header = new QStandardItem (tr("Description"));
        header->setTextAlignment(Qt::AlignCenter);
        scheduled->setHorizontalHeaderItem(SCHEDULED_DESCRIPTION, header);
        header = new QStandardItem (tr("Repeat"));
        header->setTextAlignment(Qt::AlignCenter);
        scheduled->setHorizontalHeaderItem(SCHEDULED_MONTH_AS_STRING, header);
}
