#include <QDebug>
#include <QMaemo5InformationBox>
#include <QStandardItem>
#include <QDir>
#include <QDebug>
#include "expensedetailsdialog.h"
#include "scheduleddetailsdialog.h"
#include "budgetdialog.h"
#include "conceptsdialog.h"
#include "conceptdetailsdialog.h"
#include "expenseslist.h"
#include "controller.h"
#include "scheduledlist.h"
#include "confirmationdialog.h"
#include "summarywindow.h"
#include "datechooserdialog.h"
#include "accountsdialog.h"
#include "accountdetailsdialog.h"

Controller::Controller()
{
        QString configFile(QDir::home().path());
        configFile.append(QDir::separator()).append(DATA_FOLDER);
        configFile.append(QDir::separator()).append(CONFIG_FILENAME);

        settings = new QSettings (configFile, QSettings::NativeFormat, this);
        concepts = new QStandardItemModel(0, 0, this);
        accounts = new QStandardItemModel(0, 4, this);
        scheduled = new QStandardItemModel(0, 7, this);

        loadConcepts();
        loadAccounts();

        activeAccount = NULL;
        month = NULL;

        if (accounts->rowCount() == 0) {
                /* no accounts in the database */
                QMaemo5InformationBox::information(&mainWindow, tr("You must create an account to be able to write down expenses"));
        } else {
                int activeAccountId = settings->value(ACTIVE_ACCOUNT_KEY, 0).toInt();
                QList<QStandardItem*> list = accounts->findItems(QString("%1").arg(activeAccountId),
                                                                 Qt::MatchFixedString,
                                                                 0);
                if (list.size() == 0) {
                        /* can only happen if the account is manually deleted from the database */
                        QMaemo5InformationBox::information(&mainWindow, tr("There's no valid account selected"));
                } else {
                        int row = list[0]->index().row();
                        activeAccount = new Account(accounts->item(row, 0)->text().toInt(),
                                                    accounts->item(row, 1)->text(),
                                                    accounts->item(row, 2)->text().toDouble(),
                                                    this);
                        QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                        accounts->item(row, 3)->setIcon(icon);
                        /* everything went fine. Load account dependant info */
                        loadScheduled();
                        month = new MonthData(QDate::currentDate(), activeAccount, this);
                        loadMonth();

                        mainWindow.setMonthData(month);
                }
        }

        connect(&mainWindow, SIGNAL(previousMonth()), this, SLOT(previousMonth()));
        connect(&mainWindow, SIGNAL(nextMonth()), this, SLOT(nextMonth()));
        connect(&mainWindow, SIGNAL(addExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(&mainWindow, SIGNAL(budgetSelected()), this, SLOT(budgetSelected()));
        connect(&mainWindow, SIGNAL(accountsSelected()), this, SLOT(accountsSelected()));
        connect(&mainWindow, SIGNAL(conceptsSelected()), this, SLOT(conceptsSelected()));
        connect(&mainWindow, SIGNAL(expensesSelected()), this, SLOT(expensesSelected()));
        connect(&mainWindow, SIGNAL(scheduledSelected()), this, SLOT(scheduledSelected()));
        connect(&mainWindow, SIGNAL(summarySelected()), this, SLOT(summarySelected()));
}

Controller::~Controller()
{
}

void Controller::runApp()
{
        mainWindow.show();
}

void Controller::previousMonth()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        MonthData *old = month;
        month = new MonthData(old->previous(), activeAccount, this);
        loadMonth();
        mainWindow.setMonthData(month);
        delete old;
}

void Controller::nextMonth()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        if (month->next() <= QDate::currentDate()) {
                MonthData *old = month;
                month = new MonthData(old->next(), activeAccount, this);
                loadMonth();
                mainWindow.setMonthData(month);
                delete old;
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Already at current month"));
        }
}

/* ------------------ account dialog functions ---------------------*/

void Controller::newAccountSelected()
{
        AccountDetailsDialog dialog(&mainWindow);

        connect(&dialog, SIGNAL(newAccountConfirmed(Account*)), this, SLOT(newAccountConfirmed(Account*)));

        dialog.exec();
}

void Controller::newAccountConfirmed(Account *a)
{
        if (!db.exists(a)) {
                if (db.create(a)) {
                        loadAccounts();
                        if (accounts->rowCount() == 1) {
                                /* this is the first account in the system. Activate it */
                                if (activeAccount != NULL) {
                                        delete activeAccount;
                                }
                                activeAccount = new Account(accounts->item(0, 0)->text().toInt(),
                                                            accounts->item(0, 1)->text(),
                                                            accounts->item(0, 2)->text().toDouble(),
                                                            this);
                                settings->setValue(ACTIVE_ACCOUNT_KEY, activeAccount->id());
                                QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                                accounts->item(0, 3)->setIcon(icon);
                                loadScheduled();
                                month = new MonthData(QDate::currentDate(), activeAccount, this);
                                loadMonth();
                                mainWindow.setMonthData(month);
                        }

                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error creating account"));
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Account already exists"));
        }
        delete a;
}

void Controller::accountsSelected()
{
        AccountsDialog dialog(accounts, &mainWindow);

        connect(&dialog, SIGNAL(newAccountSelected()), this, SLOT(newAccountSelected()));
        connect(&dialog, SIGNAL(accountDetailsSelected(QModelIndex)), this, SLOT(accountDetailsSelected(QModelIndex)));

        dialog.exec();
}

void Controller::accountDetailsSelected(QModelIndex index)
{
        AccountDetailsDialog dialog(accounts, index.row(), activeAccount, &mainWindow);

        connect(&dialog, SIGNAL(deleteAccountConfirmed(Account*)),
                this, SLOT(deleteAccountConfirmed(Account*)));
        connect(&dialog, SIGNAL(updateAccountConfirmed(Account*)),
                this, SLOT(updateAccountConfirmed(Account*)));
        connect(&dialog, SIGNAL(selectAccountConfirmed(Account*)),
                this, SLOT(selectAccountConfirmed(Account*)));

        dialog.exec();
}

void Controller::deleteAccountConfirmed(Account *a)
{
        if (a->id() == activeAccount->id()) {
                QMaemo5InformationBox::information(&mainWindow, tr("Cannot delete active account"));
                delete a;
                return;
        }

        QString message(tr("When deleting an account, all the expenses associated to it "
                           "will also be deleted\n"
                           "Are you sure you want to delete the account?"));
        if (ConfirmationDialog::askConfirmation(message)) {
                if (db.remove(a)) {
                        loadAccounts();
                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error deleting account"));
                }
        }
        delete a;
}

void Controller::updateAccountConfirmed(Account *a)
{
        if (db.update(a)) {
                loadAccounts();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating account"));
        }

        delete a;
}

void Controller::selectAccountConfirmed(Account *a)
{
        if (activeAccount != NULL) {
                delete activeAccount;
        }
        activeAccount = a;
        activeAccount->setParent(this);
        settings->setValue(ACTIVE_ACCOUNT_KEY, a->id());
        loadAccounts();
        loadScheduled();

        if (month != NULL) {
                delete month;
        }
        month = new MonthData(QDate::currentDate(), activeAccount, this);
        loadMonth();
        mainWindow.setMonthData(month);
}


/* ------------------ add expense dialog functions ---------------------*/

void Controller::newExpenseSelected()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        ExpenseDetailsDialog dialog(month->date(), concepts, (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(newExpenseConfirmed(Expense*)), this, SLOT(newExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::newExpenseConfirmed(Expense *e)
{
        if (db.create(month->year(), month->month(), activeAccount->id(),  e)) {
                loadMonth(true);
        }
        delete e;
}

void Controller::expensesSelected()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        ExpensesList *list = new ExpensesList (month->expenses(), &mainWindow);

        connect(list, SIGNAL(newExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(list, SIGNAL(expenseDetailsSelected(QModelIndex)), this, SLOT(expenseDetailsSelected(QModelIndex)));

        list->show();

}

void Controller::expenseDetailsSelected(QModelIndex index)
{
        ExpenseDetailsDialog dialog(month->date(), concepts, month->expenses(), index.row(), (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(deleteExpenseConfirmed(Expense*)),
                this, SLOT(deleteExpenseConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateExpenseConfirmed(Expense*)),
                this, SLOT(updateExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::deleteExpenseConfirmed(Expense *e)
{
        if (db.remove(e)) {
                loadMonth(true);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error deleting expense"));
        }

        delete e;
}

void Controller::updateExpenseConfirmed(Expense *e)
{
        if (db.update(e)) {
                loadMonth(true);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating expense"));
        }

        delete e;
}

/* ------------------ budget dialog functions ---------------------*/

void Controller::budgetSelected()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        BudgetDialog dialog(month->budget(), &mainWindow);

        connect(&dialog, SIGNAL(changeBudget(double)), this, SLOT(changeBudgetConfirmed(double)));

        dialog.exec();
}

void Controller::changeBudgetConfirmed (double budget)
{
        if (db.updateBudget(month->year(), month->month(), activeAccount->id(), budget)) {
                month->setBudget(budget);
        }
}

/* ------------------ concept management dialog functions ---------------------*/

void Controller::conceptsSelected()
{
        ConceptsDialog dialog(concepts, &mainWindow);

        connect(&dialog, SIGNAL(newConceptSelected()), this, SLOT(newConceptSelected()));
        connect(&dialog, SIGNAL(conceptDetailsSelected(QModelIndex)), this, SLOT(conceptDetailsSelected(QModelIndex)));

        dialog.exec();
}

void Controller::newConceptSelected()
{
        ConceptDetailsDialog dialog(&mainWindow);

        connect(&dialog, SIGNAL(newConceptConfirmed(Concept*)), this, SLOT(newConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::conceptDetailsSelected(const QModelIndex &index)
{
        ConceptDetailsDialog dialog(concepts, index.row(), &mainWindow);

        connect(&dialog, SIGNAL(deleteConceptConfirmed(Concept*)),
                this, SLOT(deleteConceptConfirmed(Concept*)));
        connect(&dialog, SIGNAL(updateConceptConfirmed(Concept*)),
                this, SLOT(updateConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::newConceptConfirmed(Concept *c)
{
        if (!db.exists(c)) {
                if (db.create(c)) {
                        loadConcepts();
                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error creating concept"));
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Concept already exists"));
        }
        delete c;
}

void Controller::deleteConceptConfirmed(Concept *c)
{
        QString message(tr("When deleting a concept, all the expenses associated to it "
                           "will be assigned an empty concept instead.\n"
                           "Are you sure you want to delete the concept?"));
        if (ConfirmationDialog::askConfirmation(message)) {
                if (db.remove(c)) {
                        loadConcepts();
                        if (activeAccount) {
                                loadScheduled();
                                loadMonth(true);
                        }
                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error deleting concept"));
                }
        }
        delete c;
}

void Controller::updateConceptConfirmed(Concept *c)
{
        if (db.update(c)) {
                loadConcepts();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating concept"));
        }

        delete c;
}

/* ------------------ scheduled expenses functions ---------------------*/

void Controller::scheduledSelected()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        ScheduledList *list = new ScheduledList (scheduled, &mainWindow);

        connect(list, SIGNAL(newScheduledSelected()), this, SLOT(newScheduledSelected()));
        connect(list, SIGNAL(scheduledDetailsSelected(QModelIndex)), this, SLOT(scheduledDetailsSelected(QModelIndex)));

        list->show();
}

void Controller::newScheduledSelected()
{
        ScheduledDetailsDialog dialog(concepts, (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(newScheduledConfirmed(Expense*)), this, SLOT(newScheduledConfirmed(Expense*)));

        dialog.exec();
}

void Controller::scheduledDetailsSelected(QModelIndex index)
{
        ScheduledDetailsDialog dialog(concepts, scheduled, index.row(), (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(deleteScheduledConfirmed(Expense*)),
                this, SLOT(deleteScheduledConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateScheduledConfirmed(Expense*)),
                this, SLOT(updateScheduledConfirmed(Expense*)));

        dialog.exec();
}

void Controller::newScheduledConfirmed(Expense *e)
{
        if (db.createScheduled(activeAccount->id(), e)) {
                loadScheduled();
                if (e->frecuency() & (1 << QDate::currentDate().month() -1)) {
                        /* current month fits into the just created scheduled expense. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description.\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialog::askConfirmation(message)) {
                                Expense *newExpense = new Expense (0, e->day(), e->amount(), e->concept(), e->description());
                                db.create(QDate::currentDate().year(), QDate::currentDate().month(),
                                          activeAccount->id(), newExpense);
                                loadMonth(true);
                                delete newExpense;
                        }
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error creating scheduled expense"));
        }

        delete e;
}

void Controller::deleteScheduledConfirmed(Expense *e)
{
        if (db.removeScheduled(e)) {
                loadScheduled();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error deleting scheduled expense"));
        }

        delete e;
}

void Controller::updateScheduledConfirmed(Expense *e)
{
        /* store the previous frecuency value from the updated scheduled */
        QList<QStandardItem*> list = scheduled->findItems(QString("%1").arg(e->id()),
                                                          Qt::MatchFixedString,
                                                          0);
        int previousFrecuency = scheduled->item(list[0]->index().row(), 5)->text().toInt();

        if (db.updateScheduled(e)) {
                loadScheduled();
                if (e->frecuency() & (1 << QDate::currentDate().month() - 1)
                    && !(previousFrecuency & (1 << QDate::currentDate().month() - 1))) {
                        /* current month has been added to the scheduled expense in the modification. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description after the modification.\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialog::askConfirmation(message)) {
                                Expense *newExpense = new Expense (0, e->day(), e->amount(), e->concept(), e->description());
                                db.create(QDate::currentDate().year(), QDate::currentDate().month(),
                                          activeAccount->id(), newExpense);
                                loadMonth(true);
                                delete newExpense;
                        }
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating scheduled expense"));
        }

        delete e;
}

/*---------------- query  funcions -------------*/

void Controller::summarySelected()
{
        if (!activeAccount) {
                QMaemo5InformationBox::information(&mainWindow, tr("An account must be selected to perform this action"));
                return;
        }

        int minYear, minMonth, maxYear, maxMonth;

        if (db.getDataInterval(activeAccount->id(), minYear, minMonth, maxYear, maxMonth)) {
                summaryWindow = new SummaryWindow(minYear, minMonth, maxYear, maxMonth, &mainWindow);

                connect (summaryWindow, SIGNAL(periodChanged(int, int, int, int)),
                         this, SLOT(summaryPeriodChanged(int, int, int, int)));

                summaryWindow->show();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error loading data interval"));
        }
}

void Controller::summaryPeriodChanged(int startYear, int startMonth, int endYear, int endMonth)
{
        SummaryData *data = new SummaryData(startYear, startMonth, endYear, endMonth);
        if (db.load(activeAccount->id(), data)) {
                summaryWindow->setSummaryData(data);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error loading interval data"));
                delete data;
        }
}

/*---------------- util funcions -------------*/

void Controller::loadMonth(bool reload)
{
        if (reload) {
                month->clear();
                db.load(month, true);
        } else {
                if (db.exists(month)) {
                        db.load (month, false);
                } else {
                        db.create(month);
                        /* create scheduled expenses in new month, but only if it is the current one */
                        if (month->isCurrent()) {
                                int i;
                                for (i = 0; i < scheduled->rowCount(); i++) {
                                        int frec = scheduled->item(i, 5)->text().toInt();
                                        if (frec & (1 << month->month() - 1)) {
                                                /* check if date is valid before creating the expense because of leap years */
                                                if (QDate::isValid(month->year(),
                                                                   month->month(),
                                                                   scheduled->item(i,1)->text().toInt())) {
                                                        Expense *e = new Expense(0,
                                                                                 scheduled->item(i, 1)->text().toInt(),
                                                                                 scheduled->item(i, 2)->text().toDouble(),
                                                                                 scheduled->item(i, 3)->text(),
                                                                                 scheduled->item(i, 4)->text());
                                                        db.create(month->year(), month->month(), activeAccount->id(), e);
                                                        delete e;
                                                }
                                        }
                                }
                                db.load(month, true);
                        }
                }
        }
}

void Controller::loadConcepts()
{
        concepts->clear();
        db.loadConcepts(concepts);
}

void Controller::loadScheduled()
{
        scheduled->removeRows(0, scheduled->rowCount());
        db.loadScheduled(activeAccount->id(), scheduled);
}

void Controller::loadAccounts()
{
        accounts->removeRows(0, accounts->rowCount());
        db.loadAccounts(accounts);

        if (activeAccount) {
                QList<QStandardItem*> list = accounts->findItems(QString("%1").arg(activeAccount->id()),
                                                                 Qt::MatchFixedString,
                                                                 0);
                if (list.size() != 0) {
                        int row = list[0]->index().row();
                        QIcon icon(QString(ACTIVE_ACCOUNT_ICON));
                        accounts->item(row, 3)->setIcon(icon);
                }
        }
}
