#include <QtMaemo5/QMaemo5KineticScroller>

#include "mainwindow.h"
#include "settingsdlg.h"
#include "Settings.h"
#include "SiBApplication.h"
#include "videofile.h"

MainWindow::MainWindow()
{
    QFont buttonFont;
    buttonFont.setPixelSize( 25 );

    int buttonWidth = 150;
    int buttonHeight = 80;
    int buttonLeftOffset = 15;

    m_playbutton = new QPushButton( tr( "Play" ), this );
    m_playbutton->setFont( buttonFont );
    m_playbutton->setGeometry( buttonLeftOffset, 20, buttonWidth, buttonHeight );
    m_playbutton->setEnabled( false );
    m_resumebutton = new QPushButton( tr( "Resume" ), this );
    m_resumebutton->setFont( buttonFont );
    m_resumebutton->setGeometry( buttonLeftOffset, 120, buttonWidth, buttonHeight );
    m_resumebutton->setEnabled( false );
    QPushButton* settingsbutton = new QPushButton( tr( "Folders" ), this );
    settingsbutton->setFont( buttonFont );
    settingsbutton->setGeometry( buttonLeftOffset, 220, buttonWidth, buttonHeight );
    QPushButton* helpButton = new QPushButton( tr( "Info" ), this );
    helpButton->setFont( buttonFont );
    helpButton->setGeometry( buttonLeftOffset, 320, buttonWidth, buttonHeight );
    m_fileList = new QTableWidget( 0, 3, this );
    m_fileList->setGeometry( 190, 5, 605, 415 );

    QStringList labels;
    labels << tr( "File" ) << tr( "Size" ) << tr( "Resume" );
    m_fileList->setHorizontalHeaderLabels(labels);
    m_fileList->setSelectionBehavior( QAbstractItemView::SelectRows );
    m_fileList->setSelectionMode( QAbstractItemView::SingleSelection );
    m_fileList->horizontalHeader()->setResizeMode( 0, QHeaderView::Stretch );
    m_fileList->horizontalHeader()->setResizeMode( 1, QHeaderView::ResizeToContents );
    m_fileList->horizontalHeader()->setResizeMode( 2, QHeaderView::ResizeToContents );
    m_fileList->verticalHeader()->hide();
    m_fileList->setShowGrid(false);
    //m_fileList->setAlternatingRowColors( true );
    m_fileList->setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );

    new QMaemo5KineticScroller( m_fileList );

    connect( m_playbutton, SIGNAL( clicked() ),
             this, SLOT( playButtonPressed() ) );

    connect( m_resumebutton, SIGNAL( clicked() ),
             this, SLOT( resumeButtonPressed() ) );

    connect( settingsbutton, SIGNAL( clicked() ),
             this, SLOT( settingsButtonPressed() ) );

    connect( helpButton, SIGNAL( clicked() ),
             this, SLOT( helpButtonPressed() ) );

    connect( m_fileList, SIGNAL( cellDoubleClicked( int, int ) ),
             this, SLOT( fileSelected( int, int ) ) );

    connect( m_fileList, SIGNAL( cellClicked( int, int ) ),
        this, SLOT( cellClicked( int, int ) ) );
}

MainWindow::~MainWindow()
{
}

// ------------------------------------------------------------------------------
// Called on key events. Derived from QWidget.
// ------------------------------------------------------------------------------
//
void MainWindow::keyPressEvent ( QKeyEvent* event )
{
	SiBApplication::instance()->handleKeyEvent( event );
}

void MainWindow::updateFileList( const QList<VideoFile*>& files )
{
    m_fileList->setRowCount(0);

    for ( int i = 0; i < files.count(); i++ )
    {
        int size = int( 0.5 + files[i]->file().size() / ( 1024 * 1024 ) );
        QTableWidgetItem* fileNameItem = new QTableWidgetItem( files[i]->file().fileName() );
        fileNameItem->setFlags( fileNameItem->flags() ^ Qt::ItemIsEditable );
        QTableWidgetItem* sizeItem = new QTableWidgetItem( tr("%1 MB")
                                             .arg( size ) );
        sizeItem->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
        sizeItem->setFlags( sizeItem->flags() ^ Qt::ItemIsEditable );
        QTableWidgetItem* resumePosItem = new QTableWidgetItem( tr("%1")
                                             .arg( toHHMMSS( files[i]->resumePos() ) ) );
        resumePosItem->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
        resumePosItem->setFlags( resumePosItem->flags() ^ Qt::ItemIsEditable );

        int row = m_fileList->rowCount();
        m_fileList->insertRow( row );
        m_fileList->setRowHeight( row, 50 );
        m_fileList->setItem( row, 0, fileNameItem );
        m_fileList->setItem( row, 1, sizeItem );
        m_fileList->setItem( row, 2, resumePosItem );
        m_fileList->horizontalHeader()->setResizeMode( 1, QHeaderView::ResizeToContents );
    }
}

void MainWindow::playButtonPressed()
{
	if ( !SiBApplication::instance()->isPlayerRunning() )
	{
		if ( m_fileList->currentRow() >= 0 ) {
			SiBApplication::instance()->restartFile( m_fileList->currentRow() );
		}
	}
	else
	{
		SiBApplication::instance()->stopPlayer();
	}
}

void MainWindow::resumeButtonPressed()
{
    fileSelected( m_fileList->currentRow(), m_fileList->currentColumn() );
}

void MainWindow::settingsButtonPressed()
{
    SettingsDlg settingsDlg;
    settingsDlg.exec();
    SiBApplication::instance()->updateFileList();
}

void MainWindow::helpButtonPressed()
{
	QMessageBox::information( this, tr( "Default MPlayer Keyboard Shortcuts"), tr( "\
 left/right\tseek backward/forward 10 seconds \n\
 down/up\tseek backward/forward  1 minute \n\
 p or SPACE\tpause movie (press any key to continue) \n\
 q\t\tstop playing\n\
 f\t\ttoggle fullscreen\n\
 + or -\t\tadjust audio delay by +/- 0.1 second\n\
 o\t\tcycle OSD mode:  none / seekbar / seekbar + timer\n\
 x or z\t\tadjust subtitle delay by +/- 0.1 second\n\
 r or t\t\tadjust subtitle position up/down\n\
" ) );
}

void MainWindow::fileSelected( int row, int /*column*/ )
{
    if ( row >= 0 ) {
        SiBApplication::instance()->playFile( row );
    }
}

void MainWindow::cellClicked( int row, int /*column*/ )
{
    if ( row >= 0 ) {
        if ( SiBApplication::instance()->hasFileResumePos( row ) ) {
            m_resumebutton->setEnabled( true );
        } else {
            m_resumebutton->setEnabled( false );
        }
        m_playbutton->setEnabled( true );
    } else {
        m_playbutton->setEnabled( false );
        m_resumebutton->setEnabled( false );
    }
}

void MainWindow::playerStart()
{
	m_playbutton->setText( tr( "Stop" ) );
	m_resumebutton->setEnabled( false );
}

void MainWindow::playerStop()
{
	m_playbutton->setText( tr( "Start" ) );
	cellClicked( m_fileList->currentRow(), m_fileList->currentColumn() );
}

QString MainWindow::toHHMMSS( unsigned int tenthOfASec )
{
    unsigned int secs = tenthOfASec / 10;
    unsigned int hoursFrac = secs / 3600;
    secs -= hoursFrac * 3600;
    unsigned int minsFrac = secs / 60;
    secs -= minsFrac * 60;
    unsigned int secsFrac = secs;
    return QString( "%1:%2:%3" ).arg( hoursFrac, 2, 10, QChar( '0' ) ).arg( minsFrac, 2, 10, QChar( '0' ) ).arg( secsFrac, 2, 10, QChar( '0' ) );
}
