#include "playerfacade.h"
#include "videofile.h"
#include "SiBApplication.h"
#include <QFileInfo>
#include <QProcess>
#include <QMessageBox>
#include <iostream>
#include <QTimer>

using namespace std;

PlayerFacade::PlayerFacade( QObject* parent ):
    QObject( parent ),
    m_playerIsRunning( false )
{
}

PlayerFacade::~PlayerFacade()
{
}

void PlayerFacade::playerStarted()
{
    m_playerIsRunning = true;
    emit playerStart();
}

void PlayerFacade::playerExited( int /*exitCode*/, QProcess::ExitStatus /*exitStatus*/ )
{
    m_playerIsRunning = false;
    SiBApplication::instance()->refreshFileList();
    emit playerStop();
}

void PlayerFacade::playerHasOutputData()
{
    QString output = m_mplayer->readAllStandardOutput();
    cout << output.toAscii().data();
    cout.flush();

    QRegExp regex( "^A:\\s*(\\d*\\.\\d)\\s*V:\\s*(\\d+\\.\\d)", Qt::CaseSensitive, QRegExp::RegExp2 );
    if ( regex.indexIn( output ) != -1 )
    {
        // cap 1 is audio pos, cap 2 video pos
        QString videoPosString = regex.cap( 2 );
        videoPosString.remove( '.' );
        bool ok = false;
        unsigned int videoPos = videoPosString.toInt( &ok );
        if ( ok ) {
            if ( m_currentFile ) {
                m_currentFile->setResumePos( videoPos );
            }
        }
    } else if ( output.contains( "Exiting... (End of file)" ) ) {
        if ( m_currentFile ) {
            m_currentFile->setResumePos( 0 );
        }
    }
}

int PlayerFacade::playFile( VideoFile* file )
{
    if ( !file ) {
        return -1;
    }

    if ( m_playerIsRunning ) {
    	return -2;
    }

    m_currentFile = file;
    m_mplayer = new QProcess( this );
    connect( m_mplayer, SIGNAL( started() ), this, SLOT( playerStarted() ) );
    connect( m_mplayer, SIGNAL( finished( int, QProcess::ExitStatus ) ), this, SLOT( playerExited( int, QProcess::ExitStatus ) ) );
    connect( m_mplayer, SIGNAL( readyReadStandardOutput() ), this, SLOT( playerHasOutputData() ) );
    connect( m_mplayer, SIGNAL( readyReadStandardError() ), this, SLOT( playerHasOutputData() ) );

	unsigned int secOffset( 20 ); // will be subtracted from the resume pos (tenth of a second).
	unsigned int resumepos = m_currentFile->resumePos();
	if ( resumepos >= secOffset ) {
		resumepos -= secOffset;
	}
	resumepos /= 10; // convert to sec

    QStringList arguments;
    arguments << "-fs";
    arguments << "-ss";
    arguments << QString( "%1" ).arg( resumepos );
    arguments << m_currentFile->file().absoluteFilePath();

    m_mplayer->start( "mplayer", arguments );
    if ( !m_mplayer->waitForStarted() ) {
        QMessageBox::warning( NULL, tr( "Player not found" ), tr( "Cannot open mplayer. Please make sure mplayer is installed properly" ) );
        return -1;
    }

    return 0;
}

bool PlayerFacade::isPlayerRunning()
{
	return m_playerIsRunning;
}

void PlayerFacade::stop()
{
	m_mplayer->terminate();
	QTimer::singleShot( 2000, m_mplayer, SLOT( kill() ) );
}

// ------------------------------------------------------------------------------
// Forwards a key press to mplayer.
// ------------------------------------------------------------------------------
//
void PlayerFacade::forwardKeyEvent( QKeyEvent* event )
{
	/**
	 * Map arrow keys.
	 */
	const char KEY_UP[4]    = { 0x1B, 0x5B, 0x41, 0x00 };
	const char KEY_DOWN[4]  = { 0x1B, 0x5B, 0x42, 0x00 };
	const char KEY_RIGHT[4] = { 0x1B, 0x5B, 0x43, 0x00 };
	const char KEY_LEFT[4]  = { 0x1B, 0x5B, 0x44, 0x00 };

	switch ( event->key() ) {
		case Qt::Key_Up:
			m_mplayer->write( KEY_UP, 3 );
			break;
		case Qt::Key_Down:
			m_mplayer->write( KEY_DOWN, 3 );
			break;
		case Qt::Key_Right:
			m_mplayer->write( KEY_RIGHT, 3 );
			break;
		case Qt::Key_Left:
			m_mplayer->write( KEY_LEFT, 3 );
			break;
		default:
			m_mplayer->write( event->text().toAscii().constData() );
			break;
	}
}
